package com.finconsgroup.itserr.marketplace.metrics.dm.service.impl;

import com.finconsgroup.itserr.marketplace.metrics.dm.bs.MVUpdater;
import com.finconsgroup.itserr.marketplace.metrics.dm.bs.QueuedMVUpdater;
import com.finconsgroup.itserr.marketplace.metrics.dm.dto.OutputResourcesDailyMetricsDto;
import com.finconsgroup.itserr.marketplace.metrics.dm.dto.OutputResourcesSummaryMetricsDto;
import com.finconsgroup.itserr.marketplace.metrics.dm.entity.MetricEventDailyCount;
import com.finconsgroup.itserr.marketplace.metrics.dm.entity.projection.MetricSummaryProjection;
import com.finconsgroup.itserr.marketplace.metrics.dm.mapper.MetricMapper;
import com.finconsgroup.itserr.marketplace.metrics.dm.repository.MetricEventDailyCountRepository;
import com.finconsgroup.itserr.marketplace.metrics.dm.service.MetricInterestService;
import com.finconsgroup.itserr.marketplace.metrics.dm.service.MetricService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Service;

import java.time.LocalDate;
import java.util.List;
import java.util.Set;

/**
 * Default implementation of {@link MetricService}.
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class DefaultMetricService implements MetricService {

    /** Repository for accessing daily metric event counts */
    private final MetricEventDailyCountRepository metricEventDailyCountRepository;

    /** Service for calculating and managing metrics-based interest scores */
    private final MetricInterestService metricInterestService;

    /** Mapper for converting between metric DTOs and entities */
    private final MetricMapper metricMapper;

    /** Handler for queued materialized view updates */
    private final QueuedMVUpdater queuedMVUpdater;
    /** Component that performs the actual materialized view updates */
    private final MVUpdater mvUpdater;

    @Override
    public void update() {
        update(false);
    }

    @Override
    public void update(final boolean sync) {
        if (sync) {
            mvUpdater.update();
        } else {
            queuedMVUpdater.update();
        }
    }

    @NonNull
    @Override
    public OutputResourcesDailyMetricsDto findResourcesDailyMetrics(
            @NonNull final Set<String> resourcesIds,
            @NonNull final LocalDate fromDate,
            @NonNull final LocalDate toDate) {

        final List<MetricEventDailyCount> metrics = metricEventDailyCountRepository.findByResourceIdInAndEventDayBetween(
                resourcesIds,
                fromDate,
                toDate);

        final OutputResourcesDailyMetricsDto dailyMetrics = metricMapper.toOutputResourcesDailyMetricsDto(metrics);

        dailyMetrics.values()
                .stream().flatMap(m -> m.values().stream())
                .forEach(m -> m.setInterest(metricInterestService.getTotalInterest(m)));

        return dailyMetrics;

    }

    @NonNull
    @Override
    public OutputResourcesSummaryMetricsDto findResourcesSummaryMetrics(
            @NonNull Set<String> resourcesIds,
            LocalDate toDate) {

        final List<MetricSummaryProjection> metrics = toDate == null
                ? metricEventDailyCountRepository.summarizeByResourceIds(resourcesIds)
                : metricEventDailyCountRepository.summarizeByResourceIdsAndEndDate(resourcesIds, toDate);

        return metricMapper.toOutputResourcesSummaryMetricsDto(metrics);

    }

}
