package com.finconsgroup.itserr.marketplace.metrics.dm.dto;

import com.fasterxml.jackson.annotation.JsonProperty;
import io.swagger.v3.oas.annotations.media.Schema;
import lombok.Getter;

/**
 * Enum representing different types of metrics that can be tracked.
 */
@Getter
@Schema(name = "OutputMetricEventType", description = "Type of metrics")
public enum MetricDtoType {

    /** Tracks when a user downloads a catalog item */
    @JsonProperty("catalog-item-download")
    CATALOG_ITEM_DOWNLOAD,

    /** Tracks when a user views a catalog item */
    @JsonProperty("catalog-item-view")
    CATALOG_ITEM_VIEW,

    /** Tracks when a user marks a catalog item as favorite */
    @JsonProperty("catalog-item-favourite")
    CATALOG_ITEM_FAVOURITE,

    /** Tracks when a user comments on a catalog item */
    @JsonProperty("catalog-item-comment")
    CATALOG_ITEM_COMMENT;

    /** The string representation of the metric type used in view layer */
    private final String viewValue;

    /**
     * Default constructor that initializes viewValue from JsonProperty annotation.
     *
     * @throws IllegalStateException if enum field cannot be instantiated
     */
    MetricDtoType() {
        try {
            this.viewValue = this.getClass()
                    .getField(this.name())
                    .getAnnotation(JsonProperty.class)
                    .value();
        } catch (final Exception e) {
            // Cannot happen
            throw new IllegalStateException("Cannot instantiate enum", e);
        }
    }

    /**
     * Converts a view value string to corresponding MetricDtoType enum.
     *
     * @param value the string value to convert
     * @return the matching MetricDtoType, or null if value is null
     * @throws IllegalArgumentException if no matching enum constant is found
     */
    public static MetricDtoType fromViewValue(final String value) {
        if (value == null) {
            return null;
        }
        for (MetricDtoType t : values()) {
            if (t.viewValue.equals(value))
                return t;
        }
        throw new IllegalArgumentException("Invalid metric view value: " + value);
    }

}
