package com.finconsgroup.itserr.marketplace.institutionalpage.dm.component;

import com.finconsgroup.itserr.marketplace.institutionalpage.dm.entity.MemberRequestEntity;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.enums.MemberRequestType;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.exception.MemberRequestsAlreadyExistException;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.exception.MemberRequestsNotFoundException;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.repository.MemberRequestRepository;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Component;

import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

/**
 * Helper component for managing {@link MemberRequestEntity} operations, including retrieval, validation,
 * and existence checks for invitations and join requests within an institutional page context.
 */
@Component
@RequiredArgsConstructor
public class MemberRequestHelper {

    private final MemberRequestRepository memberRequestRepository;

    public void deleteMemberRequestsForInstitutionalPage(UUID institutionalPageId) {
        memberRequestRepository.deleteAllByInstitutionalPageId(institutionalPageId);
    }

    public List<MemberRequestEntity> retrieveMemberRequests(UUID institutionalPageId, List<UUID> userIds) {
        return memberRequestRepository.findAllByInstitutionalPageIdAndUserIdIn(institutionalPageId, userIds);
    }

    public List<MemberRequestEntity> retrieveInvitations(UUID institutionalPageId, List<UUID> invitedUserIds) {
        return memberRequestRepository.findAllByInstitutionalPageIdAndUserIdInAndRequestType(
                institutionalPageId,
                invitedUserIds,
                MemberRequestType.INVITATION
        );
    }

    public MemberRequestEntity retrieveInvitationOrThrow(UUID institutionalPageId, UUID invitedUserId) {
        return retrieveInvitation(institutionalPageId, invitedUserId)
                .orElseThrow(() -> new MemberRequestsNotFoundException(institutionalPageId, Set.of(invitedUserId)));
    }

    public Optional<MemberRequestEntity> retrieveInvitation(UUID institutionalPageId, UUID invitedUserId) {
        return memberRequestRepository.findByInstitutionalPageIdAndUserIdAndRequestType(
                institutionalPageId,
                invitedUserId,
                MemberRequestType.INVITATION
        );
    }

    public MemberRequestEntity retrieveJoinRequestOrThrow(UUID institutionalPageId, UUID userId) {
        return retrieveJoinRequest(institutionalPageId, userId)
                .orElseThrow(() -> new MemberRequestsNotFoundException(institutionalPageId, Set.of(userId)));
    }

    public Optional<MemberRequestEntity> retrieveJoinRequest(UUID institutionalPageId, UUID userId) {
        return memberRequestRepository.findByInstitutionalPageIdAndUserIdAndRequestType(
                institutionalPageId,
                userId,
                MemberRequestType.JOIN_REQUEST
        );
    }

    public List<MemberRequestEntity> retrieveAllJoinRequestsByUserIdsOrThrow(UUID institutionalPageId, List<UUID> userIds) {
        List<MemberRequestEntity> joinRequestEntities = retrieveMemberRequests(
                institutionalPageId,
                userIds,
                MemberRequestType.JOIN_REQUEST
        );
        verifyAllMemberRequestsFoundOrThrow(institutionalPageId, joinRequestEntities, userIds);
        return joinRequestEntities;
    }

    public void verifyInvitationsNotAlreadyExistOrThrow(UUID institutionalPageId, List<UUID> userIds) {
        verifyMemberRequestsNotAlreadyExistOrThrow(institutionalPageId, userIds, MemberRequestType.INVITATION);
    }

    public void verifyJoinRequestsNotAlreadyExistOrThrow(UUID institutionalPageId, List<UUID> userIds) {
        verifyMemberRequestsNotAlreadyExistOrThrow(institutionalPageId, userIds, MemberRequestType.JOIN_REQUEST);
    }

    // private

    private void verifyMemberRequestsNotAlreadyExistOrThrow(
            UUID institutionalPageId,
            List<UUID> userIds,
            MemberRequestType memberRequestType
    ) {
        List<UUID> userIdsWIthExistingMemberRequests = retrieveMemberRequests(institutionalPageId, userIds, memberRequestType)
                .stream()
                .map(memberRequestEntity -> memberRequestEntity.getUserId())
                .toList();
        if (!userIdsWIthExistingMemberRequests.isEmpty()) {
            throw new MemberRequestsAlreadyExistException(institutionalPageId, userIdsWIthExistingMemberRequests);
        }
    }

    private List<MemberRequestEntity> retrieveMemberRequests(
            UUID institutionalPageId,
            List<UUID> userIds,
            MemberRequestType memberRequestType
    ) {
        return memberRequestRepository.findAllByInstitutionalPageIdAndUserIdInAndRequestType(
                institutionalPageId,
                userIds,
                memberRequestType
        );
    }

    private static void verifyAllMemberRequestsFoundOrThrow(
            UUID institutionalPageId,
            List<MemberRequestEntity> foundRequestEntities,
            List<UUID> requestedUserIds
    ) {
        Set<UUID> foundUserIds = foundRequestEntities.stream()
                .map(MemberRequestEntity::getUserId)
                .collect(Collectors.toSet());
        Set<UUID> notFoundUserIds = new HashSet<>(requestedUserIds);
        notFoundUserIds.removeAll(foundUserIds);
        if (!notFoundUserIds.isEmpty()) {
            throw new MemberRequestsNotFoundException(institutionalPageId, notFoundUserIds);
        }
    }

}
