package com.finconsgroup.itserr.marketplace.institutionalpage.dm.component;

import com.finconsgroup.itserr.marketplace.institutionalpage.dm.entity.InstitutionalPageEntity;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.enums.ModerationStatus;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.exception.InvalidModerationStatusException;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.exception.UnexpectedModerationStatusException;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import java.time.Instant;
import java.util.UUID;

/**
 * Component responsible for handling moderation transitions of {@link InstitutionalPageEntity} instances.
 *
 * <p>
 * It manages the approval, rejection, creation, and update lifecycle based on the current moderation status.
 * Manages moderation-related fields and handles versioning logic by cloning and replacing entities when necessary.
 * Ensures that moderation rules are strictly enforced and throws appropriate exceptions when constraints are violated.
 * </p>
 */
@Component
@RequiredArgsConstructor
@Slf4j
public class InstitutionalPageModerationHandler {

    private final InstitutionalPageHelper institutionalPageHelper;
    private final EntityBuilder entityBuilder;

    public InstitutionalPageEntity create(InstitutionalPageEntity institutionalPage) {
        if (institutionalPage.getId() == null && institutionalPage.getModerationStatus() == null) {
            institutionalPage.setModerationStatus(ModerationStatus.DRAFT);
            return institutionalPage;
        } else {
            throw error(institutionalPage.getId(), institutionalPage);
        }
    }

    public InstitutionalPageEntity requestModeration(InstitutionalPageEntity institutionalPage) {
        if (institutionalPage.getModerationStatus().equals(ModerationStatus.DRAFT)) {
            institutionalPage.setModerationStatus(ModerationStatus.PENDING);
            return institutionalPage;
        } else {
            throw error(institutionalPage.getId(), institutionalPage);
        }
    }

    public InstitutionalPageEntity approve(UUID institutionalPageId, InstitutionalPageEntity institutionalPage) {
        if (institutionalPage.getModerationStatus().equals(ModerationStatus.PENDING)) {
            InstitutionalPageEntity approvedInstitutionalPage;
            if (institutionalPage.getOriginalInstitutionalPageId() != null) {
                approvedInstitutionalPage = institutionalPageHelper.retrieveOriginalInstitutionalPage(institutionalPage);
                entityBuilder.updateEntity(approvedInstitutionalPage, institutionalPage);
                approvedInstitutionalPage.setUpdatedInstitutionalPageId(null);
                institutionalPageHelper.delete(institutionalPage);
            } else {
                approvedInstitutionalPage = institutionalPage;
            }
            if (approvedInstitutionalPage.getPublicationTime() == null && approvedInstitutionalPage.isPublished()) {
                approvedInstitutionalPage.setPublicationTime(Instant.now());
            }
            approvedInstitutionalPage.setModerationStatus(ModerationStatus.APPROVED);
            approvedInstitutionalPage.setRejectionMessage("");
            return approvedInstitutionalPage;
        } else {
            throw error(institutionalPageId, institutionalPage);
        }
    }

    public InstitutionalPageEntity reject(UUID institutionalPageId, InstitutionalPageEntity institutionalPage, String message) {
        if (institutionalPage.getModerationStatus().equals(ModerationStatus.PENDING)) {
            institutionalPage.setModerationStatus(ModerationStatus.REJECTED);
            institutionalPage.setRejectionMessage(message);
            return institutionalPage;
        } else {
            throw error(institutionalPageId, institutionalPage);
        }
    }

    public InstitutionalPageEntity update(UUID userId, UUID institutionalPageId, InstitutionalPageEntity institutionalPage) {
        if (institutionalPage.getModerationStatus().equals(ModerationStatus.PENDING)) {
            throw error(institutionalPageId, institutionalPage);
        } else if (institutionalPage.getModerationStatus().equals(ModerationStatus.DRAFT)) {
            return institutionalPage;
        } else if (institutionalPage.getModerationStatus().equals(ModerationStatus.REJECTED)) {
            institutionalPage.setModerationStatus(ModerationStatus.DRAFT);
            return institutionalPage;
        } else if (institutionalPage.getModerationStatus().equals(ModerationStatus.APPROVED)) {
            InstitutionalPageEntity clonedInstitutionalPage = entityBuilder.cloneInstitutionalPageEntity(institutionalPage);
            clonedInstitutionalPage.setOriginalInstitutionalPageId(institutionalPage.getId());
            clonedInstitutionalPage.setModerationStatus(ModerationStatus.DRAFT);
            InstitutionalPageEntity savedClonedInstitutionalPage = institutionalPageHelper.save(userId, clonedInstitutionalPage);
            institutionalPage.setUpdatedInstitutionalPageId(savedClonedInstitutionalPage.getId());
            institutionalPageHelper.save(userId, institutionalPage);
            return savedClonedInstitutionalPage;
        } else {
            throw new UnexpectedModerationStatusException(institutionalPage.getId(), institutionalPage.getModerationStatus());
        }
    }

    /**
     * Prepares the given institutional page for immediate deletion.
     * <p>
     * If the page is a draft (i.e., has an originalInstitutionalPageId),
     * deletes the draft and returns the original entity to be deleted.
     * Otherwise, returns the given entity as-is.
     * </p>
     *
     * @param institutionalPage the institutional page to delete
     * @return the entity to be deleted by the caller
     */
    public InstitutionalPageEntity prepareForImmediateDeletion(InstitutionalPageEntity institutionalPage) {
        InstitutionalPageEntity originalInstitutionalPage;
        if (institutionalPage.getOriginalInstitutionalPageId() != null) {
            originalInstitutionalPage = institutionalPageHelper.retrieveOriginalInstitutionalPage(institutionalPage);
            institutionalPageHelper.delete(institutionalPage);
        } else {
            originalInstitutionalPage = institutionalPage;
        }
        return originalInstitutionalPage;
    }

    private InvalidModerationStatusException error(UUID institutionalPageId, InstitutionalPageEntity institutionalPageEntity) {
        return new InvalidModerationStatusException(institutionalPageId, institutionalPageEntity.getModerationStatus());
    }

}
