package com.finconsgroup.itserr.marketplace.institutionalpage.dm.component;

import com.finconsgroup.itserr.marketplace.institutionalpage.dm.entity.InstitutionalPageEntity;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.entity.MemberEntity;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.entity.ParagraphEntity;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.repository.ParagraphRepository;
import lombok.Builder;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Page;
import org.springframework.stereotype.Component;

import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;

/**
 * Component responsible for loading and assembling related associations of {@link InstitutionalPageEntity}
 * instances in bulk or individually.
 * It Performs eager load for all related associations that are needed for mapping a page on institutional pages.
 */
@Component
@RequiredArgsConstructor
@Slf4j
public class AssociationLoader {

    public static final String ALL = "all";
    public static final String ASSOCIATION_ROOT_INSTITUTIONAL_PAGE = "rootInstitutionalPage";
    public static final String ASSOCIATION_PARENT_INSTITUTIONAL_PAGE = "parentInstitutionalPage";
    public static final String ASSOCIATION_WP_LEADS = "wpLeads";
    public static final String ASSOCIATION_MEMBERS = "members";
    public static final String ASSOCIATION_CHILD_INSTITUTIONAL_PAGES = "childInstitutionalPages";
    public static final String ASSOCIATION_PARAGRAPHS = "paragraphs";

    private static final Set<String> DEFAULT_ASSOCIATIONS = Set.of(
            ASSOCIATION_ROOT_INSTITUTIONAL_PAGE, ASSOCIATION_PARENT_INSTITUTIONAL_PAGE, ASSOCIATION_WP_LEADS,
            ASSOCIATION_MEMBERS, ASSOCIATION_CHILD_INSTITUTIONAL_PAGES, ASSOCIATION_PARAGRAPHS
    );

    private final InstitutionalPageHelper institutionalPageHelper;
    private final MemberHelper memberHelper;
    private final ParagraphRepository paragraphRepository;

    public LoadedAssociations loadAssociations(
            final Page<InstitutionalPageEntity> institutionalPageEntityPage,
            final Set<String> associationsToLoadParam
    ) {

        // Set up attribute to load or default
        Set<String> associationsToLoad = associationsToLoadParam.contains(ALL)
                ? DEFAULT_ASSOCIATIONS
                : associationsToLoadParam;
        LoadedAssociations loadedAssociations = new LoadedAssociations();

        // return immediately if no IPs provided
        List<InstitutionalPageEntity> institutionalPages = new LinkedList<>(institutionalPageEntityPage.getContent());
        if (institutionalPages.isEmpty()) {
            return loadedAssociations;
        }

        // initialize loaded associations internal objects
        Map<UUID, UUID> originalIdToInstitutionalPageIdMap = new HashMap<>();
        for (InstitutionalPageEntity institutionalPage : institutionalPages) {
            UUID institutionalPageId = institutionalPage.getId();
            UUID originalId = InstitutionalPageHelper.getOriginalInstitutionalPageId(institutionalPage);
            originalIdToInstitutionalPageIdMap.put(originalId, institutionalPageId);
            loadedAssociations.childInstitutionalPagesMap.put(institutionalPageId, new LinkedList<>());
            loadedAssociations.paragraphMap.put(institutionalPageId, new LinkedList<>());
        }

        // rootInstitutionalPage
        if (associationsToLoad.contains(ASSOCIATION_ROOT_INSTITUTIONAL_PAGE)) {
            institutionalPageHelper.retrieveOriginalRootsInstitutionalPages(institutionalPages)
                    .forEach(root ->
                            loadedAssociations.rootInstitutionalPageMap.put(root.getId(), root)
                    );
        }

        // parentInstitutionalPage
        if (associationsToLoad.contains(ASSOCIATION_PARENT_INSTITUTIONAL_PAGE)) {
            institutionalPageHelper.retrieveApprovedParentsInstitutionalPages(institutionalPages)
                    .forEach(parent ->
                            loadedAssociations.parentInstitutionalPageMap.put(parent.getId(), parent)
                    );
        }

        // childInstitutionalPages
        if (associationsToLoad.contains(ASSOCIATION_CHILD_INSTITUTIONAL_PAGES)) {
            // retrieve child institutional pages
            List<InstitutionalPageEntity> childInstitutionalPages =
                    institutionalPageHelper.retrieveApprovedChildInstitutionalPages(institutionalPages);
            for (InstitutionalPageEntity childInstitutionalPage : childInstitutionalPages) {
                // child are only associated to original ip versions, so
                // we need to retrieve the institutional page id from 'originalIdToInstitutionalPageIdMap'
                UUID institutionalPageId = originalIdToInstitutionalPageIdMap.get(childInstitutionalPage.getParentInstitutionalPageId());
                loadedAssociations.childInstitutionalPagesMap.get(institutionalPageId).add(childInstitutionalPage);
            }
        }

        // paragraphs
        if (associationsToLoad.contains(ASSOCIATION_PARAGRAPHS)) {
            List<UUID> institutionalPageIds = institutionalPages.stream()
                    .map(InstitutionalPageEntity::getId)
                    .toList();
            // retrieve paragraphs
            List<ParagraphEntity> paragraphs = paragraphRepository.findAllByInstitutionalPageIdIn(institutionalPageIds);
            for (ParagraphEntity paragraph : paragraphs) {
                // paragraphs can be associated to both original and pending versions,
                // so no need to retrieve original version in this case
                UUID institutionalPageId = paragraph.getInstitutionalPageId();
                loadedAssociations.paragraphMap.get(institutionalPageId).add(paragraph);
            }
        }

        // create map containing initial institutional page and related institutional page (parent and children)
        Map<UUID, InstitutionalPageEntity> loadedInstitutionalPageMap = new HashMap<>();
        institutionalPages.forEach(institutionalPageEntity ->
                loadedInstitutionalPageMap.put(institutionalPageEntity.getId(), institutionalPageEntity)
        );
        loadedAssociations.childInstitutionalPagesMap.values()
                .stream()
                .flatMap(Collection::stream)
                .filter(institutionalPageEntity -> !loadedInstitutionalPageMap.containsKey(institutionalPageEntity.getId()))
                .forEach(institutionalPageEntity ->
                        loadedInstitutionalPageMap.put(institutionalPageEntity.getId(), institutionalPageEntity)
                );
        loadedAssociations.parentInstitutionalPageMap.values()
                .stream()
                .filter(institutionalPageEntity -> !loadedInstitutionalPageMap.containsKey(institutionalPageEntity.getId()))
                .forEach(institutionalPageEntity ->
                        loadedInstitutionalPageMap.put(institutionalPageEntity.getId(), institutionalPageEntity)
                );

        // initialize loaded associations internal objects
        Map<UUID, List<UUID>> rootIdToOriginalIdsMap = new HashMap<>();
        for (InstitutionalPageEntity institutionalPage : loadedInstitutionalPageMap.values()) {
            UUID originalId = InstitutionalPageHelper.getOriginalInstitutionalPageId(institutionalPage);
            UUID rootId = InstitutionalPageHelper.getRootInstitutionalPageId(institutionalPage);
            rootIdToOriginalIdsMap.computeIfAbsent(rootId, unused -> new LinkedList<>()).add(originalId);
            loadedAssociations.membersMap.put(originalId, new LinkedList<>());
            loadedAssociations.wpLeadsMap.put(originalId, new LinkedList<>());
        }

        // wpLeader
        if (associationsToLoad.contains(ASSOCIATION_WP_LEADS)) {
            // retrieve wp leaders
            List<MemberEntity> wpLeaders = memberHelper.retrieveWPLeadersByInstitutionalPages(new LinkedList<>(loadedInstitutionalPageMap.values()));
            for (MemberEntity wpLeader : wpLeaders) {
                // list of original ids for that root id
                List<UUID> originalIds = rootIdToOriginalIdsMap.get(wpLeader.getInstitutionalPageId());
                for (UUID originalId : originalIds) {
                    // wp leaders are only associated to original ip versions, the map
                    // 'rootIdToOriginalIdsMap' already take care of this
                    loadedAssociations.wpLeadsMap.get(originalId).add(wpLeader);
                }
            }
        }

        // members
        if (associationsToLoad.contains(ASSOCIATION_MEMBERS)) {
            // retrieve members
            List<MemberEntity> members = memberHelper.retrieveRegularMembersByInstitutionalPages(new LinkedList<>(loadedInstitutionalPageMap.values()));
            for (MemberEntity member : members) {
                // members are only associated to original ip versions
                UUID originalId = member.getInstitutionalPageId();
                loadedAssociations.membersMap.get(originalId).add(member);
            }
        }

        return loadedAssociations;
    }

    public Associations buildAssociations(
            InstitutionalPageEntity institutionalPageEntity,
            Set<String> associationsToLoadParam,
            AssociationLoader.LoadedAssociations loadedAssociations
    ) {
        Set<String> associationsToLoad = associationsToLoadParam.contains(ALL)
                ? DEFAULT_ASSOCIATIONS
                : associationsToLoadParam;

        UUID institutionalPageId = institutionalPageEntity.getId();
        UUID originalId = InstitutionalPageHelper.getOriginalInstitutionalPageId(institutionalPageEntity);

        Associations.AssociationsBuilder associationsBuilder = AssociationLoader.Associations.builder();
        if (associationsToLoad.contains(ASSOCIATION_ROOT_INSTITUTIONAL_PAGE)) {
            UUID rootId = InstitutionalPageHelper.getRootInstitutionalPageId(institutionalPageEntity);
            String rootInstitutionalPageName = loadedAssociations.rootInstitutionalPageMap.get(rootId).getName();
            associationsBuilder.rootInstitutionalPageName(rootInstitutionalPageName);
        }
        if (associationsToLoad.contains(ASSOCIATION_PARENT_INSTITUTIONAL_PAGE)) {
            InstitutionalPageWithMembers parentInstitutionalPageWithMembers =
                    mapToParent(institutionalPageEntity, loadedAssociations);
            associationsBuilder.parentInstitutionalPage(parentInstitutionalPageWithMembers);
        }
        if (associationsToLoad.contains(ASSOCIATION_CHILD_INSTITUTIONAL_PAGES)) {
            List<InstitutionalPageWithMembers> childInstitutionalPages =
                    mapToChildren(institutionalPageId, loadedAssociations);
            associationsBuilder.childInstitutionalPages(childInstitutionalPages);
        }
        if (associationsToLoad.contains(ASSOCIATION_PARAGRAPHS)) {
            associationsBuilder.paragraphs(
                    loadedAssociations.paragraphMap.getOrDefault(institutionalPageId, new LinkedList<>())
            );
        }
        if (associationsToLoad.contains(ASSOCIATION_WP_LEADS)) {
            associationsBuilder.wpLeads(loadedAssociations.wpLeadsMap.getOrDefault(originalId, new LinkedList<>()));
        }
        if (associationsToLoad.contains(ASSOCIATION_MEMBERS)) {
            associationsBuilder.members(loadedAssociations.membersMap.getOrDefault(originalId, new LinkedList<>()));
        }
        return associationsBuilder.build();
    }

    public InstitutionalPageWithMembers mapToParent(
            InstitutionalPageEntity institutionalPageEntity,
            AssociationLoader.LoadedAssociations loadedAssociations
    ) {
        UUID parentInstitutionalPageId = institutionalPageEntity.getParentInstitutionalPageId();
        if (parentInstitutionalPageId == null) {
            return null;
        }
        InstitutionalPageEntity parentInstitutionalPage = loadedAssociations.parentInstitutionalPageMap
                .get(parentInstitutionalPageId);
        List<MemberEntity> wpLeaderEntities = new LinkedList<>(
                loadedAssociations.wpLeadsMap.getOrDefault(parentInstitutionalPageId, new LinkedList<>())
        );
        List<MemberEntity> memberEntities = new LinkedList<>(
                loadedAssociations.membersMap.getOrDefault(parentInstitutionalPageId, new LinkedList<>())
        );
        return InstitutionalPageWithMembers
                .builder()
                .institutionalPage(parentInstitutionalPage)
                .wpLeads(wpLeaderEntities)
                .members(memberEntities)
                .build();
    }

    private static List<InstitutionalPageWithMembers> mapToChildren(
            UUID institutionalPageId, LoadedAssociations loadedAssociations
    ) {
        List<InstitutionalPageEntity> childInstitutionalPages = loadedAssociations.childInstitutionalPagesMap.get(institutionalPageId);
        return childInstitutionalPages.stream()
                .map(childInstitutionalPageEntity -> {
                    UUID childOriginalId = InstitutionalPageHelper.getOriginalInstitutionalPageId(childInstitutionalPageEntity);
                    List<MemberEntity> wpLeaderEntities = new LinkedList<>(
                            loadedAssociations.wpLeadsMap.getOrDefault(childOriginalId, new LinkedList<>())
                    );
                    List<MemberEntity> memberEntities = new LinkedList<>(
                            loadedAssociations.membersMap.getOrDefault(childOriginalId, new LinkedList<>())
                    );
                    return InstitutionalPageWithMembers
                            .builder()
                            .institutionalPage(childInstitutionalPageEntity)
                            .wpLeads(wpLeaderEntities)
                            .members(memberEntities)
                            .build();
                })
                .toList();
    }

    /**
     * Container class to store the associations that have been loaded for a paginated list of institutional pages.
     */
    @RequiredArgsConstructor
    public static class LoadedAssociations {
        public final Map<UUID, InstitutionalPageEntity> rootInstitutionalPageMap = new HashMap<>();
        public final Map<UUID, InstitutionalPageEntity> parentInstitutionalPageMap = new HashMap<>();
        public final Map<UUID, List<MemberEntity>> wpLeadsMap = new HashMap<>();
        public final Map<UUID, List<MemberEntity>> membersMap = new HashMap<>();
        public final Map<UUID, List<InstitutionalPageEntity>> childInstitutionalPagesMap = new HashMap<>();
        public final Map<UUID, List<ParagraphEntity>> paragraphMap = new HashMap<>();
    }

    /**
     * Container class to store the associations that have been mapped for single institutional page.
     */
    @Builder
    @Getter
    public static class Associations {
        public final String rootInstitutionalPageName;
        public final InstitutionalPageWithMembers parentInstitutionalPage;
        public final List<MemberEntity> wpLeads;
        public final List<MemberEntity> members;
        public final List<InstitutionalPageWithMembers> childInstitutionalPages;
        public final List<ParagraphEntity> paragraphs;
    }

    /**
     * Container class to store the related institutional page entity and its member associations
     */
    @Builder
    @Getter
    public static class InstitutionalPageWithMembers {
        public final InstitutionalPageEntity institutionalPage;
        public final List<MemberEntity> wpLeads;
        public final List<MemberEntity> members;
    }

}
