package com.finconsgroup.itserr.marketplace.institutionalpage.dm.api;

import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.InputParagraphDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.OutputInstitutionalPageDto;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.security.SecurityRequirement;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.validation.Valid;
import jakarta.validation.constraints.PositiveOrZero;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;

import java.util.UUID;

/**
 * This interface defines the contract for REST API endpoints related to Paragraph Management
 * for InstitutionalPages.
 *
 * <p>
 * It provides endpoints to add, update, and remove paragraphs as a member of a specific InstitutionalPage.
 * </p>
 *
 * <p>Example usage:
 * <pre>
 * POST    /api/v1/dm/institutional-page/institutional-pages/{institutionalPageId}/paragraphs                - Add a new paragraph.
 * PUT     /api/v1/dm/institutional-page/institutional-pages/{institutionalPageId}/paragraphs/{paragraphId}  - Update a paragraph.
 * DELETE  /api/v1/dm/institutional-page/institutional-pages/{institutionalPageId}/paragraphs/{paragraphId}  - Remove a paragraph.
 * </pre>
 * </p>
 */
@Tag(
        name = "Paragraph",
        description = "The Paragraph API: it provides endpoints for " +
                "adding, updating and removing paragraphs."
)
@SecurityRequirement(name = "BearerAuth")
public interface ParagraphApi {

    /**
     * Adds a new paragraph to the specified InstitutionalPage.
     *
     * @param institutionalPageId the id of the InstitutionalPage to add the paragraph to
     * @param inputParagraphDto   the input DTO containing paragraph details
     * @param position            (optional) the position at which to insert the paragraph (0-based, if not provided appends at the end)
     * @return the updated {@link OutputInstitutionalPageDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "Adds a new paragraph to the specified InstitutionalPage",
            responses = {
                    @ApiResponse(responseCode = "200", description = "OK"),
                    @ApiResponse(responseCode = "404", description = "Not Found"),
            }
    )
    @PostMapping(
            value = "/institutional-page/institutional-pages/{institutionalPageId}/paragraphs",
            consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.OK)
    OutputInstitutionalPageDto addParagraph(
            @PathVariable("institutionalPageId") UUID institutionalPageId,
            @RequestParam(name = "position", required = false) @PositiveOrZero Integer position,
            @Valid @RequestBody InputParagraphDto inputParagraphDto
    );

    /**
     * Updates a paragraph in the specified InstitutionalPage.
     *
     * @param institutionalPageId the id of the InstitutionalPage
     * @param paragraphId         the id of the paragraph to update
     * @param inputParagraphDto   the input DTO containing updated paragraph details
     * @return the updated {@link OutputInstitutionalPageDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "Updates a paragraph in the specified InstitutionalPage",
            responses = {
                    @ApiResponse(responseCode = "200", description = "OK"),
                    @ApiResponse(responseCode = "404", description = "Not Found"),
            }
    )
    @PutMapping(
            value = "/institutional-page/institutional-pages/{institutionalPageId}/paragraphs/{paragraphId}",
            consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.OK)
    OutputInstitutionalPageDto updateParagraph(
            @PathVariable("institutionalPageId") UUID institutionalPageId,
            @PathVariable("paragraphId") UUID paragraphId,
            @Valid @RequestBody InputParagraphDto inputParagraphDto
    );

    /**
     * Removes a paragraph from the specified InstitutionalPage.
     *
     * @param institutionalPageId the id of the InstitutionalPage
     * @param paragraphId         the id of the paragraph to remove
     * @return the updated {@link OutputInstitutionalPageDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "Removes a paragraph from the specified InstitutionalPage",
            responses = {
                    @ApiResponse(responseCode = "200", description = "OK"),
                    @ApiResponse(responseCode = "404", description = "Not Found"),
            }
    )
    @DeleteMapping(
            value = "/institutional-page/institutional-pages/{institutionalPageId}/paragraphs/{paragraphId}",
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.OK)
    OutputInstitutionalPageDto removeParagraph(
            @PathVariable("institutionalPageId") UUID institutionalPageId,
            @PathVariable("paragraphId") UUID paragraphId
    );

}
