package com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.impl;

import com.finconsgroup.itserr.marketplace.institutional_page.bs.config.properties.InstitutionalPageBsConfigurationProperties;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.NotificationProducer;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.data.InstitutionalPageCreatedNotificationData;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.data.InstitutionalPageDeletedNotificationData;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.data.InstitutionalPagePublicationRequestNotificationData;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.data.InstitutionalPageStatusChangeNotificationData;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.data.InstitutionalPageUpdatedNotificationData;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.data.JoinRequestNotificationData;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.data.JoinRequestStatusChangeNotificationData;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.data.MemberInvitationNotificationData;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.data.MemberInvitationStatusChangeNotificationData;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.data.MembershipChangeNotificationData;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.data.MembershipRemovalNotificationData;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.data.ParagraphCreatedNotificationData;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.data.ParagraphDeletedNotificationData;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.data.ParagraphUpdatedNotificationData;
import com.finconsgroup.itserr.messaging.dto.MessagingEventDto;
import com.finconsgroup.itserr.messaging.exception.WP2MessagingException;
import com.finconsgroup.itserr.messaging.producer.MessageProducer;
import com.finconsgroup.itserr.messaging.producer.ProducerRegistry;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Component;

import java.util.Map;

import static com.finconsgroup.itserr.marketplace.institutional_page.bs.util.Constants.SERVICE_FILTER_KEY;
import static com.finconsgroup.itserr.marketplace.institutional_page.bs.util.Constants.SERVICE_FILTER_VALUE;

/**
 * Default implementation of {@link NotificationProducer} that publishes notification events.
 */
@Component
@ConditionalOnProperty(name = "messaging.enabled", havingValue = "true")
@Slf4j
public class DefaultNotificationProducer implements NotificationProducer {

    private final MessageProducer messageProducer;

    private final InstitutionalPageBsConfigurationProperties.Messaging messagingProperties;

    /**
     * Constructs the DefaultNotificationProducer.
     *
     * @param institutionalPageBsConfigurationProperties    application-specific configuration properties
     * @param producerRegistry                              the registry providing message producers
     */
    public DefaultNotificationProducer(
            InstitutionalPageBsConfigurationProperties institutionalPageBsConfigurationProperties,
            ProducerRegistry producerRegistry
    ) {
        this.messagingProperties = institutionalPageBsConfigurationProperties.getMessaging();
        this.messageProducer = producerRegistry.getMessageProducer(this.messagingProperties.getNotificationProducerName());
    }

    @Override
    public void publishInstitutionalPageCreatedNotification(MessagingEventDto<InstitutionalPageCreatedNotificationData> message) {
        publish(message, messagingProperties.getCreatedNotificationType());
    }

    @Override
    public void publishInstitutionalPageUpdatedNotification(MessagingEventDto<InstitutionalPageUpdatedNotificationData> message) {
        publish(message, messagingProperties.getUpdatedNotificationType());
    }

    @Override
    public void publishInstitutionalPageDeletedNotification(MessagingEventDto<InstitutionalPageDeletedNotificationData> message) {
        publish(message, messagingProperties.getDeletedNotificationType());
    }

    @Override
    public void publishInstitutionalPageStatusChangeNotification(MessagingEventDto<InstitutionalPageStatusChangeNotificationData> message) {
        publish(message, messagingProperties.getStatusChangeNotificationType());
    }

    @Override
    public void publishInstitutionalPagePublicationRequestNotification(MessagingEventDto<InstitutionalPagePublicationRequestNotificationData> message) {
        publish(message, messagingProperties.getPublicationRequestNotificationType());
    }

    @Override
    public void publishJoinRequestNotification(MessagingEventDto<JoinRequestNotificationData> message) {
        publish(message, messagingProperties.getJoinRequestCreatedNotificationType());
    }

    @Override
    public void publishJoinRequestDeletedNotification(MessagingEventDto<JoinRequestNotificationData> message) {
        publish(message, messagingProperties.getJoinRequestDeletedNotificationType());
    }

    @Override
    public void publishJoinRequestStatusChangeNotification(MessagingEventDto<JoinRequestStatusChangeNotificationData> message) {
        publish(message, messagingProperties.getJoinRequestStatusChangeNotificationType());
    }

    @Override
    public void publishMemberInvitationNotification(MessagingEventDto<MemberInvitationNotificationData> message) {
        publish(message, messagingProperties.getMemberInvitationCreatedNotificationType());
    }

    @Override
    public void publishMemberInvitationDeletedNotification(MessagingEventDto<MemberInvitationNotificationData> message) {
        publish(message, messagingProperties.getMemberInvitationDeletedNotificationType());
    }

    @Override
    public void publishMemberInvitationStatusChangeNotification(MessagingEventDto<MemberInvitationStatusChangeNotificationData> message) {
        publish(message, messagingProperties.getMemberInvitationStatusChangeNotificationType());
    }

    @Override
    public void publishMembershipChangeNotification(MessagingEventDto<MembershipChangeNotificationData> message) {
        publish(message, messagingProperties.getMembershipChangeNotificationType());
    }

    @Override
    public void publishMembershipRemovalNotification(MessagingEventDto<MembershipRemovalNotificationData> message) {
        publish(message, messagingProperties.getMembershipRemovalNotificationType());
    }

    @Override
    public void publishParagraphCreatedNotification(MessagingEventDto<ParagraphCreatedNotificationData> message) {
        publish(message, messagingProperties.getParagraphCreatedNotificationType());
    }

    @Override
    public void publishParagraphUpdatedNotification(MessagingEventDto<ParagraphUpdatedNotificationData> message) {
        publish(message, messagingProperties.getParagraphUpdatedNotificationType());
    }

    @Override
    public void publishParagraphDeletedNotification(MessagingEventDto<ParagraphDeletedNotificationData> message) {
        publish(message, messagingProperties.getParagraphDeletedNotificationType());
    }

    private void publish(@NonNull MessagingEventDto<?> message, @NonNull String eventType) {
        try {
            Map<String, MessagingEventDto<?>> messageMap = Map.of(message.getId(), message);
            Map<String, String> filterProperties = Map.of(SERVICE_FILTER_KEY, SERVICE_FILTER_VALUE);
            log.debug("Sending notification message, messageMap: %s, eventType: %s, source: %s, filterProperties: %s"
                    .formatted(messageMap, eventType, messagingProperties.getSource(), filterProperties));
            messageProducer.publishEvent(messageMap, eventType, messagingProperties.getSource(), filterProperties);
            log.info("Successfully published notification event message for object with id: %s".formatted(message.getId()));
        } catch (Exception e) {
            String errorMessage = "Error occurred while sending notification event message: %s".formatted(e.getMessage());
            throw new WP2MessagingException(errorMessage);
        }
    }
}