package com.finconsgroup.itserr.marketplace.institutional_page.bs.client.bs.impl;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2AuthenticationException;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2BusinessException;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2ResourceNotFoundException;
import com.finconsgroup.itserr.marketplace.core.web.security.jwt.JwtTokenHolder;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.client.bs.FileUploadWorkspaceBsClient;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.client.bs.dto.DocumentWorkspaceBsDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.config.properties.WorkspaceBsConfigurationProperties;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Component;

import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.util.UUID;

/**
 * Default implementation of the {@link FileUploadWorkspaceBsClient} interface for handling file upload in the workspace.
 */
@Slf4j
@Component
@RequiredArgsConstructor
public class DefaultFileUploadWorkspaceBsClient implements FileUploadWorkspaceBsClient {
    private final WorkspaceBsConfigurationProperties workspaceBsConfig;

    @Override
    public DocumentWorkspaceBsDto createFile(InputStream file, UUID documentId, String fileName) {
        try (final HttpClient client = HttpClient.newHttpClient()) {
            // Build call URI
            final URI uri = URI.create(StringUtils.appendIfMissing(workspaceBsConfig.getBaseUrl().toString(), "/"))
                    .resolve("workspace/documents/")
                    .resolve(documentId + "/")
                    .resolve("content?fileName=" + fileName);

            final HttpRequest.BodyPublisher p = HttpRequest.BodyPublishers.ofInputStream(() -> file);

            // Build the HTTP request
            final HttpRequest request = HttpRequest.newBuilder()
                    .uri(uri)
                    .header("Authorization", "Bearer " + JwtTokenHolder.getTokenOrThrow())
                    .PUT(p)
                    .build();

            // Execute call
            final HttpResponse<String> response = executeRequest(client, request);

            // Check response status
            switch (response.statusCode()) {
                case 200, 201:
                    break;
                case 401:
                    throw new WP2AuthenticationException("Unauthenticated");
                case 404, 406:
                    throw new WP2ResourceNotFoundException("Document not found");
                default:
                    log.error("Error uploading file. workspace-bs status={}, workspace-bs body={}", response.statusCode(), response.body());
                    throw new WP2BusinessException("Error uploading file. status=" + response.statusCode());
            }

            // Return response body
            ObjectMapper mapper = new ObjectMapper();
            return mapper.readValue(response.body(), DocumentWorkspaceBsDto.class);
        } catch (IOException e) {
            throw new WP2BusinessException("I/O error while uploading document", e);
        }
    }

    private static HttpResponse<String> executeRequest(final HttpClient client, final HttpRequest request) throws IOException {
        try {
            return client.send(request, HttpResponse.BodyHandlers.ofString());
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new WP2BusinessException("Interrupted", e);
        }
    }
}