package com.finconsgroup.itserr.marketplace.favourite.user.bs.service.impl;

import com.finconsgroup.itserr.marketplace.core.web.dto.OutputPageDto;
import com.finconsgroup.itserr.marketplace.core.web.enums.SortDirection;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2ResourceNotFoundException;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.bean.DetailRequest;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.client.UserProfileDmClient;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.config.FavouriteUserBsProperties;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.dto.people.InputFindUserProfilesByIdsDto;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.dto.people.OutputPeopleFavouriteDto;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.dto.people.OutputUserProfileDto;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.enums.ItemContext;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.service.FavouriteUserItemDetailProvider;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.service.FavouriteUserItemDetailProviderRegistry;
import lombok.extern.slf4j.Slf4j;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.List;
import java.util.Objects;
import java.util.UUID;

/**
 * Implementation of {@link FavouriteUserItemDetailProvider} for Institutional Pages
 */
@Service
@Slf4j
public class FavouriteUserItemPeopleDetailProvider extends AbstractFavouriteUserItemDetailProvider<OutputPeopleFavouriteDto> {

    private final UserProfileDmClient userProfileDmClient;

    @SuppressWarnings("SpringJavaInjectionPointsAutowiringInspection")
    public FavouriteUserItemPeopleDetailProvider(
        @NonNull FavouriteUserItemDetailProviderRegistry favouriteUserItemDetailProviderRegistry,
        @NonNull UserProfileDmClient userProfileDmClient,
        @NonNull FavouriteUserBsProperties favouriteUserBsProperties) {
        super(favouriteUserItemDetailProviderRegistry, favouriteUserBsProperties, ItemContext.PEOPLE);
        Objects.requireNonNull(userProfileDmClient, "userProfileDmClient must not be null");

        this.userProfileDmClient = userProfileDmClient;
    }

    @NonNull
    @Override
    public OutputPeopleFavouriteDto getDetailById(@NonNull String itemId) {
        Objects.requireNonNull(itemId, "itemId must not be null");

        InputFindUserProfilesByIdsDto inputFindUserProfilesByIdsDto = InputFindUserProfilesByIdsDto
                                                                          .builder().ids(
                List.of(UUID.fromString(itemId))).build();
        OutputPageDto<OutputUserProfileDto> outputUserProfileDtos = userProfileDmClient.findAllByIds(
            inputFindUserProfilesByIdsDto, 0, 1, "id", SortDirection.ASC);

        if (CollectionUtils.isEmpty(outputUserProfileDtos.getContent())) {
            throw new WP2ResourceNotFoundException("Favourite user profile not found");
        }

        return mapFromUserProfileDto(outputUserProfileDtos.getContent().getFirst());
    }

    @NonNull
    @Override
    public OutputPageDto<OutputPeopleFavouriteDto> getDetails(@NonNull DetailRequest request) {
        // avoid calling the business service if there are no item ids in the request
        if (request.getItemIds() == null || request.getItemIds().isEmpty()) {
            return OutputPageDto.emptyWithPageSize(request.getPageRequestDto().getPageSize());
        }

        InputFindUserProfilesByIdsDto inputFindUserProfilesByIdsDto = InputFindUserProfilesByIdsDto
                                                                          .builder()
                                                                          .ids(request.getItemIds().stream().map(
                                                                              UUID::fromString).toList())
                                                                          .build();

        return userProfileDmClient.findAllByIds(
                                      inputFindUserProfilesByIdsDto,
                                      request.getPageRequestDto().getPageNumber(),
                                      request.getPageRequestDto().getPageSize(),
                                      request.getPageRequestDto().getSort(),
                                      request.getPageRequestDto().getDirection())
                                  .map(this::mapFromUserProfileDto);
    }

    /**
     * Maps the response from user profile domain service to favourite dto
     *
     * @param userProfileDto the user profile dto to map from
     * @return the mapped {@link OutputPeopleFavouriteDto}
     */
    private OutputPeopleFavouriteDto mapFromUserProfileDto(OutputUserProfileDto userProfileDto) {
        return OutputPeopleFavouriteDto
                   .builder()
                   .id(userProfileDto.getId())
                   .firstName(userProfileDto.getFirstName())
                   .lastName(userProfileDto.getLastName())
                   .imageUrl(userProfileDto.getImageUrl())
                   .orcid(userProfileDto.getOrcid())
                   .shortBio(userProfileDto.getShortBio())
                   .organizationAffiliation(userProfileDto.getOrganizationAffiliation())
                   .ssd(userProfileDto.getSsd())
                   .expertises(userProfileDto.getExpertises())
                   .creationTime(userProfileDto.getCreationTime())
                   .updateTime(userProfileDto.getUpdateTime())
                   .language(userProfileDto.getLanguage())
                   .build();
    }
}
