package com.finconsgroup.itserr.marketplace.event.dm.entity;

import com.finconsgroup.itserr.marketplace.core.entity.AbstractUUIDEntity;
import jakarta.persistence.CascadeType;
import jakarta.persistence.Column;
import jakarta.persistence.Entity;
import jakarta.persistence.FetchType;
import jakarta.persistence.JoinColumn;
import jakarta.persistence.OneToMany;
import jakarta.persistence.Table;
import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.NoArgsConstructor;
import lombok.Setter;
import lombok.ToString;
import lombok.experimental.SuperBuilder;

import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalTime;
import java.util.List;

import static com.finconsgroup.itserr.marketplace.event.dm.util.DomainConstants.MEDIUM_TEXT_LENGTH;

/**
 * Entity class representing an archived schedule in the event system.
 */
@SuppressWarnings("DefaultAnnotationParam")
@Entity
@Table(name = "archived_schedule")
@SuperBuilder
@Getter
@Setter
@EqualsAndHashCode(callSuper = true)
@NoArgsConstructor
public class ArchivedScheduleEntity extends AbstractUUIDEntity {

    /**
     * The title.
     */
    @Column(name = "title", nullable = false, length = MEDIUM_TEXT_LENGTH)
    private String title;

    /**
     * The description.
     */
    @Column(name = "description", length = MEDIUM_TEXT_LENGTH)
    private String description;

    /**
     * Start Date of the schedule.
     */
    @Column(name = "start_date", nullable = false)
    private LocalDate startDate;

    /**
     * Start Time of the schedule.
     */
    @Column(name = "start_time", nullable = false)
    private LocalTime startTime;

    /**
     * End Time of the schedule.
     */
    @Column(name = "end_time", nullable = false)
    private LocalTime endTime;

    /**
     * City name.
     */
    @Column(name = "location_city")
    private String locationCity;

    /**
     * Region/state name.
     */
    @Column(name = "location_region")
    private String locationRegion;

    /**
     * Country name.
     */
    @Column(name = "location_country")
    private String locationCountry;

    /**
     * Postal/ZIP code.
     */
    @Column(name = "location_zip_code")
    private String locationZipCode;

    /**
     * The timestamp when the event was created.
     */
    @Column(name = "creation_time", nullable = false)
    private Instant creationTime;

    /**
     * The timestamp when the event was last updated.
     */
    @Column(name = "update_time", nullable = false)
    private Instant updateTime;

    /**
     * The version field used for optimistic locking.
     * <p>
     * This value is automatically managed by JPA to detect concurrent updates.
     * Each time the entity is updated, the version is incremented.
     * If two transactions try to update the same entity simultaneously,
     * JPA will detect the conflict based on this version
     * and throw an {@link jakarta.persistence.OptimisticLockException}.
     */
    @Column(name = "version", nullable = false)
    private long version;

    /**
     * The ordering column.
     */
    @Column(name = "schedule_order", nullable = false)
    private long scheduleOrder;

    /**
     * The programs related to the schedule.
     */
    @OneToMany(fetch = FetchType.LAZY, cascade = CascadeType.ALL, orphanRemoval = true)
    @JoinColumn(name = "schedule_id", nullable = false, updatable = false)
    @ToString.Exclude
    private List<ArchivedProgramEntity> programs;

    @Override
    public void prePersist() {
        super.prePersist();
        Instant now = Instant.now();
        if (creationTime == null) {
            creationTime = now;
        }
        if (updateTime == null) {
            updateTime = now;
        }
    }

}
