package com.finconsgroup.itserr.marketplace.event.bs.controller;

import com.finconsgroup.itserr.marketplace.core.web.dto.OutputPageDto;
import com.finconsgroup.itserr.marketplace.core.web.dto.PageRequestDto;
import com.finconsgroup.itserr.marketplace.core.web.enums.SortDirection;
import com.finconsgroup.itserr.marketplace.event.bs.api.EventApi;
import com.finconsgroup.itserr.marketplace.event.bs.dto.InputCreateEventDto;
import com.finconsgroup.itserr.marketplace.event.bs.dto.InputProgramSubscribedParticipantDto;
import com.finconsgroup.itserr.marketplace.event.bs.dto.InputUpdateEventDto;
import com.finconsgroup.itserr.marketplace.event.bs.dto.OutputEventDto;
import com.finconsgroup.itserr.marketplace.event.bs.service.EventService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.propertyeditors.StringTrimmerEditor;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.RestController;

import java.util.Set;
import java.util.UUID;

/**
 * REST controller class for handling user event related API requests.
 *
 * <p>Implements the {@link EventApi} interface.</p>
 */
@Slf4j
@RequiredArgsConstructor
@RestController
public class EventController implements EventApi {

    private final EventService eventService;

    @InitBinder
    public void initBinder(WebDataBinder binder) {
        // trim strings before triggering validation, but do not convert to null
        // as otherwise it throws - MissingServletRequestParameterException
        binder.registerCustomEditor(String.class, new StringTrimmerEditor(false));
    }

    @Override
    public OutputPageDto<OutputEventDto> findAll(
            final Set<String> associationsToLoad,
            final int pageNumber,
            final int pageSize,
            final String sort,
            final SortDirection direction
    ) {
        PageRequestDto pageRequest = PageRequestDto.builder().pageNumber(pageNumber).pageSize(pageSize).direction(direction).sort(sort).build();
        return eventService.findAll(associationsToLoad, pageRequest);
    }

    @Override
    public OutputEventDto findById(final UUID eventId) {
        return eventService.findById(eventId);
    }

    @Override
    public OutputEventDto createEvent(final InputCreateEventDto inputCreateEventDto) {
        return eventService.create(inputCreateEventDto);
    }

    @Override
    public OutputEventDto updateEventById(
            final UUID eventId,
            final InputUpdateEventDto inputUpdateEventDto
    ) {
        return eventService.updateById(eventId, inputUpdateEventDto);
    }

    @Override
    public void deleteEventById(final UUID eventId) {
        eventService.deleteById(eventId);
    }

    @Override
    public OutputEventDto register(UUID id) {
        return eventService.register(id);
    }

    @Override
    public OutputEventDto unregister(UUID id) {
        return eventService.unregister(id);
    }

    @Override
    public OutputEventDto registerProgram(UUID eventId, UUID programId, InputProgramSubscribedParticipantDto inputProgramSubscribedParticipantDto) {
        return eventService.registerProgram(eventId, programId, inputProgramSubscribedParticipantDto);
    }

    @Override
    public OutputEventDto unregisterProgram(UUID eventId, UUID programId) {
        return eventService.unregisterProgram(eventId, programId);
    }
}
