package com.finconsgroup.itserr.marketplace.event.bs.controller;

import com.finconsgroup.itserr.marketplace.core.web.exception.WP2ResourceNotFoundException;
import com.finconsgroup.itserr.marketplace.event.bs.api.DiagnosticsApi;
import com.finconsgroup.itserr.marketplace.event.bs.bean.EventApplicationEvent;
import com.finconsgroup.itserr.marketplace.event.bs.bean.EventSubscriptionApplicationEvent;
import com.finconsgroup.itserr.marketplace.event.bs.configuration.properties.EventBsConfigurationProperties;
import com.finconsgroup.itserr.marketplace.event.bs.dto.OutputEventDto;
import com.finconsgroup.itserr.marketplace.event.bs.enums.MessagingEventType;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.core.LoggerContext;
import org.apache.logging.log4j.core.appender.RollingFileAppender;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.ApplicationEventPublisher;
import org.springframework.core.io.FileSystemResource;
import org.springframework.core.io.Resource;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.RestController;

import java.io.File;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;

/**
 * REST controller class for handling debugging/analysis related API requests.
 *
 * <p>Implements the {@link DiagnosticsApi} interface.</p>
 */
@Slf4j
@RestController
@RequiredArgsConstructor
public class DiagnosticsController implements DiagnosticsApi {

    private final ApplicationEventPublisher applicationEventPublisher;
    private final EventBsConfigurationProperties eventBsConfigurationProperties;

    // For storing rabbitmq test messages
    private static final Map<String, OutputEventDto> itemMap = new HashMap<>();

    // The name of the RollingFile Log4j2 component
    // It is not the name of the log file
    // We use this to dynamically retrieve the name of the log file.
    @Value("${log.log4j2-rolling-file-name}")
    private String log4j2RollingFileName;

    @Override
    public ResponseEntity<Resource> downloadLogs() {
        log.debug("call to DiagnosticsController - downloadLogs");
        Path filePath = Paths.get(getLogFilePathFromLog4j2());
        log.debug("Trying to retrieve log file from: {}", filePath);
        File logFile = filePath.toFile();
        if (!logFile.exists() || !logFile.isFile()) {
            throw new WP2ResourceNotFoundException("Log file not found: %s".formatted(logFile.getAbsolutePath()));
        }
        Resource resource = new FileSystemResource(logFile);
        return ResponseEntity.ok()
                .header(HttpHeaders.CONTENT_DISPOSITION, getContentDispositionHeaderValue(logFile.getName()))
                .contentType(MediaType.APPLICATION_OCTET_STREAM)
                .body(resource);
    }

    // private

    public String getLogFilePathFromLog4j2() {
        LoggerContext context = (LoggerContext) LogManager.getContext(false);
        RollingFileAppender appender = context.getConfiguration().getAppender(log4j2RollingFileName);
        return appender.getFileName();
    }

    private static String getContentDispositionHeaderValue(String fileName) {
        return "attachment; filename=\"%s\"".formatted(fileName);
    }

    @Override
    public ResponseEntity<String> publishMessage(MessagingEventType messagingEventType, String itemName) {
        if (!eventBsConfigurationProperties.getDiagnostics().isPublishMessageEnabled()) {
            log.warn("Attempt to access disabled diagnostics endpoint");
            return ResponseEntity.status(HttpStatus.FORBIDDEN)
                    .body("This endpoint is disabled in the current environment");
        }

        log.debug("call to DiagnosticsController - publishMessage, name: %s".formatted(itemName));

        OutputEventDto eventDto;
        switch (messagingEventType) {
            case CREATED:
                eventDto = OutputEventDto.builder()
                        .id(UUID.randomUUID())
                        .title(itemName)
                        .content("CREATED News content")
                        .build();
                log.info("sending created event message, eventDto: {}", eventDto);
                itemMap.put(eventDto.getTitle(), eventDto);
                applicationEventPublisher.publishEvent(new EventApplicationEvent(eventDto, MessagingEventType.CREATED));
                break;

            case UPDATED:
                eventDto = itemMap.get(itemName);
                if (eventDto == null) {
                    return ResponseEntity.badRequest()
                            .body("no test item created for itemName: %s".formatted(itemName));
                }
                eventDto.setContent("the UPDATED content");
                log.info("sending updated event message, eventDto: {}", eventDto);
                applicationEventPublisher.publishEvent(new EventApplicationEvent(eventDto, MessagingEventType.UPDATED));
                break;

            case DELETED:
                eventDto = itemMap.get(itemName);
                if (eventDto == null) {
                    return ResponseEntity.badRequest()
                            .body("no test item created for itemName: %s".formatted(itemName));
                }
                log.info("sending deleted message, eventDto: {}", eventDto);
                applicationEventPublisher.publishEvent(new EventApplicationEvent(eventDto, MessagingEventType.DELETED));
                itemMap.remove(eventDto.getTitle());
                break;

            case REGISTER:
                eventDto = itemMap.get(itemName);
                if (eventDto == null) {
                    return ResponseEntity.badRequest()
                            .body("no test item created for itemName: %s".formatted(itemName));
                }
                log.info("sending register message, eventDto: {}", eventDto);
                applicationEventPublisher.publishEvent(new EventSubscriptionApplicationEvent(eventDto, MessagingEventType.REGISTER));
                break;

            case UNREGISTER:
                eventDto = itemMap.get(itemName);
                if (eventDto == null) {
                    return ResponseEntity.badRequest()
                            .body("no test item created for itemName: %s".formatted(itemName));
                }
                log.info("sending unregister message, eventDto: {}", eventDto);
                applicationEventPublisher.publishEvent(new EventSubscriptionApplicationEvent(eventDto, MessagingEventType.UNREGISTER));
                break;

            case null, default:
                return ResponseEntity.badRequest()
                        .body("Event type must be one of: CREATED, UPDATED, DELETED, REGISTER or UNREGISTER");
        }
        return ResponseEntity.ok("message sent correctly, eventDto: %s".formatted(eventDto));
    }
}