package com.finconsgroup.itserr.marketplace.discussion.bs.service.impl;

import com.finconsgroup.itserr.marketplace.core.web.security.jwt.JwtTokenHolder;
import com.finconsgroup.itserr.marketplace.discussion.bs.bean.DiscussionApplicationEvent;
import com.finconsgroup.itserr.marketplace.discussion.bs.bean.DiscussionThreadApplicationEvent;
import com.finconsgroup.itserr.marketplace.discussion.bs.bean.DiscussionThreadDetails;
import com.finconsgroup.itserr.marketplace.discussion.bs.client.dm.DiscussionDmClient;
import com.finconsgroup.itserr.marketplace.discussion.bs.client.dm.UserProfileDmClient;
import com.finconsgroup.itserr.marketplace.discussion.bs.dto.DiscussionDTO;
import com.finconsgroup.itserr.marketplace.discussion.bs.dto.DiscussionReactionDTO;
import com.finconsgroup.itserr.marketplace.discussion.bs.dto.InputFindUserProfilesByPrincipalsDto;
import com.finconsgroup.itserr.marketplace.discussion.bs.dto.InputUpdateDiscussionDto;
import com.finconsgroup.itserr.marketplace.discussion.bs.dto.OutputUserProfileDto;
import com.finconsgroup.itserr.marketplace.discussion.bs.dto.ThreadDTO;
import com.finconsgroup.itserr.marketplace.discussion.bs.dto.ThreadReactionDTO;
import com.finconsgroup.itserr.marketplace.discussion.bs.dto.UserInfoDTO;
import com.finconsgroup.itserr.marketplace.discussion.bs.enums.MessagingEventType;
import com.finconsgroup.itserr.marketplace.discussion.bs.enums.ReactionType;
import com.finconsgroup.itserr.marketplace.discussion.bs.enums.ResourceType;
import com.finconsgroup.itserr.marketplace.discussion.bs.mapper.EventToDiscussionMapper;
import com.finconsgroup.itserr.marketplace.discussion.bs.mapper.UserProfileMapper;
import com.finconsgroup.itserr.marketplace.discussion.bs.service.DiscussionService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.jetbrains.annotations.NotNull;
import org.springframework.context.ApplicationEventPublisher;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Sort;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * Default implementation of {@link com.finconsgroup.itserr.marketplace.discussion.bs.service.DiscussionService}.
 *
 * <p>It delegates CRUD operations to the Discussion DM client and enriches
 * discussion results with user profile information retrieved via the
 * UserProfile client.</p>
 */
@Service
@Slf4j
@RequiredArgsConstructor
public class DefaultDiscussionService implements DiscussionService {
    /**
     * Client for delegating discussion operations to the Discussion DM service.
     */
    private final DiscussionDmClient discussionDmClient;
    /**
     * Client used to retrieve user profile information and enrich discussions.
     */
    private final UserProfileDmClient userProfileDmClient;
    /**
     * Mapper used to convert user profile representations to {@link UserInfoDTO}.
     */
    private final UserProfileMapper userProfileMapper; // Inject the mapper

    private final ApplicationEventPublisher applicationEventPublisher;

    private final EventToDiscussionMapper eventToDiscussionMapper;


    /**
     * Retrieves discussions from DM service and enriches them with user profile information when available.
     *
     * @param page           page number (0-based)
     * @param size           page size
     * @param sortBy         sorting field
     * @param direction      sort direction (asc|desc)
     * @param visibility     optional visibility filter
     * @param includeThreads whether to include threads
     * @return a page of discussions, potentially enriched with {@link UserInfoDTO}
     */
    @Override
    public Page<DiscussionDTO> getDiscussions(int page, int size, String sortBy, String direction, String visibility, boolean includeThreads) {
        Page<DiscussionDTO> discussionDTOs = discussionDmClient.getDiscussions(page, size, sortBy, direction, visibility, includeThreads);
        fillDiscussionDTOs(includeThreads, true, discussionDTOs.getContent());
        return discussionDTOs;
    }

    @Override
    public DiscussionDTO getDiscussionForResource(String resourceId, ResourceType resourceType) {
        DiscussionDTO discussionDTO = discussionDmClient.getDiscussionForResource(resourceId, resourceType.getValue());
        List<String> preferredNames = new ArrayList<>();

        preferredNames.add(discussionDTO.getCreatedBy());
        preferredNames.addAll(getListPreferredNameFromDiscussionReaction(discussionDTO));

        Optional.ofNullable(discussionDTO.getThreads())
                .orElse(List.of())
                .forEach(thread -> {
                    preferredNames.addAll(collectPrincipalsRecursively(thread));
                    preferredNames.addAll(collectReactionUserIdsRecursively(thread));
                });

        if (!preferredNames.isEmpty()) {
            InputFindUserProfilesByPrincipalsDto inputDto = InputFindUserProfilesByPrincipalsDto.builder()
                    .principals(preferredNames)
                    .build();
            int profilesPage = 0;
            int profilesSize = preferredNames.size();

            var profiles = userProfileDmClient.findAllByPrincipals(inputDto, profilesPage, profilesSize, "preferredUsername", "ASC")
                    .getContent();

            var usernameToUserInfo = profiles.stream()
                    .filter(p -> p.getPreferredUsername() != null && !p.getPreferredUsername().isBlank())
                    .collect(Collectors.toMap(
                            p -> p.getPreferredUsername(),
                            p -> userProfileMapper.toUserInfoDTO(p),
                            (a, b) -> a));

            setUserInfoDiscussion(usernameToUserInfo, discussionDTO);
            setDiscussionReactionUserInfo(usernameToUserInfo, discussionDTO);
            setThreadAndReactionUserInfo(true, usernameToUserInfo, discussionDTO);
        }

        return discussionDTO;
    }

    @Override
    public Page<ThreadDTO> getThreadsByDiscussionId(String discussionId, int page, int size, String sortBy, Sort.Direction direction) {
        Page<ThreadDTO> threadDTOS = discussionDmClient.getThreadsByDiscussionId(discussionId, page, size, sortBy, direction);

        List<String> createdByList = threadDTOS.getContent().stream()
                .flatMap(thread -> collectPrincipalsRecursively(thread).stream())
                .filter(createdBy -> createdBy != null && !createdBy.isBlank())
                .distinct()
                .collect(Collectors.toList());

        // Collect user IDs from thread reactions (recursively)
        List<String> reactionUserIds = threadDTOS.getContent().stream()
                .flatMap(thread -> collectReactionUserIdsRecursively(thread).stream())
                .filter(userId -> userId != null && !userId.isBlank())
                .distinct()
                .collect(Collectors.toList());

        // Combine all user IDs
        createdByList.addAll(reactionUserIds);
        createdByList = createdByList.stream().distinct().collect(Collectors.toList());

        if (!createdByList.isEmpty()) {
            InputFindUserProfilesByPrincipalsDto inputDto = InputFindUserProfilesByPrincipalsDto.builder()
                    .principals(createdByList)
                    .build();

            int profilesPage = 0;
            int profilesSize = createdByList.size();

            var profiles = userProfileDmClient.findAllByPrincipals(inputDto, profilesPage, profilesSize, "preferredUsername", String.valueOf(direction)).getContent();

            var usernameToUserInfo = profiles.stream()
                    .filter(p -> p.getPreferredUsername() != null && !p.getPreferredUsername().isBlank())
                    .collect(Collectors.toMap(
                            p -> p.getPreferredUsername(),
                            p -> userProfileMapper.toUserInfoDTO(p),
                            (a, b) -> a
                    ));

            threadDTOS.getContent().forEach(thread -> {
                applyUserInfoRecursively(thread, usernameToUserInfo);
                applyReactionUserInfoRecursively(thread, usernameToUserInfo);
            });
        }

        return threadDTOS;
    }


    private List<String> collectPrincipalsRecursively(ThreadDTO thread) {
        java.util.ArrayList<String> list = new java.util.ArrayList<>();
        if (thread == null) return list;
        if (thread.getCreatedBy() != null && !thread.getCreatedBy().isBlank()) {
            list.add(thread.getCreatedBy());
        }
        if (thread.getReplies() != null) {
            for (ThreadDTO reply : thread.getReplies()) {
                list.addAll(collectPrincipalsRecursively(reply));
            }
        }
        return list;
    }

    private void applyUserInfoRecursively(ThreadDTO thread, java.util.Map<String, UserInfoDTO> usernameToUserInfo) {
        if (thread == null || usernameToUserInfo == null) return;
        String createdBy = thread.getCreatedBy();
        if (createdBy != null) {
            UserInfoDTO info = usernameToUserInfo.get(createdBy);
            if (info != null) {
                thread.setUserInfoDTO(info);
            }
        }
        if (thread.getReplies() != null) {
            for (ThreadDTO reply : thread.getReplies()) {
                applyUserInfoRecursively(reply, usernameToUserInfo);
            }
        }
    }

    /**
     * Delegates creation of a new discussion to the DM service.
     *
     * @param discussionDTO the discussion payload to create
     * @return HTTP 201 Created on success, or the response produced by the DM client
     */
    @Override
    public DiscussionDTO createDiscussion(DiscussionDTO discussionDTO) {
        //FIXME Add notification logic
        String username = JwtTokenHolder.getPreferredUsernameOrThrow();
        discussionDTO.setCreatedBy(username);
        discussionDTO.setUpdatedBy(username);
        DiscussionDTO outputDiscussionDTO = discussionDmClient.createDiscussion(discussionDTO);
        fillDiscussionDTOs(false, false, List.of(outputDiscussionDTO));
        applicationEventPublisher.publishEvent(new DiscussionApplicationEvent(outputDiscussionDTO,
                MessagingEventType.CREATED));
        return outputDiscussionDTO;
    }

    /**
     * Updates an existing discussion with the provided data.
     *
     * @param id            the unique identifier of the discussion to be updated
     * @param discussionDTO the object containing the updated discussion details
     * @return a ResponseEntity containing the updated DiscussionDTO object
     */
    @Override
    public DiscussionDTO updateDiscussion(String id, InputUpdateDiscussionDto discussionDTO) {
        DiscussionDTO updatedDiscussionDTO = discussionDmClient.updateDiscussion(id, discussionDTO, false);
        log.info("updated discussion: {}", updatedDiscussionDTO);
        fillDiscussionDTOs(false, true, List.of(updatedDiscussionDTO));
        applicationEventPublisher.publishEvent(new DiscussionApplicationEvent(updatedDiscussionDTO,
                MessagingEventType.UPDATED));
        return updatedDiscussionDTO;
    }

    /**
     * Delegates the deletion of a discussion to the DM client.
     *
     * @param id the discussion identifier
     * @return response from DM client (typically 204 No Content)
     */
    @Override
    public ResponseEntity<Void> deleteDiscussion(String id) {
        ResponseEntity<Void> response = discussionDmClient.deleteDiscussion(id);
        DiscussionDTO discussionDTO = DiscussionDTO.builder().id(UUID.fromString(id)).build();
        applicationEventPublisher.publishEvent(new DiscussionApplicationEvent(discussionDTO,
                MessagingEventType.DELETED));
        return response;
    }

    @Override
    public DiscussionDTO getDiscussion(String id) {
        DiscussionDTO discussionDTO = discussionDmClient.getDiscussionById(id, false);
        fillDiscussionDTOs(false, true, List.of(discussionDTO));
        return discussionDTO;
    }

    /**
     * Delegates adding a new thread to a discussion to the DM client.
     * After successful creation, publishes a notification event to inform relevant users.
     *
     * <p>Notification logic:
     * <ul>
     *   <li>If the thread is a reply (parentId != null): notifies both the parent thread creator and discussion creator</li>
     *   <li>If the thread is top-level (parentId == null): notifies the discussion creator</li>
     * </ul>
     *
     * <p>No notification is sent if the current user is the same as the user being notified.
     *
     * @param discussionId the discussion identifier
     * @param threadDTO    the thread payload to add
     * @return response from DM client (typically 201 Created)
     */
    @Override
    public ResponseEntity<ThreadDTO> addThread(String discussionId, ThreadDTO threadDTO) {

        ResponseEntity<ThreadDTO> response = discussionDmClient.addThread(discussionId, threadDTO);
        if (response.getStatusCode().is2xxSuccessful()) {
            Optional.of(discussionDmClient.getDiscussionByDiscussionId(discussionId))
                    .ifPresent(discussion -> {
                        Set<String> usersToNotify = collectUsersToNotify(discussion, threadDTO);

                        DiscussionThreadDetails discussionThreadDetails =
                                eventToDiscussionMapper.toDiscussionThreadDetails(response.getBody(), discussion, usersToNotify);

                        fillDiscussionDTOs(false, true, List.of(discussion));
                        applicationEventPublisher.publishEvent(
                                new DiscussionThreadApplicationEvent(discussion, discussionThreadDetails, MessagingEventType.THREAD_CREATED)
                        );
                    });
        }

        return ResponseEntity.status(response.getStatusCode()).body(response.getBody());
    }

    private Set<String> collectUsersToNotify(DiscussionDTO discussion, ThreadDTO threadDTO) {

        Set<String> usersToNotify = new HashSet<>();

        // Add discussion creator
        Optional.ofNullable(discussion.getCreatedBy())
                .ifPresent(usersToNotify::add);

        /*
         * Threads with a parentId are replies of a thread
         * In this case we are adding in addition thread creator and all repliers
         */

        if (threadDTO.getParentId() != null) {
            addInvolvedUsers(discussion.getThreads(), threadDTO.getParentId(), usersToNotify);
        }

        //remove self
        usersToNotify = usersToNotify.stream()
                .filter(username -> !username.equals(getCurrentUserPreferredUsername()))
                .collect(Collectors.toSet());

        log.info("collected users to be notified are: {}", String.join(", ", usersToNotify));
        return usersToNotify;
    }

    private void addInvolvedUsers(List<ThreadDTO> allDiscussionThreads, UUID parentThreadId, Set<String> usersToNotify) {

        flattenThreads(allDiscussionThreads)
                .filter(thread -> thread.getId().equals(parentThreadId))
                .findFirst()
                .ifPresent(parentThread -> {

                    // Add thread parent creator
                    if (parentThread.getCreatedBy() != null && !parentThread.getCreatedBy().isEmpty()) {
                        usersToNotify.add(parentThread.getCreatedBy());
                    }

                    // Add repliers creator
                    if (parentThread.getReplies() != null) {
                        parentThread.getReplies().stream()
                                .map(ThreadDTO::getCreatedBy)
                                .filter(Objects::nonNull)
                                .filter(createdBy -> !createdBy.isEmpty())
                                .forEach(usersToNotify::add);
                    }
                });
    }

    private Stream<ThreadDTO> flattenThreads(List<ThreadDTO> threads) {
        return threads == null ? Stream.empty() :
                threads.stream().flatMap(reply -> Stream.concat(
                        Stream.of(reply),
                        flattenThreads(reply.getReplies())
                ));
    }

    private String getCurrentUserPreferredUsername() {
        return JwtTokenHolder.getPreferredUsername().orElse(null);
    }

    /**
     * Delegates deletion of a thread from a discussion to the DM client.
     *
     * @param discussionId the discussion identifier
     * @param threadId     the thread identifier
     * @return response from DM client (typically 204 No Content)
     */
    @Override
    public ResponseEntity<Void> deleteThread(String discussionId, String threadId) {
        ResponseEntity<Void> response = discussionDmClient.deleteThread(discussionId, threadId);
        // publish update event as threadsCount will be updated
        DiscussionDTO discussionDTO = discussionDmClient.getDiscussionById(discussionId, false);
        applicationEventPublisher.publishEvent(new DiscussionApplicationEvent(discussionDTO,
                MessagingEventType.UPDATED));
        return response;
    }

    /**
     * Delegates updating a thread within a discussion to the DM client.
     *
     * @param discussionId the discussion identifier
     * @param threadId     the thread identifier
     * @param threadDTO    the updated thread payload
     * @return response from DM client (e.g., 200 OK or 204 No Content)
     */
    @Override
    public ResponseEntity<ThreadDTO> updateThread(String discussionId, String threadId, ThreadDTO threadDTO) {
        return discussionDmClient.updateThread(discussionId, threadId, threadDTO);
    }

    /**
     * Delegates adding a reaction to a discussion to the DM client.
     *
     * @param discussionId the discussion identifier
     * @param userId       the user applying the reaction
     * @param reactionType the reaction to apply
     * @return response from DM client (e.g., 200 OK or 201 Created)
     */
    @Override
    public ResponseEntity<Void> addReactionToDiscussion(String discussionId, String userId, ReactionType reactionType) {
        var response = discussionDmClient.addReactionToDiscussion(discussionId, userId, reactionType);
        // publish update event as reactionsCount will be updated
        DiscussionDTO discussionDTO = discussionDmClient.getDiscussionById(discussionId, false);
        applicationEventPublisher.publishEvent(new DiscussionApplicationEvent(discussionDTO,
                MessagingEventType.UPDATED));
        return response;
    }

    /**
     * Delegates removal of a user's reaction from a discussion to the DM client.
     *
     * @param discussionId the discussion identifier
     * @param userId       the user whose reaction should be removed
     * @return response from DM client (typically 204 No Content)
     */
    @Override
    public ResponseEntity<Void> removeReactionFromDiscussion(String discussionId, String userId) {
        var response = discussionDmClient.removeReactionFromDiscussion(discussionId, userId);
        // publish update event as reactionsCount will be updated
        DiscussionDTO discussionDTO = discussionDmClient.getDiscussionById(discussionId, false);
        applicationEventPublisher.publishEvent(new DiscussionApplicationEvent(discussionDTO,
                MessagingEventType.UPDATED));
        return response;
    }

    /**
     * Delegates adding a reaction to a thread to the DM client.
     *
     * @param discussionId the discussion identifier
     * @param threadId     the thread identifier
     * @param userId       the user applying the reaction
     * @param reactionType the reaction to apply
     * @return response from DM client (e.g., 200 OK or 201 Created)
     */
    @Override
    public ResponseEntity<Void> addReactionToThread(String discussionId, String threadId, String userId, ReactionType reactionType) {
        return discussionDmClient.addReactionToThread(discussionId, threadId, userId, reactionType);
    }

    /**
     * Delegates removal of a user's reaction from a thread to the DM client.
     *
     * @param discussionId the discussion identifier
     * @param threadId     the thread identifier
     * @param userId       the user whose reaction should be removed
     * @return response from DM client (typically 204 No Content)
     */
    @Override
    public ResponseEntity<Void> removeReactionFromThread(String discussionId, String threadId, String userId) {
        return discussionDmClient.removeReactionFromThread(discussionId, threadId, userId);
    }

    private void fillDiscussionDTOs(boolean includeThreads, boolean includeReactions, List<DiscussionDTO> discussionDTOs) {
        // If includeThreads is true, also collect user IDs from threads, replies, and their reactions
        List<String> threadCreatedByIds = new ArrayList<>();
        List<String> threadReactionUserIds = new ArrayList<>();

        List<String> createdByIds = getListOfPreferredNameFromDiscussion(discussionDTOs);
        List<String> discussionReactionUserIds = includeReactions ? getListPreferredNameFromDiscussionReaction(discussionDTOs) : List.of();

        if (includeThreads) {
            threadCreatedByIds = getListPreferredNameFromThreads(discussionDTOs);
            if (includeReactions) {
                threadReactionUserIds = getListPreferredNameFromThreadReaction(discussionDTOs);
            }
        }

        // Combine all user IDs
        createdByIds.addAll(discussionReactionUserIds);
        createdByIds.addAll(threadCreatedByIds);
        createdByIds.addAll(threadReactionUserIds);
        createdByIds = createdByIds.stream().distinct().collect(Collectors.toList());

        if (!createdByIds.isEmpty()) {
            var usernameToUserInfo = loadUserInfo(createdByIds);

            discussionDTOs.forEach(discussion -> {
                // Enrich discussion creator
                setUserInfoDiscussion(usernameToUserInfo, discussion);

                // Enrich discussion reactions
                setDiscussionReactionUserInfo(usernameToUserInfo, discussion);

                // Enrich threads and their reactions if includeThreads is true
                setThreadAndReactionUserInfo(includeThreads, usernameToUserInfo, discussion);
            });
        }

    }

    private Map<String, UserInfoDTO> loadUserInfo(List<String> usernames) {
        InputFindUserProfilesByPrincipalsDto inputDto = InputFindUserProfilesByPrincipalsDto.builder()
                .principals(usernames)
                .build();
        Map<String, UserInfoDTO> usernameToUserInfo;
        try {
            int profilesPage = 0;
            int profilesSize = usernames.size();

            var profiles = userProfileDmClient.findAllByPrincipals(inputDto, profilesPage, profilesSize, "preferredUsername", "ASC")
                    .getContent();

            // Create map for efficient lookup
            usernameToUserInfo = profiles.stream()
                    .filter(p -> p.getPreferredUsername() != null && !p.getPreferredUsername().isBlank())
                    .collect(Collectors.toMap(
                            OutputUserProfileDto::getPreferredUsername,
                            userProfileMapper::toUserInfoDTO,
                            (a, b) -> a));
        } catch (Exception e) {
            log.error("Error while fetching user information: {}", e.getMessage(), e);
            usernameToUserInfo = usernames.stream()
                    .collect(Collectors.toMap(
                            Function.identity(),
                            username -> UserInfoDTO.builder().preferredUsername(username).build()));
        }
        return usernameToUserInfo;
    }

    private void setThreadAndReactionUserInfo(boolean includeThreads, Map<String, UserInfoDTO> usernameToUserInfo, DiscussionDTO discussion) {
        if (includeThreads && discussion.getThreads() != null) {
            discussion.getThreads().forEach(thread -> {
                applyUserInfoRecursively(thread, usernameToUserInfo);
                applyReactionUserInfoRecursively(thread, usernameToUserInfo);
            });
        }
    }

    private static void setDiscussionReactionUserInfo(Map<String, UserInfoDTO> usernameToUserInfo, DiscussionDTO discussion) {
        if (discussion.getReactions() != null) {
            discussion.getReactions().forEach(reaction -> {
                if (reaction.getUserId() != null) {
                    UserInfoDTO userInfo = usernameToUserInfo.get(reaction.getUserId());
                    if (userInfo != null) {
                        reaction.setUserInfoDTO(userInfo);
                    }
                }
            });
        }
    }

    private static void setUserInfoDiscussion(Map<String, UserInfoDTO> usernameToUserInfo, DiscussionDTO discussion) {
        String createdBy = discussion.getCreatedBy();
        if (createdBy != null) {
            UserInfoDTO userInfo = usernameToUserInfo.get(createdBy);
            if (userInfo != null) {
                discussion.setUserInfoDTO(userInfo);
            }
        }
    }

    @NotNull
    private List<String> getListPreferredNameFromThreadReaction(List<DiscussionDTO> discussionDTOs) {
        List<String> threadReactionUserIds;
        threadReactionUserIds = discussionDTOs
                .stream()
                .filter(discussion -> discussion.getThreads() != null)
                .flatMap(discussion -> discussion.getThreads().stream())
                .flatMap(thread -> collectReactionUserIdsRecursively(thread).stream())
                .filter(userId -> userId != null && !userId.isBlank())
                .distinct()
                .collect(Collectors.toList());
        return threadReactionUserIds;
    }

    @NotNull
    private List<String> getListPreferredNameFromThreads(List<DiscussionDTO> discussionDTOs) {
        List<String> threadCreatedByIds;
        threadCreatedByIds = discussionDTOs
                .stream()
                .filter(discussion -> discussion.getThreads() != null)
                .flatMap(discussion -> discussion.getThreads().stream())
                .flatMap(thread -> collectPrincipalsRecursively(thread).stream())
                .filter(createdBy -> createdBy != null && !createdBy.isBlank())
                .distinct()
                .collect(Collectors.toList());
        return threadCreatedByIds;
    }

    @NotNull
    private static List<String> getListPreferredNameFromDiscussionReaction(List<DiscussionDTO> discussionDTOs) {
        List<String> discussionReactionUserIds = discussionDTOs
                .stream()
                .filter(discussion -> discussion.getReactions() != null)
                .flatMap(discussion -> discussion.getReactions().stream())
                .map(DiscussionReactionDTO::getUserId)
                .filter(userId -> userId != null && !userId.isBlank())
                .distinct()
                .collect(Collectors.toList());
        return discussionReactionUserIds;
    }

    @NotNull
    private static List<String> getListPreferredNameFromDiscussionReaction(DiscussionDTO discussionDTO) {
        List<String> discussionReactionUserIds = Optional.ofNullable(discussionDTO.getReactions())
                .orElse(Collections.emptyList())

                .stream()
                .map(DiscussionReactionDTO::getUserId)
                .filter(userId -> userId != null && !userId.isBlank())
                .distinct()
                .collect(Collectors.toList());
        return discussionReactionUserIds;
    }

    @NotNull
    private static List<String> getListOfPreferredNameFromDiscussion(List<DiscussionDTO> discussionDTOs) {
        List<String> createdByIds = discussionDTOs
                .stream()
                .map(DiscussionDTO::getCreatedBy)
                .filter(createdBy -> createdBy != null && !createdBy.isBlank())
                .distinct()
                .collect(Collectors.toList());
        return createdByIds;
    }

    private List<String> collectReactionUserIdsRecursively(ThreadDTO thread) {
        java.util.ArrayList<String> list = new java.util.ArrayList<>();
        if (thread == null) return list;

        // Collect username from thread reactions
        if (thread.getReactions() != null) {
            thread.getReactions().stream()
                    .map(ThreadReactionDTO::getUserId)
                    .filter(userId -> userId != null && !userId.isBlank())
                    .forEach(list::add);
        }

        // Recursively collect from replies
        if (thread.getReplies() != null) {
            for (ThreadDTO reply : thread.getReplies()) {
                list.addAll(collectReactionUserIdsRecursively(reply));
            }
        }

        return list;
    }

    private void applyReactionUserInfoRecursively(ThreadDTO thread, java.util.Map<String, UserInfoDTO> usernameToUserInfo) {
        if (thread == null || usernameToUserInfo == null) return;

        // Enrich thread reactions
        if (thread.getReactions() != null) {
            thread.getReactions().forEach(reaction -> {
                if (reaction.getUserId() != null) {
                    UserInfoDTO userInfo = usernameToUserInfo.get(reaction.getUserId());
                    if (userInfo != null) {
                        reaction.setUserInfoDTO(userInfo);
                    }
                }
            });
        }

        // Recursively apply to replies
        if (thread.getReplies() != null) {
            for (ThreadDTO reply : thread.getReplies()) {
                applyReactionUserInfoRecursively(reply, usernameToUserInfo);
            }
        }
    }
}
