package com.finconsgroup.itserr.marketplace.audit.dm.service.impl;

import com.finconsgroup.itserr.marketplace.audit.dm.dto.InputCreateAuditLogDto;
import com.finconsgroup.itserr.marketplace.audit.dm.dto.OutputAuditLogDto;
import com.finconsgroup.itserr.marketplace.audit.dm.entity.AuditLogEntity;
import com.finconsgroup.itserr.marketplace.audit.dm.mapper.AuditMapper;
import com.finconsgroup.itserr.marketplace.audit.dm.repository.AuditRepository;
import com.finconsgroup.itserr.marketplace.audit.dm.service.AuditLogService;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2ResourceNotFoundException;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.util.UUID;

/**
 * Default implementation of {@link AuditLogService} to perform operations related to audit log resources
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class DefaultAuditLogService implements AuditLogService {

    private final AuditRepository auditRepository;
    private final AuditMapper auditMapper;

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    public Page<OutputAuditLogDto> findAll(@NonNull Pageable pageable) {
        Page<AuditLogEntity> auditEntityPage = auditRepository.findAll(pageable);
        return auditEntityPage.map(auditMapper::toDto);
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    public OutputAuditLogDto findById(@NonNull UUID auditId) {
        AuditLogEntity auditLogEntity = findByIdOrThrow(auditId);
        return auditMapper.toDto(auditLogEntity);
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public OutputAuditLogDto create(@NonNull InputCreateAuditLogDto inputCreateAuditLogDto) {
        AuditLogEntity auditLogEntity = auditMapper.toEntity(inputCreateAuditLogDto);
        return auditMapper.toDto(auditRepository.saveAndFlush(auditLogEntity));
    }

    private AuditLogEntity findByIdOrThrow(@NonNull UUID auditId) {
        return auditRepository
                .findById(auditId)
                .orElseThrow(() -> new WP2ResourceNotFoundException(auditId));
    }
}
