/**
 * Ext GWT 3.0.0-rc2 - Ext for GWT
 * Copyright(c) 2007-2012, Sencha, Inc.
 * licensing@sencha.com
 *
 * http://sencha.com/license
 */
package com.sencha.gxt.chart.client.draw;

import java.util.ArrayList;

import com.sencha.gxt.core.client.util.PrecisePoint;
import com.sencha.gxt.core.client.util.PreciseRectangle;

/**
 * Represents a polygon. Provides helper functions for intersection detection.
 * 
 */
public class Polygon {

  private ArrayList<PrecisePoint> vertices = new ArrayList<PrecisePoint>();

  /**
   * Creates an empty polygon.
   */
  public Polygon() {
  }

  /**
   * Creates a clone of the given polygon.
   * 
   * @param polygon the polygon to clone
   */
  public Polygon(Polygon polygon) {
    for (int i = 0; i < polygon.size(); i++) {
      vertices.add(new PrecisePoint(polygon.getVertex(i)));
    }
  }

  /**
   * Creates a polygon using the given points.
   * 
   * @param points the points to add to the polygon
   */
  public Polygon(PrecisePoint... points) {
    for (int i = 0; i < points.length; i++) {
      vertices.add(points[i]);
    }
  }

  /**
   * Creates a polygon using the given rectangle
   * 
   * @param rectangle the rectangle used to create the polygon
   */
  public Polygon(PreciseRectangle rectangle) {
    PrecisePoint p1 = new PrecisePoint(rectangle.getX(), rectangle.getY());
    PrecisePoint p2 = new PrecisePoint(rectangle.getX() + rectangle.getWidth(), rectangle.getY());
    PrecisePoint p3 = new PrecisePoint(rectangle.getX() + rectangle.getWidth(), rectangle.getY()
        + rectangle.getHeight());
    PrecisePoint p4 = new PrecisePoint(rectangle.getX(), rectangle.getY() + rectangle.getHeight());

    vertices.add(p1);
    vertices.add(p2);
    vertices.add(p3);
    vertices.add(p4);
  }

  /**
   * Adds a vertex to the polygon using the given coordinates.
   * 
   * @param x the x coordinate of the vertex
   * @param y the y coordinate of the vertex
   */
  public void addVertex(double x, double y) {
    vertices.add(new PrecisePoint(x, y));
  }

  /**
   * Adds the vertex to the polygon.
   * 
   * @param vertex the vertex to be added
   */
  public void addVertex(PrecisePoint vertex) {
    vertices.add(vertex);
  }

  /**
   * Returns true if the vertices of the given polygon are exactly equal to this
   * polygon.
   * 
   * @param polygon the polygon to be compared
   * @return true if the vertices of the given polygon are exactly equal to this
   *         polygon
   */
  public boolean equals(Polygon polygon) {
    int count = 0;
    if (this.size() != polygon.size()) {
      return false;
    } else {
      for (int i = 0; i < this.size(); i++) {
        if (this.getVertex(i).equals(polygon.getVertex(i))) {
          count++;
        }
      }
    }
    return count == this.size();
  }

  /**
   * Returns true if the vertices of the given polygon are nearly equal to this
   * polygon.
   * 
   * @param polygon the polygon to be compared
   * @return true if the vertices of the given polygon are nearly equal to this
   *         polygon
   */
  public boolean equalsNoPrecision(Polygon polygon) {
    int count = 0;
    if (this.size() != polygon.size()) {
      return false;
    } else {
      for (int i = 0; i < this.size(); i++) {
        if (this.getVertex(i).equalsNoPrecision(polygon.getVertex(i), 0)) {
          count++;
        }
      }
    }
    return count == this.size();
  }

  /**
   * Returns the vertex at the provided index.
   * 
   * @param index the index of the vertex
   * @return the vertex at the provided vertex
   */
  public PrecisePoint getVertex(int index) {
    return vertices.get(index);
  }

  /**
   * Intersection of polygons calculated using the Sutherland-Hodgman algorithm.
   * The subject polygon is the polygon making the call and the passed polygon
   * is the clip polygon and thus must be convex.
   * 
   * @param clipPolygon must be convex
   * @return the clipped polygon
   */
  public Polygon intersect(Polygon clipPolygon) {
    if (clipPolygon == null) {
      return null;
    }
    if (this.size() == 0 || clipPolygon.size() == 0) {
      return new Polygon();
    }

    PrecisePoint cp1 = clipPolygon.getVertex(clipPolygon.size() - 1);
    PrecisePoint cp2 = null;
    PrecisePoint s = null;
    PrecisePoint e = null;
    Polygon input = new Polygon();
    Polygon output = new Polygon(this);
    for (int i = 0; i < clipPolygon.size(); i++) {
      cp2 = clipPolygon.getVertex(i);
      input = output;
      output = new Polygon();
      if (input.size() == 0) {
        continue;
      }
      s = input.getVertex(input.size() - 1);
      for (int j = 0; j < input.size(); j++) {
        e = input.getVertex(j);
        if (inside(e, cp1, cp2)) {
          if (!inside(s, cp1, cp2)) {
            output.addVertex(intersection(cp1, cp2, s, e));
          }
          output.addVertex(e);

        } else if (inside(s, cp1, cp2)) {
          output.addVertex(intersection(cp1, cp2, s, e));
        }
        s = e;
      }
      cp1 = cp2;
    }
    return output;
  }

  /**
   * Removes the vertex at the given index.
   * 
   * @param index the index to be removed
   */
  public void removeVertex(int index) {
    vertices.remove(index);
  }

  /**
   * Returns the number of vertices in the polygon
   * 
   * @return the number of vertices in the polygon
   */
  public int size() {
    return vertices.size();
  }

  @Override
  public String toString() {
    return vertices.toString();
  }

  /**
   * Determines if the given point is inside the top left and bottom right
   * points.
   * 
   * @param point the point to determine whether or not it is between the other
   *          two
   * @param topLeft the top left point
   * @param bottomRight the bottom right point
   * @return true if the point is between the top left and bottom right points
   */
  private boolean inside(PrecisePoint point, PrecisePoint topLeft, PrecisePoint bottomRight) {
    return ((bottomRight.getX() - topLeft.getX()) * (point.getY() - topLeft.getY())) > ((bottomRight.getY() - topLeft.getY()) * (point.getX() - topLeft.getX()));
  }

  /**
   * Determines the point of intersection between two lines.
   * 
   * @param line1Start the start of the first line
   * @param line1End the end of the first line
   * @param line2Start the start of the second line
   * @param line2End the end of the second line
   * @return the point of intersection
   */
  private PrecisePoint intersection(PrecisePoint line1Start, PrecisePoint line1End, PrecisePoint line2Start,
      PrecisePoint line2End) {
    double dcx = line1Start.getX() - line1End.getX();
    double dcy = line1Start.getY() - line1End.getY();
    double dpx = line2Start.getX() - line2End.getX();
    double dpy = line2Start.getY() - line2End.getY();
    double n1 = line1Start.getX() * line1End.getY() - line1Start.getY() * line1End.getX();
    double n2 = line2Start.getX() * line2End.getY() - line2Start.getY() * line2End.getX();
    double n3 = 1.0 / (dcx * dpy - dcy * dpx);
    return new PrecisePoint((n1 * dpx - n2 * dcx) * n3, (n1 * dpy - n2 * dcy) * n3);
  }

}
