package org.gcube.resourcemanagement;

import org.gcube.common.security.providers.SecretManagerProvider;
import org.gcube.informationsystem.resourceregistry.api.exceptions.ResourceRegistryException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.types.SchemaException;
import org.gcube.informationsystem.resourceregistry.schema.ResourceRegistrySchemaClient;
import org.gcube.informationsystem.resourceregistry.schema.ResourceRegistrySchemaClientFactory;
import org.gcube.informationsystem.serialization.ElementMapper;
import org.gcube.informationsystem.tree.Node;
import org.gcube.informationsystem.types.reference.Type;
import org.gcube.resourcemanagement.internal.model.impl.entities.facets.DiscoveryFacetImpl;
import org.gcube.resourcemanagement.internal.model.reference.entities.facets.DiscoveryFacet;
import org.gcube.smartgears.ApplicationManager;
import org.gcube.smartgears.ContextProvider;
import org.gcube.smartgears.configuration.Mode;
import org.gcube.smartgears.context.application.ApplicationContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * 
 * This class is used to Manage the application initialization and shutdown per
 * context; The init and shutdown methods are called one per context in which
 * the app is running respectively at init and a shutdown time. It is connected
 * to the app declaring it via the @ManagedBy annotation (@see RMInitializer).
 * 
 * @author Luca Frosini (ISTI - CNR)
 */
public class ResourceManager implements ApplicationManager {

	/**
	 * Logger
	 */
	private static Logger logger = LoggerFactory.getLogger(ResourceManager.class);
	
	/** Flag indicating whether the ResourceManager has been initialized */
	public static boolean initialised;

	static {
		initialised = false;
		logger.info("Resource Manager is being initialized");
		
		// Check DiscoveryFacet only once during class loading
		checkDiscoveryFacetExists();
	}
	
	/**
	 * Checks if the DiscoveryFacet type exists in the Resource Registry using ResourceRegistrySchemaClient
	 * This method runs once during class initialization
	 */
	private static void checkDiscoveryFacetExists() {
		try {
			ResourceRegistrySchemaClient schemaClient = ResourceRegistrySchemaClientFactory.create();
			
			// Use the exist method to check if DiscoveryFacet type is defined in the Resource Registry
			boolean exists = schemaClient.exist(DiscoveryFacet.NAME);
			
			if (exists) {
				logger.info("DiscoveryFacet type is available in the Resource Registry");
				checkDiscoveryFacetInClasspath(schemaClient);
			} else {
				logger.warn("DiscoveryFacet type is NOT available in the Resource Registry, going to add it");
				createDiscoveryFacetInResourceRegistry(schemaClient);
			}
			
		} catch (ResourceRegistryException e) {
			logger.error("Error checking DiscoveryFacet type in Resource Registry", e);
			throw new RuntimeException("Failed to initialize Resource Manager: DiscoveryFacet check failed", e);
		} catch (Exception e) {
			logger.error("Unexpected error checking DiscoveryFacet type", e);
			throw new RuntimeException("Failed to initialize Resource Manager: Unexpected error during DiscoveryFacet check", e);
		}
	}
	
	/**
	 * Checks if DiscoveryFacet implementation is available in the classpath by examining the Information System Model tree
	 */
	private static void checkDiscoveryFacetInClasspath(ResourceRegistrySchemaClient schemaClient) {
		try {
			// Check if DiscoveryFacet is in the model tree
			boolean foundInModel = false;
			try {
				Node<Type> discoveryFacetNode = schemaClient.getTypeTreeNode(DiscoveryFacet.NAME);
				foundInModel = (discoveryFacetNode != null);
				if (foundInModel) {
					logger.info("DiscoveryFacet implementation found in Information System Model tree");
				}
			} catch (Exception e) {
				logger.debug("DiscoveryFacet not found in Information System Model tree", e);
			}
			
			if (foundInModel) {
				logger.info("DiscoveryFacet implementation is available in the classpath and model");
			} else {
				logger.info("DiscoveryFacet implementation is NOT available in the classpath model, registering internal implementation");
				addDiscoveryFacetToElementMapper();
			}
		} catch (Exception e) {
			logger.error("Error checking DiscoveryFacet in classpath and model tree", e);
			// Fallback: try to register the internal implementation anyway
			try {
				addDiscoveryFacetToElementMapper();
			} catch (Exception fallbackException) {
				logger.error("Failed to register DiscoveryFacet implementation as fallback", fallbackException);
			}
		}
	}

	/**
	 * Registers the internal DiscoveryFacet implementation in ElementMapper
	 */
	private static void addDiscoveryFacetToElementMapper() {
		try {
			logger.debug("Registering DiscoveryFacet implementation in ElementMapper");
			ElementMapper.addDynamicAssociation(DiscoveryFacet.class, DiscoveryFacetImpl.class);
			logger.info("Resource Manager DiscoveryFacet implementation has been successfully registered in ElementMapper");
		} catch (Exception e) {
			logger.error("Failed to register DiscoveryFacet implementation in ElementMapper", e);
			throw new RuntimeException("Failed to register DiscoveryFacet implementation", e);
		}
	}
	
	/**
	 * Creates DiscoveryFacet type in the Resource Registry and registers the implementation
	 * @param schemaClient the ResourceRegistrySchemaClient to use
	 * @throws SchemaException if there's an error creating the schema
	 * @throws ResourceRegistryException if there's an error with the Resource Registry
	 */
	private static void createDiscoveryFacetInResourceRegistry(ResourceRegistrySchemaClient schemaClient) throws SchemaException, ResourceRegistryException {
		try {
			logger.info("Creating DiscoveryFacet type in Resource Registry");
			schemaClient.create(DiscoveryFacet.class);
			logger.info("DiscoveryFacet type has been successfully created in Resource Registry");
			
			addDiscoveryFacetToElementMapper();
			logger.info("DiscoveryFacet setup completed: type created in Resource Registry and implementation registered in ElementMapper");
		} catch (ResourceRegistryException e) {
			logger.error("Failed to create DiscoveryFacet in Resource Registry", e);
			throw e;
		} catch (Exception e) {
			logger.error("Unexpected error creating DiscoveryFacet in Resource Registry", e);
			throw new ResourceRegistryException("Unexpected error creating DiscoveryFacet", e);
		}
	}
	
	/** 
	 * {@inheritDoc}
	 */
	@Override
	public synchronized void onInit() {
		
		if (ContextProvider.get().container().configuration().mode() == Mode.offline) {
			logger.debug("Init called in offline mode");
			return;
		}
		
		String context = SecretManagerProvider.get().getContext();
		
		logger.trace(
				"\n-------------------------------------------------------\n"
				+ "Resource Manager is Starting on context {}\n"
				+ "-------------------------------------------------------",
				context);
		
		ApplicationContext applicationContext = ContextProvider.get();
		String rmEServiceID  = applicationContext.id();
		logger.info("Resource Manager has the following ID {}", rmEServiceID);
		
		logger.trace(
				"\n-------------------------------------------------------\n"
				+ "Resource Manager Started Successfully on context {}\n"
				+ "-------------------------------------------------------",
				context);
		
	}
	
	/** 
	 * {@inheritDoc} 
	 */
	@Override
	public synchronized void onShutdown(){
		if (ContextProvider.get().container().configuration().mode() == Mode.offline) {
			logger.debug("Init called in offline mode");
			return;
		}
		
		String context = SecretManagerProvider.get().getContext();
		
		logger.trace(
				"\n-------------------------------------------------------\n"
				+ "Resource Manager is Stopping on context {}\n"
				+ "-------------------------------------------------------", 
				context);
		

		
		logger.trace(
				"\n-------------------------------------------------------\n"
				+ "Resource Manager Stopped Successfully on context {}\n"
				+ "-------------------------------------------------------", 
				context);
	}
}
