package org.gcube.portlets.widgets.ckandatapublisherwidget.server.utils;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.List;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;
import javax.xml.bind.Unmarshaller;

import org.gcube.com.fasterxml.jackson.annotation.JsonAutoDetect;
import org.gcube.com.fasterxml.jackson.annotation.PropertyAccessor;
import org.gcube.com.fasterxml.jackson.core.type.TypeReference;
import org.gcube.com.fasterxml.jackson.databind.DeserializationFeature;
import org.gcube.com.fasterxml.jackson.databind.ObjectMapper;
import org.gcube.datacatalogue.metadatadiscovery.bean.jaxb.MetadataFormat;
import org.gcube.datacatalogue.metadatadiscovery.bean.jaxb.NamespaceCategories;
import org.gcube.datacatalogue.metadatadiscovery.bean.jaxb.NamespaceCategory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MetadataDiscoveryUtil {

	private static Logger LOG = LoggerFactory.getLogger(MetadataDiscoveryUtil.class);

	/**
	 * Unmarshals XML content to a MetadataFormat object
	 * 
	 * @param xml the XML content to unmarshal
	 * @return the MetadataFormat object
	 * @throws JAXBException if an error occurs during unmarshalling
	 */
	public static MetadataFormat getMetadataFormatFromXML(String xml) throws JAXBException {
		JAXBContext jaxbContext = JAXBContext.newInstance(MetadataFormat.class);
		Unmarshaller unmarshaller = jaxbContext.createUnmarshaller();
		InputStream stream = new ByteArrayInputStream(xml.getBytes());
		return (MetadataFormat) unmarshaller.unmarshal(stream);
	}

	/**
	 * Marshals a MetadataFormat object to XML string.
	 * 
	 * @param metadataFormat the metadata format object to marshal
	 * @return the XML string representation
	 * @throws JAXBException if an error occurs during marshalling
	 */
	public static String getXMLFromMetadataFormat(MetadataFormat metadataFormat) throws JAXBException {
		JAXBContext jaxbContext = JAXBContext.newInstance(MetadataFormat.class);
		Marshaller marshaller = jaxbContext.createMarshaller();
		marshaller.setProperty(Marshaller.JAXB_FORMATTED_OUTPUT, Boolean.TRUE);
		StringWriter stringWriter = new StringWriter();
		marshaller.marshal(metadataFormat, stringWriter);
		return stringWriter.toString();
	}

	/**
	 * Converts a list of JSON strings representing namespaces into a
	 * NamespaceCategories object.
	 * 
	 * Each element in the list can be either:
	 * - A single JSON object representing one NamespaceCategory
	 * - A JSON array containing multiple NamespaceCategory objects
	 * 
	 * @param namespacesJson the list of JSON strings
	 * @return the NamespaceCategories object
	 * @throws Exception if an error occurs during conversion
	 */
	public static NamespaceCategories getNamespaceCategoryFromListJsonObject(List<String> namespacesJson)
			throws Exception {

		List<NamespaceCategory> categories = new ArrayList<NamespaceCategory>();
		if (namespacesJson != null && !namespacesJson.isEmpty()) {
			ObjectMapper mapper = new ObjectMapper();
			// Configure mapper to access private fields directly (needed for JAXB classes without getters/setters)
			mapper.setVisibility(PropertyAccessor.FIELD, JsonAutoDetect.Visibility.ANY);
			// Configure mapper to ignore unknown properties not defined in the class
			mapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
			
			for (String jsonElement : namespacesJson) {
				if (jsonElement == null || jsonElement.trim().isEmpty()) {
					continue;
				}
				
				String trimmedJson = jsonElement.trim();
				
				try {
					if (trimmedJson.startsWith("[")) {
						// It's a JSON array - deserialize as list
						LOG.info("Parsing JSON array for namespace categories: {}", trimmedJson);
						List<NamespaceCategory> listFromArray = mapper.readValue(trimmedJson,
								new TypeReference<List<NamespaceCategory>>() {
								});
						if (listFromArray != null) {
							categories.addAll(listFromArray);
						}
					} else if (trimmedJson.startsWith("{")) {
						// It's a single JSON object - deserialize as single object
						LOG.info("Parsing JSON object for namespace category: {}", trimmedJson);
						NamespaceCategory singleCategory = mapper.readValue(trimmedJson, NamespaceCategory.class);
						if (singleCategory != null) {
							categories.add(singleCategory);
						}
					} else {
						LOG.warn("Unexpected JSON format for namespace: {}", trimmedJson);
					}
				} catch (Exception e) {
					LOG.warn("Failed to parse namespace JSON element: {}. Error: {}", trimmedJson, e.getMessage());
					// Continue with next element instead of failing completely
				}
			}
		}

		return new NamespaceCategories(categories);

	}
}
