package org.gcube.portlets.user.homelibrary.jcr.sharing;

import java.util.LinkedList;
import java.util.List;
import java.util.UUID;

import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.PathNotFoundException;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.query.QueryManager;
import javax.jcr.query.QueryResult;

import org.gcube.portlets.user.homelibrary.home.HomeManager;
import org.gcube.portlets.user.homelibrary.home.User;
import org.gcube.portlets.user.homelibrary.home.exceptions.HomeNotFoundException;
import org.gcube.portlets.user.homelibrary.home.exceptions.InternalErrorException;
import org.gcube.portlets.user.homelibrary.home.exceptions.UserNotFoundException;
import org.gcube.portlets.user.homelibrary.home.workspace.exceptions.ItemNotFoundException;
import org.gcube.portlets.user.homelibrary.home.workspace.sharing.WorkspaceMessage;
import org.gcube.portlets.user.homelibrary.home.workspace.sharing.WorkspaceMessageManager;
import org.gcube.portlets.user.homelibrary.jcr.repository.JCRRepository;
import org.gcube.portlets.user.homelibrary.jcr.repository.external.GCUBEStorage;
import org.gcube.portlets.user.homelibrary.jcr.sharing.JCRWorkspaceMessage.WorkspaceMessageType;
import org.gcube.portlets.user.homelibrary.jcr.workspace.JCRWorkspace;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JCRWorkspaceMessageManager implements WorkspaceMessageManager {

	private JCRWorkspace workspace;
	private static Logger logger = LoggerFactory.getLogger(JCRWorkspaceMessageManager.class);
	protected static final String CONTENT			= "jcr:content";
	protected static final String ATTACHMENTS		= "hl:attachments";

	public JCRWorkspaceMessageManager(JCRWorkspace workspace) {
		super();
		this.workspace = workspace;
	}

	@Override
	public void sendMessageToPortalLogins(String subject, String body,
			List<String> attachmentIds, List<String> addresses)
					throws InternalErrorException {

		Session session = JCRRepository.getSession();
		try {

			String messageId = UUID.randomUUID().toString();

			//node for HiddenFolder
			Node nodeHiddenFolder = workspace.getRepository().getHiddenFolder(session);

			//node for OutBox
			Node nodeOutBox = workspace.getRepository().getOutBoxFolder(session);	

			//message for folder OutBox
			Node nodeSentMessage = nodeOutBox.addNode(messageId);			
			JCRWorkspaceMessage itemInSentFolder = new JCRWorkspaceMessage(workspace, nodeSentMessage, WorkspaceMessageType.SENT, 
					messageId, subject, body, workspace.getOwner(), attachmentIds, addresses, nodeHiddenFolder);

			List<String> copyAttachmentsIds = itemInSentFolder.getCopyAttachmentsIds();

			for (String user: addresses) {

				logger.debug("Send message to user " + user);
				HomeManager homeManager = workspace.getHome().getHomeManager();
				homeManager.getHome(user, workspace.getOwner().getScope());

				Node node = workspace.getRepository().getInBoxFolder(session, user);

				Node itemNode = node.addNode(messageId);		

				//message to send
				JCRWorkspaceMessage itemToSend = new JCRWorkspaceMessage(workspace, itemNode, WorkspaceMessageType.RECEIVED,
						messageId, subject, body, workspace.getOwner(), copyAttachmentsIds, addresses, nodeHiddenFolder);
				session.save();
			}
		} catch (HomeNotFoundException e) {
			logger.error("User home not found",e);
			throw new InternalErrorException(e);
		} catch (RepositoryException e) {
			logger.error("Internal error exception",e);
			throw new InternalErrorException(e);
		} catch (UserNotFoundException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}


	}

	//	private List<String> getHardLinks(Session session, List<String> attachmentIds,
	//			Node nodeHiddenFolder) throws ItemNotFoundException, InternalErrorException{
	//		 List<String> hardLinks = new ArrayList<String>();
	//			for (String id : attachmentIds) {
	//				try {
	//					Node nodeItem = session.getNodeByIdentifier(id);
	//					WorkspaceItem item = workspace.getWorkspaceItem(nodeItem);
	//			
	//				} catch (RepositoryException e) {
	//					logger.error("Internal error exception",e);
	//					throw new InternalErrorException(e);				
	//				} catch (InternalErrorException e) {
	//					logger.error("Internal error exception",e);
	//					throw new InternalErrorException(e);
	//				}
	//				
	////				hardLinks.add(user.getPortalLogin());
	//			}
	//		return hardLinks;
	//	}

	@Override
	public void sendMessageToUsers(String subject, String body,
			List<String> attachmentIds, List<User> addresses)
					throws InternalErrorException {

		List<String> list = new LinkedList<String>();
		for (User user : addresses) {
			list.add(user.getPortalLogin());
		}

		sendMessageToPortalLogins(subject, body, attachmentIds, list);

	}

	@Override
	public WorkspaceMessage getSentMessage(String id) throws InternalErrorException,
	ItemNotFoundException {

		Session session = JCRRepository.getSession();
		try {
			Node messageNode = workspace.getRepository().
					getOutBoxFolder(session).getNode(id);

			WorkspaceMessage message = new JCRWorkspaceMessage(workspace, messageNode, WorkspaceMessageType.SENT);
			return message;
		} catch (PathNotFoundException e) {
			throw new ItemNotFoundException(e.getMessage());
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}

	@Override
	public WorkspaceMessage getReceivedMessage(String id) throws InternalErrorException,
	ItemNotFoundException {

		Session session = JCRRepository.getSession();
		try {
			Node messageNode = workspace.getRepository().
					getOwnInBoxFolder(session).getNode(id);

			WorkspaceMessage message = new JCRWorkspaceMessage(workspace, messageNode, WorkspaceMessageType.RECEIVED);
			return message;
		} catch (PathNotFoundException e) {
			throw new ItemNotFoundException(e.getMessage());
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}

	@Override
	public void deleteReceivedMessage(String id) {

		Session session = null;
		try {
			session = JCRRepository.getSession();
			Node messageNode = workspace.getRepository().getOwnInBoxFolder(session).getNode(id);

			//delete msg in Storage			
//			System.out.println("attachNode.getPath() " + messageNode.getPath());
			GCUBEStorage.removeRemoteFolder(messageNode.getPath());

			//delete msg from JR	
			messageNode.remove();
			session.save();


		} catch (Exception e) {
			logger.error("Internal error exception",e);
		} finally {
			if(session	!= null)
				session.logout();
		}
	}

	@Override
	public void deleteSentMessage(String id) {

		Session session = null;
		try {

			session = JCRRepository.getSession();
			Node messageNode = workspace.getRepository().getOutBoxFolder(session).getNode(id);

			//delete msg in Storage			
//			System.out.println("attachNode.getPath() " + messageNode.getPath());
			GCUBEStorage.removeRemoteFolder(messageNode.getPath());

			//delete msg from JR	
			messageNode.remove();
			session.save();

		} catch (Exception e) {
			logger.error("Internal error exception",e);
		} finally {
			if(session	!= null)
				session.logout();
		}
	}

	@Override
	public List<WorkspaceMessage> getReceivedMessages() {

		List<WorkspaceMessage> list = new LinkedList<WorkspaceMessage>();
		Session session = null;
		try {
			session = JCRRepository.getSession();
			Node nodeInBox = workspace.getRepository().getOwnInBoxFolder(session);

			for(NodeIterator iterator = nodeInBox.getNodes(); iterator.hasNext();) {
				Node messageNode = iterator.nextNode();
				list.add(new JCRWorkspaceMessage(workspace, messageNode, WorkspaceMessageType.RECEIVED));
			}

		} catch (Exception e) {
			logger.error("Internal error exception",e);
		} finally {
			if(session != null)
				session.logout();
		}
		return list;
	}	

	@Override
	public List<WorkspaceMessage> getSentMessages() {

		List<WorkspaceMessage> list = new LinkedList<WorkspaceMessage>();
		Session session = null;
		try {
			session = JCRRepository.getSession();
			Node nodeOutBox = workspace.getRepository().getOutBoxFolder(session);

			for(NodeIterator iterator = nodeOutBox.getNodes(); iterator.hasNext();) {
				Node messageNode = iterator.nextNode();
				list.add(new JCRWorkspaceMessage(workspace, messageNode, WorkspaceMessageType.SENT));
			}

		} catch (Exception e) {
			logger.error("Internal error exception",e);
		} finally {
			if(session != null)
				session.logout();
		}
		return list;
	}

	@Override
	public int getMessagesNotOpened() {

		int count = 0;
		for(WorkspaceMessage message : getReceivedMessages()) {
			if(!((JCRWorkspaceMessage)message).isOpened())
				count ++;
		}
		return count;
	}

	@Override
	public List<WorkspaceMessage> searchInMessages(String word) throws InternalErrorException {

		Session session = JCRRepository.getSession();
		try {
			Node nodeInBox = workspace.getRepository().getOwnInBoxFolder(session);
			NodeIterator iterator = getNodeSearched(session, nodeInBox, word); 
			List<WorkspaceMessage> list = getMessagesSearched(iterator, WorkspaceMessageType.RECEIVED);
			return list;
		} catch (RepositoryException e) {
			logger.error("Error ",e);
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}

	}

	@Override
	public List<WorkspaceMessage> searchOutMessages(String word) throws InternalErrorException {

		Session session = JCRRepository.getSession();
		try {
			Node nodeOutBox = workspace.getRepository().getOutBoxFolder(session);
			NodeIterator iterator = getNodeSearched(session, nodeOutBox, word); 
			List<WorkspaceMessage> list = getMessagesSearched(iterator, WorkspaceMessageType.SENT);
			return list;
		} catch (RepositoryException e) {
			logger.error("Error",e);
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}

	}

	private List<WorkspaceMessage> getMessagesSearched(NodeIterator iterator, WorkspaceMessageType type) {

		List<WorkspaceMessage> list = new LinkedList<WorkspaceMessage>();
		while (iterator != null && iterator.hasNext()) {

			Node node = iterator.nextNode();
			try {
				list.add(new JCRWorkspaceMessage(workspace, node, type));
			} catch (RepositoryException e) {
				try {
					logger.error("Item " + node.getName() + "unknow");
				} catch (RepositoryException e1) {
					logger.error("Error ",e1);
				} 
			}

		}
		return list;
	}

	private NodeIterator getNodeSearched(Session session, Node searchRoot, String word)
			throws InternalErrorException {
		NodeIterator iterator;
		try {
			QueryManager queryManager = session.getWorkspace().getQueryManager();			


			javax.jcr.query.Query q = queryManager.createQuery("/jcr:root" 
					+ searchRoot.getPath() + "//element()[jcr:contains(@hl:subject,'*" + word + "*') or " +
					"jcr:contains(@hl:body,'*" + word + "*') ] ",
					javax.jcr.query.Query.XPATH);
			QueryResult result = q.execute();
			iterator = result.getNodes();
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		}
		return iterator;
	}


}
