package org.gcube.portlets.admin.portlet;

import org.gcube.portlets.admin.configuration.RPTTokenConfiguration;
import org.gcube.portlets.admin.constants.RPTTokenPortletKeys;
import org.gcube.portlets.admin.service.OIDCTokenService;

import com.liferay.portal.configuration.metatype.bnd.util.ConfigurableUtil;
import com.liferay.portal.kernel.json.JSONArray;
import com.liferay.portal.kernel.json.JSONFactoryUtil;
import com.liferay.portal.kernel.json.JSONObject;
import com.liferay.portal.kernel.log.Log;
import com.liferay.portal.kernel.log.LogFactoryUtil;
import com.liferay.portal.kernel.util.Constants;
import com.liferay.portal.kernel.util.ParamUtil;
import com.liferay.portal.kernel.portlet.bridges.mvc.MVCPortlet;
import com.liferay.portal.kernel.service.UserLocalService;
import com.liferay.portal.kernel.theme.ThemeDisplay;
import com.liferay.portal.kernel.util.ParamUtil;
import com.liferay.portal.kernel.util.Portal;
import com.liferay.portal.kernel.util.WebKeys;
import com.liferay.portal.kernel.model.User;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import javax.portlet.ActionRequest;
import javax.portlet.ActionResponse;
import javax.portlet.Portlet;
import javax.portlet.PortletException;
import javax.portlet.PortletPreferences;
import javax.portlet.RenderRequest;
import javax.portlet.RenderResponse;
import javax.portlet.ResourceRequest;
import javax.portlet.ResourceResponse;
import javax.servlet.http.HttpServletRequest;

import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Modified;
import org.osgi.service.component.annotations.Reference;

/**
 * RPT Token Portlet for Liferay 7.4
 * Migrated from Liferay 6.2 version
 * 
 * @author netfarm-m2
 */
@Component(
	configurationPid = "org.gcube.portlets.admin.configuration.RPTTokenConfiguration",
	property = {
		"com.liferay.portlet.display-category=category.d4science",
		"com.liferay.portlet.header-portlet-css=/css/main.css",
		"com.liferay.portlet.instanceable=true",
		"javax.portlet.display-name=Personal Token",
		"javax.portlet.init-param.template-path=/",
		"javax.portlet.init-param.view-template=/view.jsp",
		"javax.portlet.init-param.config-template=/config.jsp",
		"javax.portlet.name=" + RPTTokenPortletKeys.RPTTOKEN,
		"javax.portlet.resource-bundle=content.Language",
		"javax.portlet.security-role-ref=power-user,user,administrator",
		"javax.portlet.supports.mime-type=text/html",
		"javax.portlet.portlet-mode=text/html;view,config",
		"javax.portlet.expiration-cache=0"
	},
	service = Portlet.class
)
public class RPTTokenPortlet extends MVCPortlet {

	private static final Log _log = LogFactoryUtil.getLog(RPTTokenPortlet.class);
	
	private volatile RPTTokenConfiguration _configuration;

	@Override
	public void processAction(ActionRequest actionRequest, ActionResponse actionResponse)
			throws IOException, PortletException {
		
		String cmd = ParamUtil.getString(actionRequest, Constants.CMD);
		
		if (cmd.equals(Constants.UPDATE)) {
			try {
				PortletPreferences preferences = actionRequest.getPreferences();
				
				String exchangeClientId = ParamUtil.getString(actionRequest, "exchangeClientId");
				String exchangeClientSecret = ParamUtil.getString(actionRequest, "exchangeClientSecret");
				String defaultTokenUrl = ParamUtil.getString(actionRequest, "defaultTokenUrl");
				boolean debugMode = ParamUtil.getBoolean(actionRequest, "debugMode");
				
				preferences.setValue("exchangeClientId", exchangeClientId);
				preferences.setValue("exchangeClientSecret", exchangeClientSecret);
				preferences.setValue("defaultTokenUrl", defaultTokenUrl);
				preferences.setValue("debugMode", String.valueOf(debugMode));
				
				preferences.store();
				
				if (_log.isInfoEnabled()) {
					_log.info("RPT Token Portlet configuration updated successfully");
				}
				
			} catch (Exception e) {
				_log.error("Error updating portlet configuration", e);
				throw new PortletException("Error updating configuration", e);
			}
		}
		
		super.processAction(actionRequest, actionResponse);
	}

	@Override
	public void render(RenderRequest renderRequest, RenderResponse renderResponse)
			throws PortletException, IOException {
		
		try {
			ThemeDisplay themeDisplay = (ThemeDisplay) renderRequest.getAttribute(WebKeys.THEME_DISPLAY);
			User user = themeDisplay.getUser();
			
			if (_log.isDebugEnabled()) {
				_log.debug("Rendering RPT Token Portlet for user: " + user.getEmailAddress());
			}

			// Get current context - simplified version for 7.4
			String currentContext = getCurrentContext(renderRequest);
			
			// For now, we'll create a simplified context list
			// In a full migration, you would integrate with gCube GroupManager
			List<String> userContexts = new ArrayList<>();
			List<String> vreContexts = new ArrayList<>();
			
			// Add current context if available
			if (currentContext != null && !currentContext.isEmpty()) {
				userContexts.add(currentContext);
				// Simple check if it's a VRE context (4 levels deep)
				if (currentContext.split("/").length == 4) {
					vreContexts.add(currentContext);
				}
			}
			
			// Add some default contexts for demonstration
			userContexts.add("/gcube/devsec/devVRE");
			vreContexts.add("/gcube/devsec/devVRE");
			
			renderRequest.setAttribute("userContexts", userContexts);
			renderRequest.setAttribute("vreContexts", vreContexts);
			renderRequest.setAttribute("configuration", _configuration);
			
		} catch (Exception e) {
			_log.error("Error in render method", e);
		}
		
		super.render(renderRequest, renderResponse);
	}

	@Override
	public void serveResource(ResourceRequest resourceRequest, ResourceResponse resourceResponse)
			throws IOException, PortletException {
		
		String context = ParamUtil.getString(resourceRequest, "context", null);
		
		if (_log.isDebugEnabled()) {
			_log.debug("Token exchange requested for context: " + context);
		}

		resourceResponse.setContentType("application/json");
		JSONObject jsonObject = JSONFactoryUtil.createJSONObject();

		try {
			ThemeDisplay themeDisplay = (ThemeDisplay) resourceRequest.getAttribute(WebKeys.THEME_DISPLAY);
			User user = themeDisplay.getUser();
			
			if (user == null || user.isDefaultUser()) {
				jsonObject.put("success", false);
				jsonObject.put("error", "User not authenticated");
				resourceResponse.getWriter().println(jsonObject);
				return;
			}

			// Get access token from session or user attributes
			HttpServletRequest httpRequest = _portal.getHttpServletRequest(resourceRequest);
			String accessToken = getAccessTokenFromSession(httpRequest);
			
			if (accessToken == null || accessToken.isEmpty()) {
				jsonObject.put("success", false);
				jsonObject.put("error", "No access token available");
				resourceResponse.getWriter().println(jsonObject);
				return;
			}

			// Get configuration from portlet preferences (primary) or OSGi configuration (fallback)
			PortletPreferences preferences = resourceRequest.getPreferences();
			
			String clientId = preferences.getValue("exchangeClientId", "");
			String clientSecret = preferences.getValue("exchangeClientSecret", "");
			String tokenUrl = preferences.getValue("defaultTokenUrl", 
				"https://accounts.d4science.org/auth/realms/d4science/protocol/openid-connect/token");
			
			// Fallback to OSGi configuration if preferences are empty
			if ((clientId.isEmpty() || clientSecret.isEmpty()) && _configuration != null) {
				if (clientId.isEmpty()) {
					clientId = _configuration.exchangeClientId();
				}
				if (clientSecret.isEmpty()) {
					clientSecret = _configuration.exchangeClientSecret();
				}
				if (tokenUrl.equals("https://accounts.d4science.org/auth/realms/d4science/protocol/openid-connect/token")) {
					tokenUrl = _configuration.defaultTokenUrl();
				}
				
				if (_log.isDebugEnabled()) {
					_log.debug("Using OSGi configuration as fallback - ClientId: " + (clientId.isEmpty() ? "EMPTY" : "SET") + 
							  ", TokenUrl: " + tokenUrl);
				}
			} else if (_log.isDebugEnabled()) {
				_log.debug("Using portlet preferences - ClientId: " + (clientId.isEmpty() ? "EMPTY" : "SET") + 
						  ", TokenUrl: " + tokenUrl);
			}
			
			if (clientId.isEmpty() || clientSecret.isEmpty()) {
				jsonObject.put("success", false);
				jsonObject.put("error", "OIDC exchange credentials not configured. Configuration status: " + 
							  (_configuration != null ? "loaded but empty" : "not loaded"));
				jsonObject.put("debug_info", "ClientId empty: " + clientId.isEmpty() + 
							  ", ClientSecret empty: " + clientSecret.isEmpty());
				resourceResponse.getWriter().println(jsonObject);
				return;
			}

			// Perform token exchange
			JSONObject exchangeResult = _oidcTokenService.exchangeToken(
				tokenUrl, accessToken, context, clientId, clientSecret
			);
			
			resourceResponse.getWriter().println(exchangeResult);
			
		} catch (Exception e) {
			_log.error("Error during token exchange", e);
			
			jsonObject.put("success", false);
			jsonObject.put("error", "Token exchange failed");
			jsonObject.put("message", e.getMessage());
			
			resourceResponse.getWriter().println(jsonObject);
		}
		
		super.serveResource(resourceRequest, resourceResponse);
	}

	/**
	 * Get current context from request - simplified version
	 */
	private String getCurrentContext(RenderRequest request) {
		try {
			ThemeDisplay themeDisplay = (ThemeDisplay) request.getAttribute(WebKeys.THEME_DISPLAY);
			long groupId = themeDisplay.getScopeGroupId();
			
			// For demonstration, return a default context
			// In a full migration, this would integrate with gCube PortalContext
			return "/gcube/devsec/devVRE";
			
		} catch (Exception e) {
			_log.error("Error getting current context", e);
			return null;
		}
	}

	/**
	 * Extract access token from session - placeholder implementation
	 */
	private String getAccessTokenFromSession(HttpServletRequest request) {
		// This is a placeholder implementation
		// In a real scenario, you would extract the token from:
		// 1. OIDC session attributes
		// 2. JWT cookies
		// 3. Authorization headers
		// 4. Integration with Liferay's OIDC provider
		
		// For demonstration, return a placeholder
		return "placeholder_access_token";
	}

	@Activate
	@Modified
	protected void activate(Map<String, Object> properties) {
		try {
			_configuration = ConfigurableUtil.createConfigurable(
				RPTTokenConfiguration.class, properties);
			
			if (_log.isInfoEnabled()) {
				_log.info("RPT Token Portlet activated with configuration");
				if (_configuration != null) {
					_log.info("Configuration details - ClientId: " + 
							 (_configuration.exchangeClientId().isEmpty() ? "EMPTY" : "SET") +
							 ", TokenUrl: " + _configuration.defaultTokenUrl() +
							 ", Debug: " + _configuration.debugMode());
				} else {
					_log.warn("Configuration is null after activation");
				}
			}
			
			// Log properties for debugging
			if (_log.isDebugEnabled() && properties != null) {
				_log.debug("Activation properties: " + properties.toString());
			}
			
		} catch (Exception e) {
			_log.error("Error during portlet activation", e);
		}
	}

	@Reference
	private OIDCTokenService _oidcTokenService;

	@Reference
	private UserLocalService _userLocalService;

	@Reference
	private Portal _portal;
}