package org.gcube.informationsystem.resourceregistry.rest;

import java.util.UUID;

import org.eclipse.microprofile.openapi.annotations.Operation;
import org.eclipse.microprofile.openapi.annotations.media.Content;
import org.eclipse.microprofile.openapi.annotations.media.Schema;
import org.eclipse.microprofile.openapi.annotations.responses.APIResponse;
import org.eclipse.microprofile.openapi.annotations.tags.Tag;
import org.gcube.informationsystem.resourceregistry.ResourceInitializer;
import org.gcube.informationsystem.resourceregistry.api.exceptions.AvailableInAnotherContextException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.NotFoundException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.ResourceRegistryException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.contexts.ContextNotFoundException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.entities.resource.ResourceNotFoundException;
import org.gcube.informationsystem.resourceregistry.api.rest.InstancePath;
import org.gcube.informationsystem.resourceregistry.api.rest.SharingPath;
import org.gcube.informationsystem.resourceregistry.base.ElementManagement;
import org.gcube.informationsystem.resourceregistry.instances.model.ERManagementUtility;
import org.gcube.informationsystem.resourceregistry.rest.requests.ServerRequestInfo;

import jakarta.ws.rs.Consumes;
import jakarta.ws.rs.DELETE;
import jakarta.ws.rs.DefaultValue;
import jakarta.ws.rs.GET;
import jakarta.ws.rs.HEAD;
import jakarta.ws.rs.PUT;
import jakarta.ws.rs.Path;
import jakarta.ws.rs.PathParam;
import jakarta.ws.rs.Produces;
import jakarta.ws.rs.QueryParam;
import jakarta.ws.rs.core.MediaType;
import jakarta.ws.rs.core.Response;
import jakarta.ws.rs.core.Response.Status;


/**
 * @author Luca Frosini (ISTI - CNR)
 */
@Path(InstancePath.INSTANCES_PATH_PART)
@Tag(name = "Instances", description = "Operations for managing entities and relation instances in the Resource Registry.")
public class InstancesManager extends BaseRest {
	
	public static final String INSTANCE_UUID_PATH_PARAMETER = "instance-uuid";
	
	public static final String INSTANCE = "Instance";
	public static final String GET_INSTANCE_CONTEXTS_METHOD = "getInstanceContexts";
	
	public InstancesManager() {
		super();
	}
	
	/**
	 * Retrieves all instances of a specified type from the Resource Registry.
	 * 
	 * <strong>REST Endpoint:</strong> {@code GET /instances/{type-name}}
	 * 
	 * <strong>Request Examples:</strong>
	 * <ul>
	 * <li>Basic query: {@code GET /instances/EService}</li>
	 * <li>With pagination: {@code GET /instances/EService?limit=20&offset=40}</li>
	 * <li>With subtypes and metadata: {@code GET /instances/EService?polymorphic=true&includeMeta=true}</li>
	 * <li>Full metadata context: {@code GET /instances/EService?includeMeta=true&amp;allMeta=true}</li>
	 * <li>With context information: {@code GET /instances/EService?includeContexts=true}</li>
	 * <li>Hierarchical (admin only): {@code GET /instances/EService?hierarchical=true}</li>
	 * <li>Hierarchical with contexts (admin only): {@code GET /instances/EService?hierarchical=true&includeContexts=true}.</li>
	 * </ul>
	 * 
	 * <strong>Path Parameters:</strong>
	 * <ul>
	 * <li><strong>type-name</strong>: The name of the Information System type whose instances should be retrieved (e.g., "EService", "ContactFacet").</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>polymorphic</strong> (optional):
	 * <ul>
	 * <li>Whether to include instances of subtypes in the response</li>
	 * <li>Default value: true (includes all subtypes)</li>
	 * <li>Values: true|false</li>
	 * <li>Example: ?polymorphic=false (returns only exact type instances).</li>
	 * </ul>
	 * 
	 * <strong>limit</strong> (optional):
	 * <ul>
	 * <li>Maximum number of instances to return in a single response</li>
	 * <li>Default value: 10</li>
	 * <li>Example: ?limit=50 (returns at most 50 instances).</li>
	 * </ul>
	 * 
	 * <strong>offset</strong> (optional):
	 * <ul>
	 * <li>Number of instances to skip from the beginning of the result set</li>
	 * <li>Default value: 0 (starts from the first instance)</li>
	 * <li>Example: ?offset=100 (skips the first 100 instances).</li>
	 * </ul>
	 * 
	 * <strong>includeMeta</strong> (optional):
	 * <ul>
	 * <li>Whether to include metadata in the response instances</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Values: true|false</li>
	 * <li>Example: ?includeMeta=true (includes metadata with appropriate filtering based on user role)</li>
	 * <li>Metadata includes only the requested type and subtypes (if polymorphic=true)</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (optional):
	 * <ul>
	 * <li>Whether to include metadata for all nested instances (ConsistsOf relations, Facets, etc.)</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main instances, more human-readable)</li>
	 * <li>Values: true|false</li>
	 * <li>Example: ?includeMeta=true&amp;allMeta=true (includes metadata for all nested elements)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process.</li>
	 * </ul>
	 * 
	 * <strong>hierarchical</strong> (optional, admin-only):
	 * <ul>
	 * <li>Whether to include instances from child contexts of the current context</li>
	 * <li>Default value: false (current context only)</li>
	 * <li>Values: true|false</li>
	 * <li><strong>Restriction:</strong> Only available to IS-Manager, Infrastructure-Manager, and Context-Manager roles (Context-Manager must be of the current context derived from the authorization token)</li>
	 * <li>Example: ?hierarchical=true (includes instances from current and child contexts)</li>
	 * <li>Current context is determined from the authorization token.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (optional):
	 * <ul>
	 * <li>Whether to include the list of contexts where each instance is available</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Values: true|false</li>
	 * <li>Example: ?includeContexts=true (shows context availability for each instance)</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * 
	 * <strong>IS-Manager:</strong>
	 * <ul>
	 * <li>Full access to all metadata and hierarchical querying capabilities</li>
	 * <li>Can access instances across all contexts</li>
	 * <li>Receives complete, unfiltered metadata including all administrative fields.</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Full access to all metadata and hierarchical querying capabilities</li>
	 * <li>Can access instances across all contexts</li>
	 * <li>Receives complete, unfiltered metadata including all administrative fields.</li>
	 * </ul>
	 * 
	 * <strong>Context-Manager:</strong>
	 * <ul>
	 * <li>Full access to all metadata and hierarchical querying capabilities</li>
	 * <li>Must be Context-Manager of the current context derived from the authorization token</li>
	 * <li>Can access hierarchical querying capabilities</li>
	 * <li>Receives complete, unfiltered metadata including all administrative fields.</li>
	 * </ul>
	 * 
	 * <strong>Other Users:</strong>
	 * <ul>
	 * <li>Basic access with metadata filtering and no hierarchical querying</li>
	 * <li>Cannot use hierarchical querying (hierarchical parameter ignored)</li>
	 * <li>Receive metadata with sensitive information filtered when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Instances successfully retrieved</li>
	 * <li><strong>400 Bad Request</strong>: Invalid query parameters (e.g., invalid limit/offset values)</li>
	 * <li><strong>404 Not Found</strong>: The specified type does not exist</li>
	 * <li><strong>403 Forbidden</strong>: Insufficient permissions to access instances in the current context.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Returns a JSON array containing the requested instances</li>
	 * <li>Each instance includes its properties, relations, and metadata (if requested)</li>
	 * <li>Empty array is returned if no instances of the specified type exist in the accessible contexts.</li>
	 * </ul>
	 * 
	 * <strong>Example Metadata (filtered for non-admin users):</strong>
	 * ``{@code json
	 * "metadata": {
	 *     "type": "Metadata",
	 *     "creationTime": "2025-03-18 17:13:40.952 +0100",
	 *     "createdBy": "HIDDEN_FOR_PRIVACY",
	 *     "lastUpdateBy": "HIDDEN_FOR_PRIVACY",
	 *     "lastUpdateTime": "2025-03-19 16:21:16.805 +0100"
	 * }
	 * }``
	 * 
	 * @param type The name of the Information System type whose instances to retrieve
	 * @param polymorphic Whether to include instances of subtypes (default: true)
	 * @return JSON array containing the requested instances with their properties and relations
	 * @throws NotFoundException If the specified type does not exist
	 * @throws ResourceRegistryException If an error occurs during instance retrieval
	 */
	@GET
	@Path("/{" + TypeManager.TYPE_PATH_PARAMETER + "}")
	@Consumes({MediaType.TEXT_PLAIN, ResourceInitializer.APPLICATION_JSON_CHARSET_UTF_8})
	@Produces(ResourceInitializer.APPLICATION_JSON_CHARSET_UTF_8)
	@Operation(
		summary = "List Instances by specified Type ",
		description = """
		Retrieves all instances of a specified type from the Resource Registry.


		**Request Examples:**
		- Basic query: GET /instances/EService;
		- With pagination: GET /instances/EService?limit=20&offset=40;
		- With subtypes and metadata: GET /instances/EService?polymorphic=true&includeMeta=true;
		- Full metadata context: GET /instances/EService?includeMeta=true&allMeta=true;
		- With context information: GET /instances/EService?includeContexts=true;
		- Hierarchical (admin only): GET /instances/EService?hierarchical=true;
		- Hierarchical with contexts (admin only): GET /instances/EService?hierarchical=true&includeContexts=true.


		**Authorization Requirements:**
		- **IS-Manager:**
			- Full access to all metadata and hierarchical querying capabilities;
			- Can access instances across all contexts;
			- Receives complete, unfiltered metadata including all administrative fields.

		- **Infrastructure-Manager:**
			- Full access to all metadata and hierarchical querying capabilities;
			- Can access instances across all contexts;
			- Receives complete, unfiltered metadata including all administrative fields.

		- **Context-Manager:**
			- Full access to all metadata and hierarchical querying capabilities;
			- Must be Context-Manager of the current context derived from the authorization token;
			- Can access hierarchical querying capabilities;
			- Receives complete, unfiltered metadata including all administrative fields.

		- **Other Users:**
			- Basic access with metadata filtering and no hierarchical querying;
			- Cannot use hierarchical querying (hierarchical parameter ignored);
			- Receive metadata with sensitive information filtered when includeMeta=true.


		**Response Format:**
		- Returns a JSON array containing the requested instances;
		- Each instance includes its properties, relations, and metadata (if requested);
		- Empty array is returned if no instances of the specified type exist in the accessible contexts.

		**Example Metadata (filtered for non-admin users):**
		```json
		"metadata": {
		    "type": "Metadata",
		    "creationTime": "2025-03-18 17:13:40.952 +0100",
		    "createdBy": "HIDDEN_FOR_PRIVACY",
		    "lastUpdateBy": "HIDDEN_FOR_PRIVACY",
		    "lastUpdateTime": "2025-03-19 16:21:16.805 +0100"
		}
		```
		"""
	)
	@APIResponse(
		responseCode = "200",
		description = "Instances successfully retrieved",
		content = @Content(mediaType = "application/json")
	)
	@APIResponse(
		responseCode = "400",
		description = "Invalid query parameters (e.g., invalid limit/offset values)"
	)
	@APIResponse(
		responseCode = "404",
		description = "The specified type does not exist"
	)
	@APIResponse(
		responseCode = "403",
		description = "Insufficient permissions to access instances in the current context"
	)
	public String readAll(
			@PathParam(TypeManager.TYPE_PATH_PARAMETER)
			@Schema(description = "The name of the Information System type whose instances should be retrieved (e.g., \"EService\", \"ContactFacet\")", example = "EService")
			String type,
			@QueryParam(InstancePath.POLYMORPHIC_QUERY_PARAMETER) @DefaultValue("true") Boolean polymorphic)
			throws NotFoundException, ResourceRegistryException {
		logger.info("Requested all {}instances of {}", polymorphic ? InstancePath.POLYMORPHIC_QUERY_PARAMETER + " " : "", type);
		setAccountingMethod(Method.LIST, InstancesManager.INSTANCE);
		
		ServerRequestInfo serverRequestInfo = initRequestInfo();
		serverRequestInfo.checkAllBooleanQueryParameters();
		serverRequestInfo.checkLimitOffset();
		
		ElementManagement<?,?> erManagement = ERManagementUtility.getERManagement(type);
		return erManagement.all(polymorphic);
	}
	
	/**
	 * Checks if an instance with the specified type and UUID exists in the Resource Registry.
	 * 
	 * <strong>REST Endpoint:</strong> {@code HEAD /instances/{type-name}/{instance-uuid}}
	 * 
	 * <strong>Request Examples:</strong>
	 * <ul>
	 * <li>Basic check: {@code HEAD /instances/EService/48af15ad-7e56-4157-b624-71c98cea4f8f}</li>
	 * <li>Hierarchical check (admin only): {@code HEAD /instances/EService/48af15ad-7e56-4157-b624-71c98cea4f8f?hierarchical=true}.</li>
	 * </ul>
	 * 
	 * <strong>Path Parameters:</strong>
	 * <ul>
	 * <li><strong>type-name</strong>: The name of the Information System type (e.g., "EService", "ContactFacet")</li>
	 * <li><strong>instance-uuid</strong>: The unique identifier of the instance to check.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>hierarchical</strong> (optional, admin-only):
	 * <ul>
	 * <li>Whether to check for instance existence in child contexts of the current context</li>
	 * <li>Default value: false (current context only)</li>
	 * <li>Values: true|false</li>
	 * <li><strong>Restriction:</strong> Only available to IS-Manager, Infrastructure-Manager, and Context-Manager roles (Context-Manager must be of the current context derived from the authorization token)</li>
	 * <li>Example: ?hierarchical=true (checks current context and all child contexts)</li>
	 * <li>Current context is determined from the authorization token.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <strong>IS-Manager:</strong>
	 * <ul>
	 * <li>Can check existence of any instance type in any context</li>
	 * <li>Can use hierarchical querying to check across context hierarchies</li>
	 * <li>Has full administrative access to existence checking.</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Can check existence of any instance type in any context</li>
	 * <li>Can use hierarchical querying to check across context hierarchies</li>
	 * <li>Has full administrative access to existence checking.</li>
	 * </ul>
	 * 
	 * <strong>Context-Manager:</strong>
	 * <ul>
	 * <li>Can check existence of instances in the current context</li>
	 * <li>Can use hierarchical querying within their managed context hierarchy</li>
	 * <li>Must be Context-Manager of the current context (derived from the authorization token)</li>
	 * <li>Cannot check instances in contexts where they don't have Context-Manager role.</li>
	 * </ul>
	 * 
	 * <strong>Other Users:</strong>
	 * <ul>
	 * <li>Can check existence of instances in the current context only</li>
	 * <li>Cannot use hierarchical querying (parameter is ignored)</li>
	 * <li>Only within contexts where they have authorized access through their token.</li>
	 * </ul>
	 * 
	 * <strong>Response Codes:</strong>
	 * <ul>
	 * <li><strong>204 No Content</strong>: Instance exists and is accessible in the current context</li>
	 * <li><strong>404 Not Found</strong>: Instance does not exist</li>
	 * <li><strong>403 Forbidden</strong>: Instance exists but is available in another context (not accessible in current context).</li>
	 * </ul>
	 * 
	 * @param type The name of the Information System type
	 * @param instanceId The unique identifier of the instance to check
	 * @return HTTP response with appropriate status code (204, 404, or 403)
	 * @throws NotFoundException If the specified type or instance does not exist
	 * @throws ResourceRegistryException If an error occurs during the existence check
	 */
	@HEAD
	@Path("/{" + TypeManager.TYPE_PATH_PARAMETER + "}" + "/{" + InstancesManager.INSTANCE_UUID_PATH_PARAMETER + "}")
	@Consumes({MediaType.TEXT_PLAIN, ResourceInitializer.APPLICATION_JSON_CHARSET_UTF_8})
	@Produces(ResourceInitializer.APPLICATION_JSON_CHARSET_UTF_8)
	@Operation(
		summary = "Check Instance existence by Type and UUID ",
		description = """
		Checks if an instance with the specified type and UUID exists in the Resource Registry.


		**Request Examples:**
		- Basic check: HEAD /instances/EService/48af15ad-7e56-4157-b624-71c98cea4f8f;
		- Hierarchical check (admin only): HEAD /instances/EService/48af15ad-7e56-4157-b624-71c98cea4f8f?hierarchical=true.


		**Authorization Requirements:**
		- **IS-Manager:**
			- Can check existence of any instance type in any context;
			- Can use hierarchical querying to check across context hierarchies;
			- Has full administrative access to existence checking.

		- **Infrastructure-Manager:**
			- Can check existence of any instance type in any context;
			- Can use hierarchical querying to check across context hierarchies;
			- Has full administrative access to existence checking.

		- **Context-Manager:**
			- Can check existence of instances in the current context;
			- Can use hierarchical querying within their managed context hierarchy;
			- Must be Context-Manager of the current context (derived from the authorization token);
			- Cannot check instances in contexts where they don't have Context-Manager role.

		- **Other Users:**
			- Can check existence of instances in the current context only;
			- Cannot use hierarchical querying (parameter is ignored);
			- Only within contexts where they have authorized access through their token.
		"""
	)
	@APIResponse(
		responseCode = "204",
		description = "Instance exists and is accessible in the current context"
	)
	@APIResponse(
		responseCode = "404",
		description = "Instance does not exist"
	)
	@APIResponse(
		responseCode = "403",
		description = "Instance exists but is available in another context (not accessible in current context)"
	)
	public Response exists(
			@PathParam(TypeManager.TYPE_PATH_PARAMETER)
			@Schema(description = "The name of the Information System type (e.g., \"EService\", \"ContactFacet\")", example = "EService")
			String type,
			@PathParam(InstancesManager.INSTANCE_UUID_PATH_PARAMETER) String instanceId) throws NotFoundException, ResourceRegistryException {
		logger.info("Requested to check if {} with id {} exists", type, instanceId);
		setAccountingMethod(Method.EXIST, InstancesManager.INSTANCE);
		
		ServerRequestInfo serverRequestInfo = initRequestInfo();
		serverRequestInfo.checkBooleanQueryParameter(InstancePath.HIERARCHICAL_MODE_QUERY_PARAMETER);
		
		@SuppressWarnings("rawtypes")
		ElementManagement erManagement = ERManagementUtility.getERManagement(type);
		try {
			erManagement.setUUID(UUID.fromString(instanceId));
			boolean found = erManagement.exists();
			if(found) {
				return Response.status(Status.NO_CONTENT).build();
			} else {
				// This code should never be reached due to exception management
				// anyway adding it for safety reason
				return Response.status(Status.NOT_FOUND).build();
			}
		} catch(NotFoundException e) {
			return Response.status(Status.NOT_FOUND).build();
		} catch(AvailableInAnotherContextException e) {
			return Response.status(Status.FORBIDDEN).build();
		} catch(ResourceRegistryException e) {
			throw e;
		}
	}
	
	/**
	 * Retrieves a single instance with the specified type and UUID.
	 * 
	 * <strong>REST Endpoint:</strong> {@code GET /instances/{type-name}/{instance-uuid}}
	 * 
	 * <strong>Request Examples:</strong>
	 * <ul>
	 * <li>Basic query: {@code GET /instances/EService/48af15ad-7e56-4157-b624-71c98cea4f8f}</li>
	 * <li>With metadata: {@code GET /instances/EService/48af15ad-7e56-4157-b624-71c98cea4f8f?includeMeta=true}</li>
	 * <li>Full metadata context: {@code GET /instances/EService/48af15ad-7e56-4157-b624-71c98cea4f8f?includeMeta=true&amp;allMeta=true}</li>
	 * <li>With context information: {@code GET /instances/EService/48af15ad-7e56-4157-b624-71c98cea4f8f?includeContexts=true}</li>
	 * <li>Hierarchical (admin only): {@code GET /instances/EService/48af15ad-7e56-4157-b624-71c98cea4f8f?hierarchical=true}</li>
	 * <li>Hierarchical with contexts (admin only): {@code GET /instances/EService/48af15ad-7e56-4157-b624-71c98cea4f8f?hierarchical=true&includeContexts=true}.</li>
	 * </ul>
	 * 
	 * <strong>Path Parameters:</strong>
	 * <ul>
	 * <li><strong>type-name</strong>: The name of the Information System type (e.g., "EService", "ContactFacet")</li>
	 * <li><strong>instance-uuid</strong>: The unique identifier of the instance to retrieve (e.g., "48af15ad-7e56-4157-b624-71c98cea4f8f").</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>includeMeta</strong> (optional):
	 * <ul>
	 * <li>Whether to include metadata in the response instance</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Values: true|false</li>
	 * <li>Example: ?includeMeta=true (includes metadata with appropriate filtering based on user role)</li>
	 * <li>Metadata includes the requested instance and nested elements based on polymorphic setting</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (optional):
	 * <ul>
	 * <li>Whether to include metadata for all nested instances (ConsistsOf relations, Facets, etc.)</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main instance, more human-readable)</li>
	 * <li>Values: true|false</li>
	 * <li>Example: ?includeMeta=true&amp;allMeta=true (includes metadata for all nested elements)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process.</li>
	 * </ul>
	 * 
	 * <strong>hierarchical</strong> (optional, admin-only):
	 * <ul>
	 * <li>Whether to retrieve the instance from child contexts of the current context</li>
	 * <li>Default value: false (current context only)</li>
	 * <li>Values: true|false</li>
	 * <li><strong>Restriction:</strong> Only available to IS-Manager, Infrastructure-Manager, and Context-Manager roles (Context-Manager must be of the current context derived from the authorization token)</li>
	 * <li>Example: ?hierarchical=true (searches current and child contexts for the instance)</li>
	 * <li>Current context is determined from the authorization token.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (optional):
	 * <ul>
	 * <li>Whether to include the list of contexts where the instance and its nested elements are available</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Values: true|false</li>
	 * <li>Example: ?includeContexts=true (shows context availability for the instance and nested elements)</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <strong>IS-Manager:</strong>
	 * <ul>
	 * <li>Full access to retrieve any instance in any context</li>
	 * <li>Can use hierarchical querying to search across context hierarchies</li>
	 * <li>Receive complete, unfiltered metadata for the instance and all nested elements</li>
	 * <li>Metadata includes all administrative fields without obfuscation.</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Full access to retrieve any instance in any context</li>
	 * <li>Can use hierarchical querying to search across context hierarchies</li>
	 * <li>Receive complete, unfiltered metadata for the instance and all nested elements</li>
	 * <li>Metadata includes all administrative fields without obfuscation.</li>
	 * </ul>
	 * 
	 * <strong>Context-Manager:</strong>
	 * <ul>
	 * <li>Can retrieve instances in the current context and child contexts</li>
	 * <li>Can use hierarchical querying within their managed context hierarchy</li>
	 * <li>Must be Context-Manager of the current context (derived from the authorization token)</li>
	 * <li>Receive complete, unfiltered metadata for the instance and all nested elements</li>
	 * <li>Metadata includes all administrative fields without obfuscation.</li>
	 * </ul>
	 * 
	 * <strong>Other Users:</strong>
	 * <ul>
	 * <li>Can retrieve instances in the current context only</li>
	 * <li>Cannot use hierarchical querying (hierarchical parameter ignored)</li>
	 * <li>Receive metadata with sensitive information filtered when includeMeta=true:</li>
	 * </ul>
	 *   • Date fields (creation, modification) are visible;
	 *   • User identifiers (createdBy, lastUpdateBy) are obfuscated or hidden;
	 *   • Other administrative details may be filtered.
	 * 
	 * <strong>Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Instance successfully retrieved</li>
	 * <li><strong>404 Not Found</strong>: Instance does not exist</li>
	 * <li><strong>403 Forbidden</strong>: Instance exists but is available in another context (not accessible in current context).</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Returns a JSON object containing the requested instance</li>
	 * <li>Includes its properties, relations, and metadata (if requested)</li>
	 * <li>Nested elements are included based on polymorphic and metadata settings.</li>
	 * </ul>
	 * 
	 * <strong>Example Metadata (filtered for non-admin users):</strong>
	 * ``{@code json
	 * "metadata": {
	 *     "type": "Metadata",
	 *     "creationTime": "2025-03-18 17:13:40.952 +0100",
	 *     "createdBy": "HIDDEN_FOR_PRIVACY",
	 *     "lastUpdateBy": "HIDDEN_FOR_PRIVACY",
	 *     "lastUpdateTime": "2025-03-19 16:21:16.805 +0100"
	 * }
	 * }``
	 * 
	 * @param type The name of the Information System type
	 * @param instanceId The unique identifier of the instance to retrieve
	 * @return JSON object containing the requested instance with its properties and relations
	 * @throws NotFoundException If the specified type or instance does not exist
	 * @throws ResourceRegistryException If an error occurs during instance retrieval
	 */
	@GET
	@Path("/{" + TypeManager.TYPE_PATH_PARAMETER + "}" + "/{" + InstancesManager.INSTANCE_UUID_PATH_PARAMETER + "}")
	@Consumes({MediaType.TEXT_PLAIN, ResourceInitializer.APPLICATION_JSON_CHARSET_UTF_8})
	@Produces(ResourceInitializer.APPLICATION_JSON_CHARSET_UTF_8)
	@Operation(
		summary = "Read Instance by Type and UUID ",
		description = """
		Retrieves a single instance with the specified type and UUID.


		**Request Examples:**
		- Basic query: GET /instances/EService/48af15ad-7e56-4157-b624-71c98cea4f8f;
		- With metadata: GET /instances/EService/48af15ad-7e56-4157-b624-71c98cea4f8f?includeMeta=true;
		- Full metadata context: GET /instances/EService/48af15ad-7e56-4157-b624-71c98cea4f8f?includeMeta=true&allMeta=true;
		- With context information: GET /instances/EService/48af15ad-7e56-4157-b624-71c98cea4f8f?includeContexts=true;
		- Hierarchical (admin only): GET /instances/EService/48af15ad-7e56-4157-b624-71c98cea4f8f?hierarchical=true;
		- Hierarchical with contexts (admin only): GET /instances/EService/48af15ad-7e56-4157-b624-71c98cea4f8f?hierarchical=true&includeContexts=true.


		**Authorization Requirements:**
		- **IS-Manager:**
			- Full access to retrieve any instance in any context;
			- Can use hierarchical querying to search across context hierarchies;
			- Receive complete, unfiltered metadata for the instance and all nested elements;
			- Metadata includes all administrative fields without obfuscation.

		- **Infrastructure-Manager:**
			- Full access to retrieve any instance in any context;
			- Can use hierarchical querying to search across context hierarchies;
			- Receive complete, unfiltered metadata for the instance and all nested elements;
			- Metadata includes all administrative fields without obfuscation.

		- **Context-Manager:**
			- Can retrieve instances in the current context and child contexts;
			- Can use hierarchical querying within their managed context hierarchy;
			- Must be Context-Manager of the current context (derived from the authorization token);
			- Receive complete, unfiltered metadata for the instance and all nested elements;
			- Metadata includes all administrative fields without obfuscation.

		- **Other Users:**
			- Can retrieve instances in the current context only;
			- Cannot use hierarchical querying (hierarchical parameter ignored);
			- Receive metadata with sensitive information filtered when includeMeta=true:
				- Date fields (creation, modification) are visible;
				- User identifiers (createdBy, lastUpdateBy) are obfuscated or hidden;
				- Other administrative details may be filtered.


		**Response Format:**
		- Returns a JSON object containing the requested instance;
		- Includes its properties, relations, and metadata (if requested);
		- Nested elements are included based on polymorphic and metadata settings.

		**Example Metadata (filtered for non-admin users):**
		```json
		"metadata": {
		    "type": "Metadata",
		    "creationTime": "2025-03-18 17:13:40.952 +0100",
		    "createdBy": "HIDDEN_FOR_PRIVACY",
		    "lastUpdateBy": "HIDDEN_FOR_PRIVACY",
		    "lastUpdateTime": "2025-03-19 16:21:16.805 +0100"
		}
		```
		"""
	)
	@APIResponse(
		responseCode = "200",
		description = "Instance successfully retrieved",
		content = @Content(mediaType = "application/json")
	)
	@APIResponse(
		responseCode = "404",
		description = "Instance does not exist"
	)
	@APIResponse(
		responseCode = "403",
		description = "Instance exists but is available in another context (not accessible in current context)"
	)
	public String read(
			@PathParam(TypeManager.TYPE_PATH_PARAMETER)
			@Schema(description = "The name of the Information System type (e.g., \"EService\", \"ContactFacet\")", example = "EService")
			String type,
			@PathParam(InstancesManager.INSTANCE_UUID_PATH_PARAMETER) String instanceId) throws NotFoundException, ResourceRegistryException {
		logger.info("Requested to read {} with id {}", type, instanceId);
		setAccountingMethod(Method.READ, InstancesManager.INSTANCE);
		
		ServerRequestInfo serverRequestInfo = initRequestInfo();
		serverRequestInfo.checkAllBooleanQueryParameters();
		
		ElementManagement<?,?> erManagement = ERManagementUtility.getERManagement(type);
		erManagement.setElementType(type);
		erManagement.setUUID(UUID.fromString(instanceId));
		return erManagement.read().toString();
	}
	
	/**
	 * Creates a new instance or updates an existing instance with the specified type and UUID in the Resource Registry.
	 * 
	 * <strong>REST Endpoint:</strong> {@code PUT /instances/{type-name}/{instance-uuid}}
	 * 
	 * <strong>Request Examples:</strong>
	 * <ul>
	 * <li>Basic create/update: {@code PUT /instances/EService/48af15ad-7e56-4157-b624-71c98cea4f8f}</li>
	 * <li>With metadata in response: {@code PUT /instances/EService/48af15ad-7e56-4157-b624-71c98cea4f8f?includeMeta=true}</li>
	 * <li>Full metadata context: {@code PUT /instances/EService/48af15ad-7e56-4157-b624-71c98cea4f8f?includeMeta=true&amp;allMeta=true}</li>
	 * <li>With context information: {@code PUT /instances/EService/48af15ad-7e56-4157-b624-71c98cea4f8f?includeContexts=true}.</li>
	 * </ul>
	 * 
	 * <strong>Path Parameters:</strong>
	 * <ul>
	 * <li><strong>type-name</strong>: The name of the Information System type (e.g., "EService", "ContactFacet")</li>
	 * <li><strong>instance-uuid</strong>: The unique identifier of the instance to create or update.</li>
	 * </ul>
	 * 
	 * <strong>Request Body:</strong>
	 * <ul>
	 * <li>JSON representation of the instance to create or update</li>
	 * <li>Must conform to the specified type schema.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>includeMeta</strong> (optional):
	 * <ul>
	 * <li>Whether to include metadata in the response instance</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Values: true|false</li>
	 * <li>Example: ?includeMeta=true (includes metadata with appropriate filtering based on user role)</li>
	 * <li>Metadata includes the created/updated instance and nested elements based on polymorphic setting</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (optional):
	 * <ul>
	 * <li>Whether to include metadata for all nested instances (ConsistsOf relations, Facets, etc.) in the response</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main instance, more human-readable)</li>
	 * <li>Values: true|false</li>
	 * <li>Example: ?includeMeta=true&amp;allMeta=true (includes metadata for all nested elements)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (optional):
	 * <ul>
	 * <li>Whether to include the list of contexts where the instance and its nested elements are available in the response</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Values: true|false</li>
	 * <li>Example: ?includeContexts=true (shows context availability for the instance and nested elements)</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present</li>
	 * <li><strong>Note:</strong> Unlike read operations, this parameter does not require hierarchical=true.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <strong>IS-Manager:</strong>
	 * <ul>
	 * <li>Full access to create/update instances in any context</li>
	 * <li>Can create/update instances across all contexts within the infrastructure</li>
	 * <li>Receive complete, unfiltered metadata for the instance and all nested elements in the response</li>
	 * <li>Metadata includes all administrative fields without obfuscation.</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Full access to create/update instances in any context</li>
	 * <li>Can create/update instances across all contexts within the infrastructure</li>
	 * <li>Receive complete, unfiltered metadata for the instance and all nested elements in the response</li>
	 * <li>Metadata includes all administrative fields without obfuscation.</li>
	 * </ul>
	 * 
	 * <strong>Context-Manager:</strong>
	 * <ul>
	 * <li>Can create/update instances in the current context</li>
	 * <li>Must be Context-Manager of the current context (derived from the authorization token)</li>
	 * <li>Cannot create/update instances in contexts where they don't have Context-Manager role</li>
	 * <li>Receive complete, unfiltered metadata for the instance and all nested elements in the response</li>
	 * <li>Metadata includes all administrative fields without obfuscation.</li>
	 * </ul>
	 * 
	 * <strong>Other Users:</strong>
	 * <ul>
	 * <li>Limited create/update permissions based on instance ownership and context access</li>
	 * <li>Only within contexts where they have authorized access through their token</li>
	 * <li>Receive metadata with sensitive information filtered when includeMeta=true:</li>
	 * </ul>
	 *   • Date fields (creation, modification) are visible;
	 *   • User identifiers (createdBy, lastUpdateBy) are obfuscated or hidden;
	 *   • Other administrative details may be filtered.
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li><strong>Create</strong>: If the instance with the specified UUID does not exist, creates a new instance</li>
	 * <li><strong>Update</strong>: If the instance with the specified UUID exists, updates the existing instance</li>
	 * <li>The operation is idempotent and atomic.</li>
	 * </ul>
	 * 
	 * <strong>Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Instance successfully updated (existing instance modified)</li>
	 * <li><strong>201 Created</strong>: Instance successfully created (new instance)</li>
	 * <li><strong>400 Bad Request</strong>: Invalid request body, malformed JSON, or schema validation failed</li>
	 * <li><strong>403 Forbidden</strong>: Insufficient permissions to create/update the instance</li>
	 * <li><strong>404 Not Found</strong>: The specified type does not exist.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Returns a JSON object containing the created or updated instance</li>
	 * <li>Includes its properties, relations, and metadata (if requested)</li>
	 * <li>Response structure depends on includeMeta, allMeta, and includeContexts parameters.</li>
	 * </ul>
	 * 
	 * <strong>Example Metadata (filtered for non-admin users):</strong>
	 * ``{@code json
	 * "metadata": {
	 *     "type": "Metadata",
	 *     "creationTime": "2025-03-18 17:13:40.952 +0100",
	 *     "createdBy": "HIDDEN_FOR_PRIVACY",
	 *     "lastUpdateBy": "HIDDEN_FOR_PRIVACY",
	 *     "lastUpdateTime": "2025-03-19 16:21:16.805 +0100"
	 * }
	 * }``
	 * 
	 * @param type The name of the Information System type
	 * @param instanceId The unique identifier of the instance to create or update
	 * @param json JSON representation of the instance to create or update
	 * @return JSON object containing the created or updated instance with its properties and relations
	 * @throws ResourceRegistryException If an error occurs during instance creation or update
	 */
	@PUT
	@Path("/{" + TypeManager.TYPE_PATH_PARAMETER + "}" + "/{" + InstancesManager.INSTANCE_UUID_PATH_PARAMETER + "}")
	@Consumes({MediaType.TEXT_PLAIN, ResourceInitializer.APPLICATION_JSON_CHARSET_UTF_8})
	@Produces(ResourceInitializer.APPLICATION_JSON_CHARSET_UTF_8)
	@Operation(
		summary = "Create or Update Instance",
		description = """
		Creates a new instance or updates an existing instance with the specified type and UUID in the Resource Registry.


		**Request Examples:**
		- Basic create/update: PUT /instances/EService/48af15ad-7e56-4157-b624-71c98cea4f8f;
		- With metadata in response: PUT /instances/EService/48af15ad-7e56-4157-b624-71c98cea4f8f?includeMeta=true;
		- Full metadata context: PUT /instances/EService/48af15ad-7e56-4157-b624-71c98cea4f8f?includeMeta=true&allMeta=true;
		- With context information: PUT /instances/EService/48af15ad-7e56-4157-b624-71c98cea4f8f?includeContexts=true.


		**Authorization Requirements:**
		- **IS-Manager:**
			- Full access to create/update instances in any context;
			- Can create/update instances across all contexts within the infrastructure;
			- Receive complete, unfiltered metadata for the instance and all nested elements in the response;
			- Metadata includes all administrative fields without obfuscation.

		- **Infrastructure-Manager:**
			- Full access to create/update instances in any context;
			- Can create/update instances across all contexts within the infrastructure;
			- Receive complete, unfiltered metadata for the instance and all nested elements in the response;
			- Metadata includes all administrative fields without obfuscation.

		- **Context-Manager:**
			- Can create/update instances in the current context;
			- Must be Context-Manager of the current context (derived from the authorization token);
			- Cannot create/update instances in contexts where they don't have Context-Manager role;
			- Receive complete, unfiltered metadata for the instance and all nested elements in the response;
			- Metadata includes all administrative fields without obfuscation.

		- **Other Users:**
			- Limited create/update permissions based on instance ownership and context access;
			- Only within contexts where they have authorized access through their token;
			- Receive metadata with sensitive information filtered when includeMeta=true:
				- Date fields (creation, modification) are visible;
				- User identifiers (createdBy, lastUpdateBy) are obfuscated or hidden;
				- Other administrative details may be filtered.


		**Operation Behavior:**
		- **Create**: If the instance with the specified UUID does not exist, creates a new instance;
		- **Update**: If the instance with the specified UUID exists, updates the existing instance;
		- The operation is idempotent and atomic.


		**Response Format:**
		- Returns a JSON object containing the created or updated instance;
		- Includes its properties, relations, and metadata (if requested);
		- Response structure depends on includeMeta, allMeta, and includeContexts parameters.

		**Example Metadata (filtered for non-admin users):**
		```json
		"metadata": {
		    "type": "Metadata",
		    "creationTime": "2025-03-18 17:13:40.952 +0100",
		    "createdBy": "HIDDEN_FOR_PRIVACY",
		    "lastUpdateBy": "HIDDEN_FOR_PRIVACY",
		    "lastUpdateTime": "2025-03-19 16:21:16.805 +0100"
		}
		```
		"""
	)
	@APIResponse(
		responseCode = "200",
		description = "Instance successfully updated (existing instance modified)",
		content = @Content(mediaType = "application/json")
	)
	@APIResponse(
		responseCode = "201",
		description = "Instance successfully created (new instance)",
		content = @Content(mediaType = "application/json")
	)
	@APIResponse(
		responseCode = "400",
		description = "Invalid request body, malformed JSON, or schema validation failed"
	)
	@APIResponse(
		responseCode = "403",
		description = "Insufficient permissions to create/update the instance"
	)
	@APIResponse(
		responseCode = "404",
		description = "The specified type does not exist"
	)
	public String updateOrCreate(
			@PathParam(TypeManager.TYPE_PATH_PARAMETER)
			@Schema(description = "The name of the Information System type (e.g., \"EService\", \"ContactFacet\")", example = "EService")
			String type,
			@PathParam(InstancesManager.INSTANCE_UUID_PATH_PARAMETER) String instanceId, String json) throws ResourceRegistryException {
		logger.info("Requested to update/create {} with id {}", type, instanceId);
		logger.trace("Requested to update/create {} with id {} with json {}", type, instanceId, json);
		setAccountingMethod(Method.UPDATE, InstancesManager.INSTANCE);
		
		ServerRequestInfo serverRequestInfo = initRequestInfo();
		serverRequestInfo.checkAllIncludeQueryParameters();
		
		@SuppressWarnings("rawtypes")
		ElementManagement erManagement = ERManagementUtility.getERManagement(type);
		erManagement.setUUID(UUID.fromString(instanceId));
		erManagement.setElementType(type);
		erManagement.setJson(json);
		return erManagement.createOrUpdate();
	}
	
	/**
	 * Deletes an instance with the specified type and UUID from the Resource Registry.
	 * 
	 * <strong>REST Endpoint:</strong> {@code DELETE /instances/{type-name}/{instance-uuid}}
	 * 
	 * <strong>Request Examples:</strong>
	 * <ul>
	 * <li>Delete resource (e.g., EService): {@code DELETE /instances/EService/48af15ad-7e56-4157-b624-71c98cea4f8f}</li>
	 * <li>Delete facet (e.g., ContactFacet): {@code DELETE /instances/ContactFacet/4023d5b2-8601-47a5-83ef-49ffcbfc7d86}.</li>
	 * </ul>
	 * 
	 * <strong>Path Parameters:</strong>
	 * <ul>
	 * <li><strong>type-name</strong>: The name of the Information System type (e.g., "EService", "ContactFacet")</li>
	 * <li><strong>instance-uuid</strong>: The unique identifier of the instance to delete.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * <ul>
	 * <li>This endpoint does not accept any query parameters</li>
	 * <li>The deletion operates only in the current context (derived from the authorization token)</li>
	 * <li>No hierarchical or metadata options are available for delete operations.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <strong>IS-Manager:</strong>
	 * <ul>
	 * <li>Can delete any instance from any context</li>
	 * <li>Full administrative access to deletion operations across all contexts.</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Can delete any instance from any context</li>
	 * <li>Full administrative access to deletion operations across all contexts.</li>
	 * </ul>
	 * 
	 * <strong>Context-Manager:</strong>
	 * <ul>
	 * <li>Can delete instances if the request is made from a context where they have Context-Manager role</li>
	 * <li>Must be Context-Manager of the current context (derived from the authorization token)</li>
	 * <li>The current context must be one of the contexts where the instance exists.</li>
	 * </ul>
	 * 
	 * <strong>Other Users:</strong>
	 * <ul>
	 * <li>Can delete instances if the request is made from a context where the instance is present</li>
	 * <li>The current context (derived from the authorization token) must be one of the contexts where the instance exists</li>
	 * <li>Only within contexts where they have authorized access through their token.</li>
	 * </ul>
	 *
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li><strong>Global Deletion</strong>: The instance is deleted from all contexts where it exists, not just the current context</li>
	 * <li><strong>IsRelatedTo Relations Cleanup</strong>: All IsRelatedTo relations involving the deleted instance (both as source and target) are deleted from all contexts to prevent broken relations and maintain referential integrity</li>
	 * <li><strong>Cascading Effects for IsRelatedTo</strong>: Connected Resources are deleted only if propagation constraints require it and only for relations where the deleted instance is the source (relations are followed in their direction) and only within the current context</li>
	 * <li><strong>Complete ConsistsOf Deletion</strong>: All ConsistsOf relations and their target Facets are deleted regardless of context</li>
	 * <li><strong>Facet/ConsistsOf Deletion Behavior</strong>: When deleting a Facet or ConsistsOf relation, both the ConsistsOf relation and the Facet must be present in the current context. The source Resource is revalidated against its type schema after deletion - if the Resource becomes invalid, the deletion operation fails with a 400 Bad Request error (SchemaViolationException)</li>
	 * <li><strong>Schema Validation</strong>: When deleting Facets or ConsistsOf relations, the source Resource is validated against its type schema after deletion to ensure consistency - validation failures result in a 400 Bad Request error (SchemaViolationException)</li>
	 * <li><strong>Permission Validation</strong>: User permissions are validated against the current context</li>
	 * <li><strong>Atomicity</strong>: The operation is atomic - either the instance and all cascaded deletions succeed, or the entire operation fails.</li>
	 * </ul>
	 * 
	 * <strong>Response Codes:</strong>
	 * <ul>
	 * <li><strong>204 No Content</strong>: Instance successfully deleted</li>
	 * <li><strong>400 Bad Request</strong>: Schema validation failed after deletion (SchemaViolationException) - occurs when deleting a Facet or ConsistsOf relation would leave the source Resource in an invalid state according to its type schema</li>
	 * <li><strong>404 Not Found</strong>: Instance does not exist</li>
	 * <li><strong>403 Forbidden</strong>: Instance exists but is available in another context (not accessible in current context).</li>
	 * </ul>
	 * 
	 * @param type The name of the Information System type
	 * @param instanceId The unique identifier of the instance to delete
	 * @return HTTP response with status 204 (No Content) on successful deletion
	 * @throws ResourceRegistryException If an error occurs during instance deletion, including permission or constraint violations
	 */
	@DELETE
	@Path("/{" + TypeManager.TYPE_PATH_PARAMETER + "}" + "/{" + InstancesManager.INSTANCE_UUID_PATH_PARAMETER + "}")
	@Operation(
		summary = "Delete Instance",
		description = """
		Deletes an instance with the specified type and UUID from the Resource Registry.


		**Request Examples:**
		- Delete resource (e.g., EService): DELETE /instances/EService/48af15ad-7e56-4157-b624-71c98cea4f8f;
		- Delete facet (e.g., ContactFacet): DELETE /instances/ContactFacet/4023d5b2-8601-47a5-83ef-49ffcbfc7d86.


		**Authorization Requirements:**
		- **IS-Manager:**
			- Can delete any instance from any context;
			- Full administrative access to deletion operations across all contexts.

		- **Infrastructure-Manager:**
			- Can delete any instance from any context;
			- Full administrative access to deletion operations across all contexts.

		- **Context-Manager:**
			- Can delete instances if the request is made from a context where they have Context-Manager role;
			- Must be Context-Manager of the current context (derived from the authorization token);
			- The current context must be one of the contexts where the instance exists.

		- **Other Users:**
			- Can delete instances if the request is made from a context where the instance is present;
			- The current context (derived from the authorization token) must be one of the contexts where the instance exists;
			- Only within contexts where they have authorized access through their token.


		**Operation Behavior:**
		- **Global Deletion**: The instance is deleted from all contexts where it exists, not just the current context;
		- **IsRelatedTo Relations Cleanup**: All IsRelatedTo relations involving the deleted instance (both as source and target) are deleted from all contexts to prevent broken relations and maintain referential integrity;
		- **Cascading Effects for IsRelatedTo**: Connected Resources are deleted only if propagation constraints require it and only for relations where the deleted instance is the source (relations are followed in their direction) and only within the current context;
		- **Complete ConsistsOf Deletion**: All ConsistsOf relations and their target Facets are deleted regardless of context;
		- **Facet/ConsistsOf Deletion Behavior**: When deleting a Facet or ConsistsOf relation, both the ConsistsOf relation and the Facet must be present in the current context. The source Resource is revalidated against its type schema after deletion - if the Resource becomes invalid, the deletion operation fails with a 400 Bad Request error (SchemaViolationException);
		- **Schema Validation**: When deleting Facets or ConsistsOf relations, the source Resource is validated against its type schema after deletion to ensure consistency - validation failures result in a 400 Bad Request error (SchemaViolationException);
		- **Permission Validation**: User permissions are validated against the current context;
		- **Atomicity**: The operation is atomic - either the instance and all cascaded deletions succeed, or the entire operation fails.
		"""
	)
	@APIResponse(
		responseCode = "204",
		description = "Instance successfully deleted"
	)
	@APIResponse(
		responseCode = "400",
		description = "Schema validation failed after deletion (SchemaViolationException) - occurs when deleting a Facet or ConsistsOf relation would leave the source Resource in an invalid state according to its type schema"
	)
	@APIResponse(
		responseCode = "404",
		description = "Instance does not exist"
	)
	@APIResponse(
		responseCode = "403",
		description = "Instance exists but is available in another context (not accessible in current context)"
	)
	public Response delete(
			@PathParam(TypeManager.TYPE_PATH_PARAMETER)
			@Schema(description = "The name of the Information System type (e.g., \"EService\", \"ContactFacet\")", example = "EService")
			String type,
			@PathParam(InstancesManager.INSTANCE_UUID_PATH_PARAMETER) String instanceId) throws ResourceRegistryException {
		logger.info("Requested to delete {} with id {}", type, instanceId);
		setAccountingMethod(Method.DELETE, InstancesManager.INSTANCE);
		
		ElementManagement<?,?> erManagement = ERManagementUtility.getERManagement(type);
		erManagement.setUUID(UUID.fromString(instanceId));
		erManagement.delete();
		
		return Response.status(Status.NO_CONTENT).build();
	}
	
	
	/**
	 * Retrieves the list of contexts where a specific instance is present in the Resource Registry.
	 * 
	 * <strong>REST Endpoint:</strong> {@code GET /instances/{type-name}/{instance-uuid}/contexts}
	 * 
	 * <strong>Request Examples:</strong>
	 * <ul>
	 * <li>Get contexts for a resource: {@code GET /instances/EService/48af15ad-7e56-4157-b624-71c98cea4f8f/contexts}</li>
	 * <li>Get contexts for a facet: {@code GET /instances/ContactFacet/4023d5b2-8601-47a5-83ef-49ffcbfc7d86/contexts}.</li>
	 * </ul>
	 * 
	 * <strong>Path Parameters:</strong>
	 * <ul>
	 * <li><strong>type-name</strong>: The name of the Information System type (e.g., "EService", "ContactFacet")</li>
	 * <li><strong>instance-uuid</strong>: The unique identifier of the instance whose contexts to retrieve.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * <ul>
	 * <li>This endpoint does not accept any query parameters</li>
	 * <li>No filtering, pagination, or metadata options are available.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <strong>IS-Manager:</strong>
	 * <ul>
	 * <li>Can retrieve context information for any instance</li>
	 * <li>Full access to view contexts across the entire infrastructure.</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Can retrieve context information for any instance</li>
	 * <li>Full access to view contexts across the entire infrastructure.</li>
	 * </ul>
	 * 
	 * <strong>Context-Manager:</strong>
	 * <ul>
	 * <li>Can retrieve context information for any accessible instance</li>
	 * <li>Access depends on their Context-Manager role in relevant contexts.</li>
	 * </ul>
	 * 
	 * <strong>Other Users:</strong>
	 * <ul>
	 * <li>Can retrieve context information for any accessible instance</li>
	 * <li>Access depends on their permissions and context availability through their token.</li>
	 * </ul>
	 * 
	 * <strong>Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Context list successfully retrieved</li>
	 * <li><strong>404 Not Found</strong>: Instance does not exist</li>
	 * <li><strong>403 Forbidden</strong>: Instance exists but is available in another context (not accessible for context listing).</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Returns a JSON object containing the contexts where the instance is present</li>
	 * <li>Each key is a context UUID and each value is the current full context path</li>
	 * <li>Empty object is returned if the instance exists but is not present in any contexts.</li>
	 * </ul>
	 * 
	 * <strong>Example Response:</strong>
	 * ``{@code json
	 * {
	 *   "3d846e2f-3582-4344-b03a-629d4cd34d53": "/gcube/devsec/devVRE",
	 *   "4eb849d5-efbb-4430-9eb6-1968123921b4": "/gcube",
	 *   "a7bc333f-84df-498d-ae9d-748b358267b5": "/gcube/devsec"
	 * }
	 * }``
	 * 
	 * @param type The name of the Information System type
	 * @param instanceId The unique identifier of the instance
	 * @return JSON object containing context UUID to context path mappings where the instance is present
	 * @throws ResourceNotFoundException If the instance does not exist
	 * @throws ContextNotFoundException If context-related errors occur
	 * @throws ResourceRegistryException If an error occurs during context retrieval
	 */
	@GET
	@Path("{" + TypeManager.TYPE_PATH_PARAMETER + "}" + "/{" + InstancesManager.INSTANCE_UUID_PATH_PARAMETER + "}/"
			+ SharingPath.CONTEXTS_PATH_PART)
	@Produces(ResourceInitializer.APPLICATION_JSON_CHARSET_UTF_8)
	@Operation(
		summary = "Get Instance Contexts",
		description = """
		Retrieves the list of contexts where a specific instance is present in the Resource Registry.


		**Request Examples:**
		- Get contexts for a resource: GET /instances/EService/48af15ad-7e56-4157-b624-71c98cea4f8f/contexts;
		- Get contexts for a facet: GET /instances/ContactFacet/4023d5b2-8601-47a5-83ef-49ffcbfc7d86/contexts.


		**Authorization Requirements:**
		- **IS-Manager:**
			- Can retrieve context information for any instance;
			- Full access to view contexts across the entire infrastructure.

		- **Infrastructure-Manager:**
			- Can retrieve context information for any instance;
			- Full access to view contexts across the entire infrastructure.

		- **Context-Manager:**
			- Can retrieve context information for any accessible instance;
			- Access depends on their Context-Manager role in relevant contexts.

		- **Other Users:**
			- Can retrieve context information for any accessible instance;
			- Access depends on their permissions and context availability through their token.


		**Response Format:**
		- Returns a JSON object containing the contexts where the instance is present;
		- Each key is a context UUID and each value is the current full context path;
		- Empty object is returned if the instance exists but is not present in any contexts.

		**Example Response:**
		```json
		{
		  "3d846e2f-3582-4344-b03a-629d4cd34d53": "/gcube/devsec/devVRE",
		  "4eb849d5-efbb-4430-9eb6-1968123921b4": "/gcube",
		  "a7bc333f-84df-498d-ae9d-748b358267b5": "/gcube/devsec"
		}
		```
		"""
	)
	@APIResponse(
		responseCode = "200",
		description = "Context list successfully retrieved",
		content = @Content(mediaType = "application/json")
	)
	@APIResponse(
		responseCode = "404",
		description = "Instance does not exist"
	)
	@APIResponse(
		responseCode = "403",
		description = "Instance exists but is available in another context (not accessible for context listing)"
	)
	public String getInstanceContexts(
			@PathParam(TypeManager.TYPE_PATH_PARAMETER)
			@Schema(description = "The name of the Information System type (e.g., \"EService\", \"ContactFacet\")", example = "EService")
			String type,
			@PathParam(InstancesManager.INSTANCE_UUID_PATH_PARAMETER) String instanceId)
			throws ResourceNotFoundException, ContextNotFoundException, ResourceRegistryException {
		logger.info("Requested to get contexts of {} with UUID {}", type, instanceId);
		setAccountingMethod(InstancesManager.GET_INSTANCE_CONTEXTS_METHOD);

		@SuppressWarnings("rawtypes")
		ElementManagement erManagement = ERManagementUtility.getERManagement(type);
		erManagement.setUUID(UUID.fromString(instanceId));
		return erManagement.getContexts();
	}
	
}
