package org.gcube.keycloak.protocol.oidc;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.http.HttpEntity;
import org.apache.http.HttpHeaders;
import org.apache.http.NameValuePair;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.util.EntityUtils;
import org.jboss.logging.Logger;
import org.keycloak.events.Details;
import org.keycloak.events.Errors;
import org.keycloak.events.EventBuilder;
import org.keycloak.jose.jws.JWSInput;
import org.keycloak.jose.jws.JWSInputException;
import org.keycloak.models.KeycloakSession;
import org.keycloak.protocol.oidc.AccessTokenIntrospectionProvider;
import org.keycloak.protocol.oidc.TokenIntrospectionProvider;
import org.keycloak.representations.AccessToken;
import org.keycloak.services.Urls;
import org.keycloak.util.BasicAuthHelper;
import org.keycloak.util.JsonSerialization;

import com.fasterxml.jackson.databind.node.ObjectNode;

import jakarta.ws.rs.core.MediaType;
import jakarta.ws.rs.core.Response;

public class EOSCNodeAccessTokenIntrospectionProvider<T extends AccessToken> extends AccessTokenIntrospectionProvider<T>
        implements TokenIntrospectionProvider {

    private static final int SOCKET_TIMEOUT = 10000;

    private static final int CONNECTION_TIMEOUT = 5000;

    private static final Logger logger = Logger.getLogger(EOSCNodeAccessTokenIntrospectionProvider.class);

    protected static String serverIssuer;

    protected final MyAccessIDConfiguration myAccessIDConfiguration;

    public EOSCNodeAccessTokenIntrospectionProvider(KeycloakSession session,
            MyAccessIDConfiguration myAccessIDConfiguration) {

        super(session);
        if (serverIssuer == null) {
            // Getting the actual KC server's issuer
            //            serverIssuer = ((OIDCConfigurationRepresentation) new OIDCWellKnownProviderFactory()
            //                    .create(session).getConfig()).getIssuer();
            serverIssuer = Urls.realmIssuer(session.getContext().getUri().getBaseUri(),
                    session.getContext().getRealm().getName());

            logger.infof("Keycloak server instance 'issuer' is: %s", serverIssuer);
        }
        this.myAccessIDConfiguration = myAccessIDConfiguration;
    }

    @Override
    public void close() {
        super.close();
    }

    @Override
    public Response introspect(String accessTokenString, EventBuilder eventBuilder) {
        AccessToken accessToken = null;
        try {
            logger.debug("Deserializing the recevide access token");
            accessToken = new JWSInput(accessTokenString).readJsonContent(AccessToken.class);
//            accessToken = JsonSerialization.readValue(accessTokenString, AccessToken.class);
        } catch (JWSInputException e) {
            logger.error("Can't deserialize access token from string", e);
            eventBuilder.detail(Details.REASON,
                    "Can't deserialize access token from string. Reason: " + e.getMessage());
            eventBuilder.error(Errors.TOKEN_INTROSPECTION_FAILED);
            throw new RuntimeException("Error parsing access token string", e);
        }
        logger.trace("Getting the issuer from the access token");
        String tokenIssuer = accessToken.getIssuer();
        logger.debugf("Access token issued by: %s", tokenIssuer);

        if (serverIssuer.equals(tokenIssuer)) {
            logger.debugf("Token is issued by this server, introspect it internally", tokenIssuer);
            return super.introspect(accessTokenString, eventBuilder);
        } else {
            logger.debugf("Token is issued by server at %s, introspect it on MyAccess", tokenIssuer);
            return performIntrospectionOnMyAccess(accessTokenString);
        }

    }

    protected Response performIntrospectionOnMyAccess(String accessTokenString) {
        logger.trace("Creating and perfom POST with the HTTP client");
        try (CloseableHttpClient httpClient = HttpClients.createDefault()) {
            logger.trace("Starting HTTP post creation");
            HttpPost httpPost = new HttpPost(this.myAccessIDConfiguration.getIntrospectionURL());

            logger.trace("Setting the Authorization header");
            httpPost.setHeader(HttpHeaders.AUTHORIZATION,
                    BasicAuthHelper.createHeader(this.myAccessIDConfiguration.getClientId(),
                            this.myAccessIDConfiguration.getClientScret()));

            logger.tracef("Setting the token parameter: %s", accessTokenString);
            List<NameValuePair> form = new ArrayList<>();
            form.add(new BasicNameValuePair("token", accessTokenString));
            UrlEncodedFormEntity entity = new UrlEncodedFormEntity(form, StandardCharsets.UTF_8);
            httpPost.setEntity(entity);

            logger.tracef("Setting connection timeout to %d millis", CONNECTION_TIMEOUT);
            logger.tracef("Setting socket timeout to %d millis", SOCKET_TIMEOUT);
            RequestConfig requestConfig = RequestConfig.custom()
                    .setConnectTimeout(CONNECTION_TIMEOUT)
                    .setSocketTimeout(SOCKET_TIMEOUT)
                    .build();

            httpPost.setConfig(requestConfig);

            logger.debug("Performing the request...");
            try (CloseableHttpResponse response = httpClient.execute(httpPost)) {
                int statusCode = response.getStatusLine().getStatusCode();
                logger.debugf("Resulting status code is %d", statusCode);

                logger.trace("Getting the response entity");
                HttpEntity responseEntity = response.getEntity();

                logger.trace("Reading the response body");
                String responseBodyString = (responseEntity != null)
                        ? EntityUtils.toString(responseEntity, StandardCharsets.UTF_8)
                        : "";

                logger.debugf("Resulting body is: %s", responseBodyString);
                if (statusCode >= 200 && statusCode < 300) {
                    ObjectNode myAccessIntrospectionResponseObjectNode = JsonSerialization.readValue(responseBodyString,
                            ObjectNode.class);
                    if (!this.myAccessIDConfiguration.getClaimRenamings().isEmpty()) {
                        logger.debug("Performing the claim renaming...");
                        for (String originalClaimName : this.myAccessIDConfiguration.getClaimRenamings().keySet()) {
                            String newClaimName = this.myAccessIDConfiguration.getClaimRenamings()
                                    .get(originalClaimName);

                            logger.tracef("Renaming the '%s' claim to '%s'", originalClaimName, newClaimName);
                            myAccessIntrospectionResponseObjectNode.set(newClaimName,
                                    myAccessIntrospectionResponseObjectNode.get(originalClaimName));
                            myAccessIntrospectionResponseObjectNode.remove(originalClaimName);
                        }
                    } else {
                        logger.debug("Claim renaming is not needed");
                    }
                    logger.tracef("Serializing response JSON object node as string");
                    byte[] myAccessIntrospectionResponseBytes = JsonSerialization
                            .writeValueAsBytes(myAccessIntrospectionResponseObjectNode);
                    logger.tracef("Resulting response JSON string is: %s", myAccessIntrospectionResponseBytes);
                    // TODO set info on eventBuilder if it makes sense to trace them for success external introspections
                    return Response.ok(myAccessIntrospectionResponseBytes).type(MediaType.APPLICATION_JSON_TYPE)
                            .build();
                } else {
                    logger.errorf("MyAccess server response is not OK [%d]: %s", statusCode, responseBodyString);
                    eventBuilder.detail(Details.REASON, responseBodyString);
                    eventBuilder.error(Errors.TOKEN_INTROSPECTION_FAILED);
                    throw new RuntimeException("Error calling MyAccess for token introspection");
                }
            }
        } catch (IOException e) {
            logger.error("An errord occurred performing the token introspection on MyAccess", e);
            eventBuilder.detail(Details.REASON, e.getMessage());
            eventBuilder.error(Errors.TOKEN_INTROSPECTION_FAILED);
            throw new RuntimeException("Error performing the token introspection on MyAccess", e);
        }
    }

    public static class MyAccessIDConfiguration {

        private final String introspectionURL;
        private final String clientId;
        private final String clientScret;
        private final Map<String, String> claimRenamings;

        public MyAccessIDConfiguration(String introspectionURL, String clientId, String clientScret,
                String[] claimRenamings) {

            this.introspectionURL = introspectionURL;
            this.clientId = clientId;
            this.clientScret = clientScret;

            this.claimRenamings = new HashMap<>();
            if (claimRenamings != null && claimRenamings.length > 0) {
                for (String claimRenaming : claimRenamings) {
                    String[] old2new = claimRenaming.split(":");
                    this.claimRenamings.put(old2new[0], old2new[1]);
                }
            }

        }

        public String getIntrospectionURL() {
            return introspectionURL;
        }

        public String getClientId() {
            return clientId;
        }

        public String getClientScret() {
            return clientScret;
        }

        public Map<String, String> getClaimRenamings() {
            return claimRenamings;
        }

        @Override
        public String toString() {
            return "introspectionURL: " + getIntrospectionURL() + "; clientId: " + getClientId() + "; clientScret: "
                    + (getClientScret() != null ? getClientScret().replaceAll(".", "*") : null) + "; claimRenamings: "
                    + getClaimRenamings();
        }

    }

}
