package org.gcube.common.iam;

import org.gcube.common.keycloak.KeycloakClient;
import org.gcube.common.keycloak.KeycloakClientException;
import org.gcube.common.keycloak.model.TokenResponse;

/**
 * D4Science IAM client authentication implementation for client credentials flow.
 * 
 * @author Luca Frosini (ISTI-CNR)
 */
public class D4ScienceIAMClientAuthn4Client extends D4ScienceIAMClientAuthn {

    /**
     * Creates a new client authentication instance using client credentials.
     * 
     * @param iamClient the D4Science IAM client instance
     * @param clientId the client ID
     * @param clientSecret the client secret
     * @throws D4ScienceIAMClientException if authentication fails
     */
    protected D4ScienceIAMClientAuthn4Client(D4ScienceIAMClient iamClient, String clientId, String clientSecret)
            throws D4ScienceIAMClientException {

        this(iamClient, clientId, clientSecret, null);
    }

    /**
     * Creates a new client authentication instance with specific context.
     * 
     * @param iamClient the D4Science IAM client instance
     * @param clientId the client ID
     * @param clientSecret the client secret
     * @param context the requested token context audience
     * @throws D4ScienceIAMClientException if authentication fails
     */
    protected D4ScienceIAMClientAuthn4Client(D4ScienceIAMClient iamClient, String clientId, String clientSecret,
            String context) throws D4ScienceIAMClientException {

        super(iamClient, performClientAuthn(iamClient, clientId, clientSecret, context));
    }

    /**
     * Performs the actual client authentication using client credentials flow.
     * 
     * @param iamClient the D4Science IAM client instance
     * @param clientId the client ID
     * @param clientSecret the client secret
     * @param context the requested token context audience
     * @return the token response from authentication
     * @throws D4ScienceIAMClientException if authentication fails
     */
    protected static final TokenResponse performClientAuthn(D4ScienceIAMClient iamClient, String clientId,
            String clientSecret, String context) throws D4ScienceIAMClientException {

        KeycloakClient keycloakClient = iamClient.getKeycloakClient();
        try {
            return keycloakClient.queryOIDCTokenWithContext(
                    keycloakClient.getTokenEndpointURL(iamClient.getRealmBaseURL()), clientId, clientSecret, context);

        } catch (KeycloakClientException e) {
            throw new D4ScienceIAMClientException(e);
        }
    }

}