package org.gcube.common.software.processor.zenodo;

import java.io.File;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.TimeUnit;

import jakarta.ws.rs.client.Client;
import jakarta.ws.rs.client.ClientBuilder;
import jakarta.ws.rs.client.Entity;
import jakarta.ws.rs.client.WebTarget;
import jakarta.ws.rs.client.Invocation.Builder;
import jakarta.ws.rs.core.MediaType;
import jakarta.ws.rs.core.Response;
import jakarta.ws.rs.core.Response.StatusType;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.JsonNodeType;
import com.fasterxml.jackson.databind.node.ObjectNode;
import org.gcube.common.software.analyser.AnalyserFactory;
import org.gcube.common.software.config.Config;
import org.gcube.common.software.model.ElaborationType;
import org.gcube.common.software.model.SoftwareArtifactFile;
import org.gcube.common.software.model.SoftwareArtifactMetadata;
import org.gcube.common.software.processor.SoftwareArtifactProcessor;
import org.gcube.common.software.utils.Utils;
import org.glassfish.jersey.client.ClientProperties;
import org.glassfish.jersey.media.multipart.FormDataMultiPart;
import org.glassfish.jersey.media.multipart.MultiPartFeature;
import org.glassfish.jersey.media.multipart.file.FileDataBodyPart;
import org.glassfish.jersey.jackson.JacksonFeature;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author Luca Frosini (ISTI - CNR)
 */
public class ZenodoExporter extends SoftwareArtifactProcessor {
	
	private static final Logger logger = LoggerFactory.getLogger(ZenodoExporter.class);
	
	public static final String EXPORT_FILENAME_EXTENSION = ".json";
	
	public static final String GUCBE_ZENODO_SOFTWARE_DEPOSIT = "gCubeSoftwareDeposit";
	
	public static final String HTML_DESCRIPTION_CONFIG_FIELD_NAME = "html_description";
	public static final String ADDITIONAL_HTML_DESCRIPTION_CONFIG_FIELD_NAME = "additional_html_description";
	public static final String SKIP_GRANTS_CONFIG_FIELD_NAME = "skip_grants";
	
	public static final String METADATA_FIELD_NAME = "metadata";
	public static final String COMMUNITIES_FIELD_NAME = "communities";
	
	public static final String DEPOSITIONS_COLLECTION_PATH = "/api/deposit/depositions";
	public static final String DEPOSITION_PATH = DEPOSITIONS_COLLECTION_PATH + "/:id";
	
	public static final String RECORD_PATH = "/api/records/:id";
	
	public static final String DEPOSTION_FILES_PATH = 		DEPOSITION_PATH + "/files";
	
	public static final String DEPOSTION_NEW_VERSION_PATH = DEPOSITION_PATH + "/actions/newversion";
	public static final String DEPOSTION_EDIT_PATH = 		DEPOSITION_PATH + "/actions/edit";
	public static final String DEPOSTION_PUBLISH_PATH = 	DEPOSITION_PATH + "/actions/publish";
	
	protected URL zenodoBaseURL;
	protected String accessToken;
	
	protected String zenodoID;
	protected JsonNode response;
	protected String doiBaseURL;
	
	protected String getZenodoIDFromDOIURL(URL doiURL) {
		return getZenodoIDFromDOIURL(doiURL.toString());
	}
	
	protected String getZenodoIDFromDOIURL(String doiURL) {
		return doiURL.replace(doiBaseURL, "");
	}
	
	public ZenodoExporter() {
		super(ZenodoExporter.EXPORT_FILENAME_EXTENSION);
	}

	/**
	 * Handles Jersey Response objects and applies rate limiting
	 */
	private JsonNode getResponse(Response response) throws Exception {
		String content = null;
		try {
			StatusType statusType = response.getStatusInfo();
			int responseCode = statusType.getStatusCode();
			String responseMessage = statusType.getReasonPhrase();
			
			StringBuffer responseBuffer = new StringBuffer();
			responseBuffer.append("Response: ").append(responseCode).append(" ").append(responseMessage);
			try {
				content = response.readEntity(String.class);
				if(content != null &&! content.trim().isEmpty()) {
					responseBuffer.append(" - Content: ");
					responseBuffer.append(content);
				}
			} catch (Exception e) {
				logger.trace("Could not read response content: {}", e.getMessage());
				content = "Could not read response content: " + e.getMessage();
				responseBuffer.append(" - ").append(content);
			}
			
			// logger.trace(responseBuffer.toString());

			if(responseCode >= HttpURLConnection.HTTP_BAD_REQUEST) {
				logger.error(responseBuffer.toString());
				throw new RuntimeException(responseBuffer.toString());
			}
			
			// Success case - parse JSON if content is available
			if(content != null && !content.trim().isEmpty()) {
				return Utils.getObjectMapper().readTree(content);
			} else {
				return null;
			}
		} finally {
			response.close();
		}
	}
	
	protected void addFilesToDeposition(List<File> files) throws Exception {
		logger.debug("Going to add files {} to deposition for {}", files, softwareArtifactMetadata.getTitle());
		String depositID = getZenodoIDFromDOIURL(softwareArtifactMetadata.getVersionDOIURL());
		String newFilePath = DEPOSTION_FILES_PATH.replace(":id", depositID);
		
		for(File file : files) {
			Client client = getJerseyClientWithAutoHeaders(zenodoBaseURL.toString());
			try {
				client.register(MultiPartFeature.class);
				FormDataMultiPart multi = new FormDataMultiPart();
				FileDataBodyPart fileDataBodyPart = new FileDataBodyPart("file", file, MediaType.APPLICATION_OCTET_STREAM_TYPE);
				multi.field("name", file.getName());
				multi.bodyPart(fileDataBodyPart);
	
				Response response = client.target(zenodoBaseURL.toString())
						.path(newFilePath)
						.request(MediaType.APPLICATION_JSON)
						.post(Entity.entity(multi, multi.getMediaType()));
	
				// getResponse() internally calls response.close()
				getResponse(response);
			} finally {
				// Always close the client to free resources (connection pools, threads, etc.)
				client.close();
			}
		}
	}
	
	/**
	 * Client Response Filter to automatically apply rate limiting after each response
	 */
	public static class RateLimitResponseFilter implements jakarta.ws.rs.client.ClientResponseFilter {
		private static final Logger logger = LoggerFactory.getLogger(RateLimitResponseFilter.class);
		
		// Default delays for different scenarios
		private static final long DEFAULT_DELAY_MS = 1000; // 1 second default
		private static final long LOW_REMAINING_DELAY_MS = 2000; // 2 seconds when approaching limit
		private static final long VERY_LOW_REMAINING_DELAY_MS = 5000; // 5 seconds when very close to limit
		
		public RateLimitResponseFilter() {
			
		}
		
		@Override
		public void filter(jakarta.ws.rs.client.ClientRequestContext requestContext, 
				jakarta.ws.rs.client.ClientResponseContext responseContext) {
			
			// Log Zenodo rate limiting headers if present
			String rateLimitLimit = responseContext.getHeaderString("X-RateLimit-Limit");
			String rateLimitRemaining = responseContext.getHeaderString("X-RateLimit-Remaining");
			String rateLimitReset = responseContext.getHeaderString("X-RateLimit-Reset");
			
			if (rateLimitLimit != null || rateLimitRemaining != null || rateLimitReset != null) {
				logger.debug("Zenodo Rate Limit Info - Limit: {}, Remaining: {}, Reset: {}", 
					rateLimitLimit != null ? rateLimitLimit : "N/A",
					rateLimitRemaining != null ? rateLimitRemaining : "N/A", 
					rateLimitReset != null ? rateLimitReset : "N/A");
			}
			
			// Calculate intelligent delay based on rate limit information
			long delayMs = calculateDelay(rateLimitLimit, rateLimitRemaining, rateLimitReset);
			
			if (delayMs > 0) {
				try {
					logger.trace("Applying intelligent rate limit delay of {} ms", delayMs);
					Thread.sleep(delayMs);
				} catch (InterruptedException e) {
					Thread.currentThread().interrupt();
					logger.warn("Rate limit delay interrupted", e);
				}
			}
		}
		
		/**
		 * Calculate appropriate delay based on Zenodo rate limit headers
		 * @param limitHeader X-RateLimit-Limit header value
		 * @param remainingHeader X-RateLimit-Remaining header value
		 * @param resetHeader X-RateLimit-Reset header value
		 * @return delay in milliseconds
		 */
		private long calculateDelay(String limitHeader, String remainingHeader, String resetHeader) {
			// If no rate limit headers present, use default delay
			if (remainingHeader == null) {
				return DEFAULT_DELAY_MS;
			}
			
			try {
				int remaining = Integer.parseInt(remainingHeader);
				
				// Parse limit and reset time if available for more sophisticated calculation
				Integer limit = null;
				Long resetTime = null;
				
				if (limitHeader != null) {
					try {
						limit = Integer.parseInt(limitHeader);
					} catch (NumberFormatException e) {
						logger.trace("Could not parse rate limit value: {}", limitHeader);
					}
				}
				
				if (resetHeader != null) {
					try {
						resetTime = Long.parseLong(resetHeader);
					} catch (NumberFormatException e) {
						logger.trace("Could not parse reset time value: {}", resetHeader);
					}
				}
				
				// Calculate delay based on remaining requests
				if (remaining <= 2) {
					logger.warn("Zenodo Rate Limit Critical: Only {} requests remaining - applying extended delay", remaining);
					return VERY_LOW_REMAINING_DELAY_MS;
				} else if (remaining <= 10) {
					logger.warn("Zenodo Rate Limit Warning: Only {} requests remaining - applying increased delay", remaining);
					return LOW_REMAINING_DELAY_MS;
				} else if (limit != null && remaining <= (limit * 0.2)) {
					// Less than 20% of requests remaining
					logger.debug("Zenodo Rate Limit: Less than 20% requests remaining ({}/{})", remaining, limit);
					return LOW_REMAINING_DELAY_MS;
				} else if (resetTime != null && limit != null) {
					// Calculate optimal delay to distribute remaining requests over time until reset
					long currentTime = System.currentTimeMillis() / 1000; // Convert to seconds
					long timeUntilReset = resetTime - currentTime;
					
					if (timeUntilReset > 0 && remaining > 0) {
						// Calculate delay to spread requests evenly over remaining time
						long optimalDelay = (timeUntilReset * 1000) / remaining; // Convert back to milliseconds
						
						// Cap the delay between reasonable bounds
						optimalDelay = Math.max(500, Math.min(optimalDelay, 10000)); // Between 0.5s and 10s
						
						logger.trace("Calculated optimal delay: {} ms (remaining: {}, time until reset: {} s)", 
							optimalDelay, remaining, timeUntilReset);
						return optimalDelay;
					}
				}
				
				// Default delay for normal operation
				return DEFAULT_DELAY_MS;
				
			} catch (NumberFormatException e) {
				logger.trace("Could not parse remaining rate limit value: {}", remainingHeader);
				return DEFAULT_DELAY_MS;
			}
		}
	}
	
	/**
	 * Client Request Filter to automatically add headers to all requests
	 */
	public static class AutoHeaderFilter implements jakarta.ws.rs.client.ClientRequestFilter {
		private final String userAgent;
		private final String authorization;
		
		public AutoHeaderFilter(String userAgent, String authorization) {
			this.userAgent = userAgent;
			this.authorization = authorization;
		}
		
		@Override
		public void filter(jakarta.ws.rs.client.ClientRequestContext requestContext) {
			// Add headers to every request automatically
			requestContext.getHeaders().add("User-Agent", userAgent);
			requestContext.getHeaders().add("Authorization", authorization);
		}
	}
	
	/**
	 * Creates a Jersey Client with automatic header injection and rate limiting
	 * @param baseUrl the base URL for the API
	 * @return configured Jersey Client instance with automatic headers and rate limiting
	 */
	protected Client getJerseyClientWithAutoHeaders(String baseUrl) {
		Client client = ClientBuilder.newClient();
		
		// Register Jackson feature for JSON serialization/deserialization
		client.register(JacksonFeature.class);
		
		// Configure timeouts
		client.property(ClientProperties.CONNECT_TIMEOUT, 30000);
		client.property(ClientProperties.READ_TIMEOUT, 60000);
		
		// Register features
		client.property(ClientProperties.SUPPRESS_HTTP_COMPLIANCE_VALIDATION, true);
		
		// Register the auto-header filter
		client.register(new AutoHeaderFilter(
			GUCBE_ZENODO_SOFTWARE_DEPOSIT, 
			"Bearer " + accessToken
		));
		
		// Register the rate limiting response filter
		client.register(new RateLimitResponseFilter());
		
		client.property("access_token", accessToken);

		return client;
	}
	
	/**
	 * Alternative method that returns both client and builder for manual resource management
	 * @param path the API endpoint path
	 * @return ClientAndBuilder wrapper containing both client and builder
	 */
	protected ClientAndBuilder getZenodoRequestBuilderWithClient(String path) {
		Client client = getJerseyClientWithAutoHeaders(zenodoBaseURL.toString());
		WebTarget webTarget = client.target(zenodoBaseURL.toString()).path(path);
		Builder builder = webTarget.request(MediaType.APPLICATION_JSON)
			.header("Accept", MediaType.APPLICATION_JSON); // Set default headers
		return new ClientAndBuilder(client, builder);
	}
	
	/**
	 * Wrapper class to hold both client and builder for proper resource management
	 */
	protected static class ClientAndBuilder implements AutoCloseable {
		public final Client client;
		public final Builder builder;
		
		public ClientAndBuilder(Client client, Builder builder) {
			this.client = client;
			this.builder = builder;
		}
		
		@Override
		public void close() {
			if (client != null) {
				client.close();
			}
		}
	}
	
	protected void updateMetadata() throws Exception {
		logger.debug("Going to update metadata of {}",  softwareArtifactMetadata.getTitle());

		ObjectNode metadata = generateMetadata();
		String id = getZenodoIDFromDOIURL(softwareArtifactMetadata.getVersionDOIURL());
		String path = DEPOSITION_PATH.replace(":id", id);
		
		try (ClientAndBuilder clientAndBuilder = getZenodoRequestBuilderWithClient(path)) {
			clientAndBuilder.builder.header("Content-Type", MediaType.APPLICATION_JSON);
			Response response = clientAndBuilder.builder.put(Entity.json(metadata));
			getResponse(response);
		}
	}
	
	protected void commitVersion() throws Exception {
		logger.debug("Going to confirm the version of {}", softwareArtifactMetadata.getTitle());
		String id = getZenodoIDFromDOIURL(softwareArtifactMetadata.getVersionDOIURL());
		String path = DEPOSTION_PUBLISH_PATH.replace(":id", id);
		
		try (ClientAndBuilder clientAndBuilder = getZenodoRequestBuilderWithClient(path)) {
			clientAndBuilder.builder.header("Content-Type", MediaType.APPLICATION_JSON);
			Response response = clientAndBuilder.builder.post(Entity.text(""));
			getResponse(response);
		}
	}
	
	protected void finalizeProcessing() throws Exception {
		List<File> files = new ArrayList<>(); 
		for(SoftwareArtifactFile svf : softwareArtifactMetadata.getFiles()) {
			File file = svf.downloadFile();
			files.add(file);
			Thread.sleep(TimeUnit.SECONDS.toMillis(1));
		}
		
		try {
			//Add depositionFiles
			addFilesToDeposition(files);
			
			//Update deposit metadata
			updateMetadata();
			
			// Publish the version
			commitVersion();
		}finally {
			for(File file : files) {
				logger.trace("Going to delete file {}", file.getAbsolutePath());
				if(!file.exists()) {
		        	throw new RuntimeException(file.getAbsolutePath() + " does not exist");
		        }
				try {
					int i = 0;
					while(!file.delete() && i<10) {
						int millis = 100;
						logger.warn("File {} not deleted at the attemp {}. Retrying in {} milliseconds.", file.getAbsolutePath(), i+1, millis);
						++i;
						Thread.sleep(millis);
					}
					
					if(i==10) {
						logger.warn("After {} attemps the file {} was not deleted. Trying using deleteOnExit().", i, file.getAbsolutePath());
						file.deleteOnExit();
					}
				}catch (Exception e) {
					logger.error("Unable to delete file {}", file.getAbsolutePath());
				}
			}
		}
	}
		
	protected String createZenodoDOIURLFromID(String id) throws MalformedURLException {
		return doiBaseURL + id;
	}
	
	public void create() throws Exception {
		logger.info("Going to create new deposition for {}", softwareArtifactMetadata.getTitle());
		ObjectNode metadata = generateMetadata();
		
		try (ClientAndBuilder clientAndBuilder = getZenodoRequestBuilderWithClient(DEPOSITIONS_COLLECTION_PATH)) {
			clientAndBuilder.builder.header("Content-Type", MediaType.APPLICATION_JSON);
			Response response = clientAndBuilder.builder.post(Entity.json(metadata));
			
			JsonNode responseJson = getResponse(response);
			
			String conceptDOIURL = createZenodoDOIURLFromID(responseJson.get("conceptrecid").asText());
			softwareArtifactMetadata.setConceptDOIURL(conceptDOIURL);
			String versionDOIURL = createZenodoDOIURLFromID(responseJson.get("id").asText());
			softwareArtifactMetadata.setVersionDOIURL(versionDOIURL);
		}
		finalizeProcessing();
	}
	
	private ArrayNode getAuthors(){
		ArrayNode authors = softwareArtifactMetadata.getAuthors().deepCopy();
		return authors;
	}
	
	private String getDescription() {
		StringBuffer stringBuffer = new StringBuffer();
		stringBuffer.append(softwareArtifactMetadata.getAdditionalProperty(HTML_DESCRIPTION_CONFIG_FIELD_NAME).asText());
		
		if(processorConfig.getProperty(ADDITIONAL_HTML_DESCRIPTION_CONFIG_FIELD_NAME)!=null) {
			String additionalHTMLDescription = processorConfig.getProperty(ADDITIONAL_HTML_DESCRIPTION_CONFIG_FIELD_NAME).asText();
			stringBuffer.append(additionalHTMLDescription);
		}

		return stringBuffer.toString();
	}
	
	private ArrayNode getGrants(){
		ObjectMapper objectMapper = Utils.getObjectMapper();
		ArrayNode grants = objectMapper.createArrayNode();
		ArrayNode arrayNode = (ArrayNode) processorConfig.getProperty(SKIP_GRANTS_CONFIG_FIELD_NAME);
		Set<String> idToSkip = new HashSet<>();
		for(JsonNode idNode : arrayNode) {
			idToSkip.add(idNode.asText());
		}
		for(JsonNode g : softwareArtifactMetadata.getGrants()) {
			String id = g.get("id").asText();
			if(idToSkip.contains(id)) {
				continue;
			}
			ObjectNode grant = objectMapper.createObjectNode();
			grant.put("id", id);
			grants.add(grant);
		}
		return grants;
	}
	
	private ArrayNode getKeywords(){
		Set<String> keywords = softwareArtifactMetadata.getKeywords();
		ObjectMapper objectMapper = Utils.getObjectMapper();
		ArrayNode keywordsArrayNode = objectMapper.createArrayNode();
		for(String keyword : keywords) {
			keywordsArrayNode.add(keyword);
		}
		return keywordsArrayNode;
	}
	
	private ArrayNode getCommunities() {
		return (ArrayNode) softwareArtifactMetadata.getAdditionalProperty(COMMUNITIES_FIELD_NAME);
	}

	private String getLicense() {
		return softwareArtifactMetadata.getLicense().get("id").asText();
	}
	
	private String getDate() {
		return Utils.getDateAsString(softwareArtifactMetadata.getDate());
	}
	
	private ObjectNode generateMetadata() {
		ObjectMapper objectMapper = Utils.getObjectMapper();
		ObjectNode metadatWrapper = objectMapper.createObjectNode();
		
		ObjectNode metadata = objectMapper.createObjectNode();
		metadata.put("access_right", "open");
		metadata.put("upload_type", "software");
		metadata.replace("creators", getAuthors());
		metadata.put("description", getDescription());
		metadata.replace("communities", getCommunities());
		metadata.replace("grants", getGrants());
		metadata.replace("keywords", getKeywords());
		metadata.put("license", getLicense());
		metadata.put("publication_date", getDate());
		metadata.put("title", softwareArtifactMetadata.getTitle());
		metadata.put("version", softwareArtifactMetadata.getVersion());
		
		metadatWrapper.set(METADATA_FIELD_NAME, metadata);
		return metadatWrapper;
	}

	public void update() throws Exception {
		// Enable deposit edit 
		logger.debug("Going to unlock already submitted deposition {} to update it",  softwareArtifactMetadata.getTitle());

		String id = getZenodoIDFromDOIURL(softwareArtifactMetadata.getVersionDOIURL());
		String editPath = DEPOSTION_EDIT_PATH.replace(":id", id);
		
		ClientAndBuilder clientAndBuilder = getZenodoRequestBuilderWithClient(editPath);
		try {
			clientAndBuilder.builder.header("Accept", MediaType.APPLICATION_JSON);
			Response response = clientAndBuilder.builder.post(Entity.json(""));
			getResponse(response);
		} finally {
			clientAndBuilder.close();
		}
		
		//Update deposit metadata
		updateMetadata();
		
		// Publish the version
		commitVersion();
	}
	
	/**
	 * Remove previous depositionFiles
	 * @throws Exception
	 */
	protected void deletePreviousFiles(JsonNode jsonNode) throws Exception {
		logger.debug("Going to delete previous deposition files to properly publish {}", softwareArtifactMetadata.getTitle());
		ArrayNode files = (ArrayNode) jsonNode.get("files");
		for(int i=0; i<files.size(); i++) {
			ObjectNode file = (ObjectNode) files.get(i);
			String fileURLString = file.get("links").get("self").asText();
			
			// Extract path from the full URL
			String filePath = fileURLString.replace(zenodoBaseURL.toString(), "");
			
			ClientAndBuilder clientAndBuilder = getZenodoRequestBuilderWithClient(filePath);
			try {
				clientAndBuilder.builder.header("Accept", MediaType.APPLICATION_JSON);
				Response response = clientAndBuilder.builder.delete();
				getResponse(response);
			} finally {
				clientAndBuilder.close();
			}
		}
	}
	
	public void newVersion() throws Exception {
		
		// Reading Record using conceptID to get the latest published version
		String conceptDOIURL = softwareArtifactMetadata.getConceptDOIURL();
		String conceptID = getZenodoIDFromDOIURL(conceptDOIURL);
		String recordPath = RECORD_PATH.replace(":id", conceptID);
		
		logger.debug("Going to read record using conceptID {} to get the latest published version (should be {}) to be able to publish {}", 
			conceptID, softwareArtifactMetadata.getPrevious().getTitle(), softwareArtifactMetadata.getTitle());
		
		ClientAndBuilder clientAndBuilder = getZenodoRequestBuilderWithClient(recordPath);
		JsonNode jsonNode;
		try {
			clientAndBuilder.builder.header("Content-Type", MediaType.APPLICATION_JSON);
			clientAndBuilder.builder.header("Accept", MediaType.APPLICATION_JSON);
			Response response = clientAndBuilder.builder.get();
			jsonNode = getResponse(response);
		} finally {
			clientAndBuilder.close();
		}
		
		/*
		 * Comparing obtained latestDOI and its declared version with the previuos version DOI and its declared version.
		 * If they differs the configuration is not up to date and must be fixed 
		 * this should avoid errors on softwareConcept.
		 */
		String latestVersionDOI = jsonNode.get("links").get("doi").asText();
		String previousVersionDOI = softwareArtifactMetadata.getPrevious().getVersionDOIURL().toString(); 
		if(previousVersionDOI.compareTo(latestVersionDOI)!=0) {
			logger.error("Zenodo obtained latest DOI {} != {} DOI from previous version", latestVersionDOI, previousVersionDOI);
			throw new RuntimeException("It seems that your json is not up to date with Zenodo.");
		}

		String latestVersionVersion = jsonNode.get("metadata").get("version").asText();
		String previousVersionVersion = softwareArtifactMetadata.getPrevious().getVersion().toString();
		if(latestVersionVersion.compareTo(previousVersionVersion)!=0) {
			logger.error("Zenodo obtained latest Version {} != {} Version from previous version", latestVersionVersion, previousVersionVersion);
			throw new RuntimeException("It seems that your json is not up to date with Zenodo.");
		}
		
		logger.debug("Going to create the new version {} starting from latest deposited version {}", 
			softwareArtifactMetadata.getTitle(), softwareArtifactMetadata.getPrevious().getTitle());
		// Creating new version from latest deposited version
		String latestID = getZenodoIDFromDOIURL(latestVersionDOI);
		String newVersionPath = DEPOSTION_NEW_VERSION_PATH.replace(":id", latestID);
		
		clientAndBuilder = getZenodoRequestBuilderWithClient(newVersionPath);
		try {
			clientAndBuilder.builder.header("Content-Type", MediaType.APPLICATION_JSON);
			clientAndBuilder.builder.header("Accept", MediaType.APPLICATION_JSON);
			Response response = clientAndBuilder.builder.post(Entity.json(""));
			jsonNode = null;
			jsonNode = getResponse(response);
		} finally {
			clientAndBuilder.close();
		}
		
		// Getting draft new Version ID
		String newVersionId = jsonNode.get("id").toString();
		
		softwareArtifactMetadata.setVersionDOIURL(doiBaseURL + newVersionId);
		
		if(jsonNode!=null){
			// Remove previous depositionFiles
			deletePreviousFiles(jsonNode);
		}

		finalizeProcessing();
	}
	
	protected String getConfig(String propertyName) throws Exception {
		String conf = null;
		JsonNode node = processorConfig.getProperty(propertyName);
		if(node == null || node.getNodeType()==JsonNodeType.NULL) {
			conf = Config.getProperties().getProperty(propertyName);
		}
		if(conf==null) {
			throw new Exception("No configuration for '" + propertyName + "' property found.");
		}
		return conf;
	}
	
	protected void getZenodoConnectionConfig() throws Exception {
		this.zenodoBaseURL = new URL(getConfig("zenodo_base_url"));
		this.accessToken = getConfig("zenodo_access_token");
		this.doiBaseURL = getConfig("doi_base_url");
	}
	
	@Override
	public void export() throws Exception {
		if(first) {
			File exportFile = super.getOutputFile();
			if(exportFile.exists()) {
				exportFile.delete();
			}
			exportFile.createNewFile();
		}
		
		getZenodoConnectionConfig();
		
		String title = softwareArtifactMetadata.getTitle();
		
		ElaborationType elaborationType = processorConfig.getElaborationType();
		
		if(elaborationType==ElaborationType.NONE) {
			logger.info("Zenodo Deposit is disabled for {}.",title);
			return;
		}
		
		if(softwareArtifactMetadata.getVersionDOIURL()!=null) {
			
			softwareArtifactMetadata.setNewDeposition(false);
			
			if(elaborationType==ElaborationType.ALL || 
					elaborationType==ElaborationType.UPDATE_ONLY) {
				logger.info("Going to update {}.",title);
				update();
			}else {
				logger.info("{} has been already deposited.", title);
			}
			
		}else {
			if(elaborationType==ElaborationType.ALL || 
					elaborationType==ElaborationType.NEW) {
				
				softwareArtifactMetadata.setNewDeposition(true);
				
				if(softwareArtifactMetadata.getConceptDOIURL()==null) {
					logger.info("Going to create {}", title);
					create();
				}else {
					logger.info("Going to create new version for {}", title);
					newVersion();				
				}
			}
		}
		
	}

	protected ObjectNode getObjectNode() throws Exception {
		ObjectMapper objectMapper = Utils.getObjectMapper();
		ObjectNode toBeExported = objectMapper.createObjectNode();
		toBeExported.replace(AnalyserFactory.CONFIGURATION_PROPERTY_NAME, globalConfig.getOriginalJson().deepCopy());
		ArrayNode array = objectMapper.createArrayNode();
		SoftwareArtifactMetadata previous = softwareArtifactMetadata;
		boolean firstNode = true;
		while(previous!=null){
			ObjectNode node = previous.getOriginalJson().deepCopy();
			node.put(SoftwareArtifactMetadata.CONCEPT_DOI_URL_PROPERTY_NAME, previous.getConceptDOIURL());
			if(firstNode) {
				toBeExported.put(SoftwareArtifactMetadata.CONCEPT_DOI_URL_PROPERTY_NAME, previous.getConceptDOIURL());
				firstNode = false;
			}
			node.put(SoftwareArtifactMetadata.VERSION_DOI_URL_PROPERTY_NAME, previous.getVersionDOIURL());
			array.insert(0, node);
			previous = previous.getPrevious();
		}
		toBeExported.replace(AnalyserFactory.ARTIFACTS_PROPERTY_NAME, array);
		return toBeExported;
	}
	
	
	protected void writeObjectNodeToFile(ObjectNode toBeExported, File file) throws Exception {
		ObjectMapper objectMapper = Utils.getObjectMapper();
		objectMapper.writeValue(file, toBeExported);
	}
	
	@Override
	public File getOutputFile() throws Exception {
		File exportFile = super.getOutputFile();
		if(last) {
			ObjectNode toBeExported = getObjectNode();
			writeObjectNodeToFile(toBeExported, exportFile);
		}
		return exportFile;
	}

}