/**
 * 
 */
package org.gcube.vremanagement.executor.persistence;

import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.UUID;

import org.gcube.vremanagement.executor.plugin.PluginState;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author Luca Frosini (ISTI - CNR) http://www.lucafrosini.com/
 */
public class JDBCPersistenceConnector extends PersistenceConnector {
	
	/**
	 * Logger
	 */
	private static Logger logger = LoggerFactory.getLogger(JDBCPersistenceConnector.class);
	
	protected Connection connection;
	
	public static final String driverClass = "org.h2.Driver";
	public static final String jdbcConnection = "jdbc:h2:";
	public static final String username = "username";
	public static final String password = "password";
	public static final String dbName = "executor";
	
	public final static String PLUGIN_INSTANCE_EVOLUTION_TABLE = "PluginInstanceEvolution";
	public final static String PLUGIN_INSTANCE_EVOLUTION_TABLE_ID_FIELD = "id";
	public final static String PLUGIN_INSTANCE_EVOLUTION_TABLE_UUID_FIELD = "uuid";
	public final static String PLUGIN_INSTANCE_EVOLUTION_TABLE_PLUGIN_NAME_FIELD = "pluginName";
	public final static String PLUGIN_INSTANCE_EVOLUTION_TABLE_TIMESTAMP_FIELD = "timestamp";
	public final static String PLUGIN_INSTANCE_EVOLUTION_TABLE_STATE_FIELD = "state";
	
	public final static String CATALINA_HOME = "CATALINA_HOME";
	
	@SuppressWarnings("unused")
	private JDBCPersistenceConnector(){
		super();
	}
	
	/**
	 * This constructor is used to provide a location where creating persistence
	 * files
	 * @param location directory where creating the DB file
	 * @throws Exception if fails
	 */
	public JDBCPersistenceConnector(String location) throws Exception {
		super(location);
		
		try {
			Class.forName(driverClass);
			logger.debug(String.format("JDBC Driver (%s) Registered!", driverClass));
		} catch (ClassNotFoundException e) {
			logger.error(String.format("Driver Class (%s) NOT available : %s", driverClass, e));
			throw e;
		} 
	 
		String dbFilePath = String.format("%s/%s", location, dbName);
		
		String jdbcURL = String.format("%s%s", jdbcConnection, dbFilePath);
		logger.debug(String.format("DB URL : %s", jdbcURL));
		
		try{
			connection = DriverManager.getConnection(jdbcURL, username, password);
		} catch(SQLException e) {
			logger.error(String.format("Unable to connect to JDBC URL : %s", jdbcURL));
		}
		
		try {
			connection.setAutoCommit(false); // transaction block start
			
			String createTable = String.format(
					"CREATE TABLE IF NOT EXISTS `%s` ("
					+ "`%s` INT PRIMARY KEY AUTO_INCREMENT NOT NULL,"
					+ "`%s` VARCHAR(36) NOT NULL,"
					+ "`%s` VARCHAR(255) NOT NULL,"
					+ "`%s` BIGINT NOT NULL,"
					+ "`%s` INT NOT NULL);", 
					PLUGIN_INSTANCE_EVOLUTION_TABLE, 
					PLUGIN_INSTANCE_EVOLUTION_TABLE_ID_FIELD,
					PLUGIN_INSTANCE_EVOLUTION_TABLE_UUID_FIELD,
					PLUGIN_INSTANCE_EVOLUTION_TABLE_PLUGIN_NAME_FIELD,
					PLUGIN_INSTANCE_EVOLUTION_TABLE_TIMESTAMP_FIELD,
					PLUGIN_INSTANCE_EVOLUTION_TABLE_STATE_FIELD);
			
			logger.info(String.format("Creating %s Table : %s.", 
					PLUGIN_INSTANCE_EVOLUTION_TABLE, createTable));

			Statement createTableStatement = connection.createStatement();
			createTableStatement.execute(createTable);
			connection.commit(); //transaction block end
			
		} catch(Exception e){
			logger.error("Error while creating DB", e);
			throw e;
		}
		
	}
	
	/**
	 * @return the connection to the DB
	 */
	public Connection getConnection() {
		return connection;
	}

	/**{@inheritDoc} */
	@Override
	public PluginState getPluginInstanceState(UUID uuid) throws Exception {
		String query = String.format(
				"SELECT `%s`,`%s` FROM `%s` WHERE `%s`=? ORDER BY `%s` DESC LIMIT 1", 
				PLUGIN_INSTANCE_EVOLUTION_TABLE_TIMESTAMP_FIELD, 
				PLUGIN_INSTANCE_EVOLUTION_TABLE_STATE_FIELD,
				PLUGIN_INSTANCE_EVOLUTION_TABLE, 
				PLUGIN_INSTANCE_EVOLUTION_TABLE_UUID_FIELD,
				PLUGIN_INSTANCE_EVOLUTION_TABLE_TIMESTAMP_FIELD);
		logger.debug(query);
		PreparedStatement ps = connection.prepareStatement(query);
		ps.setString(1, uuid.toString());
		ResultSet resultSet = ps.executeQuery();
		resultSet.first();
		int stateOrdinal = resultSet.getInt(PLUGIN_INSTANCE_EVOLUTION_TABLE_STATE_FIELD);
		return PluginState.values()[stateOrdinal];
	}

	/**{@inheritDoc} */
	@Override
	public void close() throws Exception {
		connection.close();
	}

}
