/**
 * 
 */
package org.gcube.vremanagement.executor.api.types;

import java.util.Map;

import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.adapters.XmlJavaTypeAdapter;

import org.gcube.vremanagement.executor.api.types.adapter.MapAdapter;
import org.gcube.vremanagement.executor.utils.MapCompare;

/**
 * @author Luca Frosini (ISTI - CNR) http://www.lucafrosini.com/
 */
@XmlRootElement()
@XmlAccessorType(XmlAccessType.FIELD)
public class LaunchParameter implements Comparable<LaunchParameter>{

	@XmlElement
	/**
	 * The name of the plugin to launch
	 */
	protected String pluginName;
	
	@XmlJavaTypeAdapter(MapAdapter.class)
	/**
	 * The plugin capabilities which has to be satisfied to launch the
	 * plugin instance execution. The SmartExectuor Service check that this 
	 * capabilities match the capabilities declared from the discovered plugin.
	 * If the capabilities does not match the service will not launch the 
	 * execution.
	 */
	protected Map<String, String> pluginCapabilities;
	
	@XmlJavaTypeAdapter(MapAdapter.class)
	/**
	 * Inputs to provide to the plugin instance which have to be executed.
	 */
	protected Map<String, Object> inputs;

	@XmlElement
	/**
	 * Scheduling parameters. See {#Scheduling} fro futher details
	 */
	protected Scheduling scheduling;
	
	@XmlElement
	/**
	 * Used only for scheduled tasks. Indicate if the task has to be persisted
	 * so that if the SmartExectuor Service instance die, another one take in 
	 * charge that execution.
	 */
	protected boolean persist;

	protected LaunchParameter(){}
	
	public LaunchParameter(String pluginName, Map<String, Object> inputs) {
		this(pluginName, null, inputs, null, false);
	}
	
	public LaunchParameter(String pluginName, Map<String, String> pluginCapabilities, Map<String, Object> inputs) {
		this(pluginName, pluginCapabilities, inputs, null, false);
	}
	
	public LaunchParameter(String pluginName, Map<String, Object> inputs, Scheduling scheduling) {
		this(pluginName, null, inputs, scheduling, false);
	}
	
	public LaunchParameter(String pluginName, Map<String, String> pluginCapabilities, Map<String, Object> inputs, Scheduling scheduling) {
		this(pluginName, pluginCapabilities, inputs, scheduling, false);
	}
	
	public LaunchParameter(String pluginName, Map<String, Object> inputs, Scheduling scheduling, boolean persist) {
		this(pluginName, null, inputs, scheduling, persist);
	}
	
	public LaunchParameter(String pluginName, Map<String, String> pluginCapabilities, Map<String, Object> inputs, Scheduling scheduling, boolean persist) {
		this.pluginName = pluginName;
		this.pluginCapabilities = pluginCapabilities;
		this.inputs = inputs;
		this.scheduling = scheduling;
		this.persist = persist;
	}
	
	/**
	 * @return the name
	 */
	@Deprecated
	public String getName() {
		return getPluginName();
	}
	
	/**
	 * @return the name
	 */
	public String getPluginName() {
		return pluginName;
	}

	/**
	 * @return the pluginCapabilities
	 */
	public Map<String, String> getPluginCapabilities() {
		return pluginCapabilities;
	}

	/**
	 * @param pluginCapabilities the pluginCapabilities to set
	 */
	public void setPluginCapabilities(Map<String, String> pluginCapabilities) {
		this.pluginCapabilities = pluginCapabilities;
	}


	/**
	 * @return the inputs
	 */
	public Map<String, Object> getInputs() {
		return inputs;
	}
	
	
	/**
	 * @return the scheduling
	 */
	public Scheduling getScheduling() {
		return scheduling;
	}
	
	/**
	 * @param scheduling the scheduling
	 */
	public void setScheduling(Scheduling scheduling) {
		this.scheduling = scheduling;
	}
	
	@Override
	public String toString(){
		return String.format("%s : { Plugin : { Name: %s , Capabilites : %s}, Persist : %b, Scheduling : {%s}, Inputs : %s}", 
				this.getClass().getSimpleName(), pluginName, pluginCapabilities,
				persist, scheduling, inputs);
	}

	/**
	 * @return the persist
	 */
	public boolean isPersist() {
		return persist;
	}

	/**
	 * @param persist the persist to set
	 */
	public void setPersist(boolean persist) {
		this.persist = persist;
	}

	/** {@inheritDoc}	 */
	@Override
	public int compareTo(LaunchParameter launchParameter) {
		int compareResult = 0;
		
		compareResult = (new Boolean(persist)).compareTo(new Boolean(launchParameter.persist));
		if(compareResult!=0){
			return compareResult;
		}
		
		compareResult = pluginName.compareTo(launchParameter.pluginName);
		if(compareResult!=0){
			return compareResult;
		}
		
		compareResult = scheduling.compareTo(launchParameter.scheduling);
		if(compareResult!=0){
			return compareResult;
		}
		
		MapCompare<Map<String, Object>, String, Object> mapCompare = new MapCompare<>();
		return mapCompare.compareMaps(inputs, launchParameter.inputs);
	}


}
