package org.gcube.tools;

import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.util.ArrayList;

import org.apache.commons.io.FileUtils;
import org.gcube.spatial.data.geonetwork.GeoNetwork;
import org.gcube.spatial.data.geonetwork.GeoNetworkPublisher;
import org.gcube.spatial.data.geonetwork.LoginLevel;
import org.gcube.spatial.data.geonetwork.configuration.Configuration;
import org.gcube.spatial.data.geonetwork.model.faults.EncryptionException;
import org.gcube.spatial.data.geonetwork.model.faults.MissingConfigurationException;
import org.gcube.spatial.data.geonetwork.utils.ScopeUtils;

import lombok.extern.slf4j.Slf4j;

@Slf4j
public abstract class Publisher {

	private static final String TEMP_NAME="GN_PUBLISH";

	private String baseFolder=null;

	private boolean destroyTemp=false;

	
	public void execute(){
		log.debug("Starting publishing, baseFolder is : {} ",baseFolder);

		if(baseFolder==null) throw new RuntimeException("Base fodler is null");
		File baseFolderDir=new File(baseFolder);
		if(!baseFolderDir.exists()) throw new RuntimeException("Base folder not found");
		if(!baseFolderDir.canRead()) throw new RuntimeException("Cannot read base folder");
		if(!baseFolderDir.isDirectory()) throw new RuntimeException("Base folder is not a dir");

		File temp=new File(System.getProperty("java.io.tmpdir"),TEMP_NAME);

		if(temp.exists()){
			log.debug("Found temp folder "+temp.getAbsolutePath());
			if(destroyTemp){
				log.debug("Cleaning temp .. ");
				try {
					FileUtils.cleanDirectory(temp);
				} catch (IOException e) {
					throw new RuntimeException("Unable to clean temp ",e);
				}			
			}
		}else{
			log.debug("Creating temp folder .. ");
			temp.mkdirs();
		}

		log.debug("Getting baseFolder info ..");
		
		for(File firstLevel:baseFolderDir.listFiles(NOT_HIDDEN)){ 
			
			try{
				String scope=folderNameToScope(firstLevel.getName());
				log.debug("Found folder {}, assuming scope {} ",firstLevel.getName(),scope);
				TokenSetter.set(scope);
				ArrayList<String> parents=ScopeUtils.getParentScopes();
				log.debug("Initializing parent scope configuration, parents are : "+parents);
				String scopeString="/";
				for(String parent:parents){
					scopeString=scopeString+parent;
					TokenSetter.set(scopeString);
					GeoNetwork.get(getConfiguration());
					scopeString=scopeString+"/";
				}
				log.debug("Logging in default");
				TokenSetter.set(scope);
				GeoNetworkPublisher publisher=GeoNetwork.get(getConfiguration());
				publisher.login(LoginLevel.DEFAULT);
				File currentScopeTemp=new File(temp,firstLevel.getName());
				currentScopeTemp.mkdirs();
				
				File[] toPublishFiles=firstLevel.listFiles(XML_FILTER);
				log.debug("Found {} xml to publish in current scope {} ",toPublishFiles.length,scope);
				
				long currentCount=0l;
				
				for(File toPublish:toPublishFiles){
					File publishedCopy=null;
					try{
						log.debug("Checking {} ",toPublish.getName());
						//check if already published
						publishedCopy=new File(currentScopeTemp,toPublish.getName());
						if(publishedCopy.exists())
							log.debug("Found copy file {} ",publishedCopy.getAbsolutePath());
						else{
							publisher.insertMetadata(publisher.getCurrentUserConfiguration("dataset", "_none_"),toPublish);
							log.debug("Published file {}, copying..",toPublish.getAbsolutePath());
							FileUtils.copyFile(toPublish, publishedCopy);
							currentCount++;
						}						
					}catch(Throwable t){
						log.warn("Unable to publish {}.",toPublish.getAbsolutePath(),t);
						if(publishedCopy!=null) FileUtils.deleteQuietly(publishedCopy);
					}
				}
				File[] publishedFiles=currentScopeTemp.listFiles(XML_FILTER);
				int totalPublishedCount=publishedFiles.length;
				log.debug("Published {} files, total count in publishedFolder {}",currentCount,totalPublishedCount);
				
				if(totalPublishedCount!=toPublishFiles.length)throw 
				new Exception("Incoherent file number between "+firstLevel.getAbsolutePath()+" and "+currentScopeTemp.getAbsolutePath());
				
				
			}catch(Throwable t){
				log.warn("ERRORS WHILE TRYING TO HANDLE FOLDER {}.",firstLevel.getAbsolutePath(),t);				
			}
		}
		
	}

	public void setBaseFolder(String baseFolder) {
		this.baseFolder = baseFolder;
	}
	public void setDestroyTemp(boolean destroyTemp) {
		this.destroyTemp = destroyTemp;
	}
	
	public static String folderNameToScope(String folderName){
		return folderName.replace("_", "/");
	}

	
	public static final FileFilter NOT_HIDDEN=new FileFilter() {
		
		@Override
		public boolean accept(File pathname) {
			return !pathname.isHidden();
		}
	};
	
	public static final FileFilter XML_FILTER=new FileFilter(){
		@Override
		public boolean accept(File pathname) {			
			return !pathname.isHidden()&&pathname.getName().endsWith(".xml");
		}
	};
	
	public abstract Configuration getConfiguration();
		
	public static void main(String[] args){
		System.out.println("Instantiating publisher..");
		Publisher pub=new Publisher(){
			@Override
			public Configuration getConfiguration() {
				try{
					return new LocalResourceConfiguration(LocalResourceConfiguration.DEFAULT_FILENAME, LocalResourceConfiguration.DEFAULT_RES_CONFIG);
				}catch(Exception e){
					throw new RuntimeException(e);
				}
			}
		};
		pub.setBaseFolder("/home/fabio/workspaces/GNMigration/TEST");
		System.out.println("Launching..");
		pub.execute();
	}
}
