package org.gcube.deploytest.client;

import java.io.IOException;
import java.io.StringBufferInputStream;
import java.io.StringWriter;
import java.rmi.RemoteException;
import java.util.Properties;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.apache.axis.message.addressing.Address;
import org.apache.axis.message.addressing.EndpointReferenceType;
import org.gcube.common.core.contexts.GCUBERemotePortTypeContext;
import org.gcube.common.core.faults.GCUBEFault;
import org.gcube.common.core.resources.GCUBERunningInstance;
import org.gcube.common.core.resources.GCUBEService;
import org.gcube.common.core.scope.GCUBEScope;
import org.gcube.common.core.security.GCUBESecurityManagerImpl;
import org.gcube.common.core.utils.logging.GCUBEClientLog;
import org.gcube.common.core.utils.logging.GCUBELog;
import org.gcube.vremanagement.resourcemanager.stubs.binder.AddResourcesParameters;
import org.gcube.vremanagement.resourcemanager.stubs.binder.PackageItem;
import org.gcube.vremanagement.resourcemanager.stubs.binder.RemoveResourcesParameters;
import org.gcube.vremanagement.resourcemanager.stubs.binder.ResourceBinderPortType;
import org.gcube.vremanagement.resourcemanager.stubs.binder.ResourceItem;
import org.gcube.vremanagement.resourcemanager.stubs.binder.ResourceList;
import org.gcube.vremanagement.resourcemanager.stubs.binder.SoftwareList;
import org.gcube.vremanagement.resourcemanager.stubs.binder.service.ResourceBinderServiceAddressingLocator;
import org.gcube.vremanagement.resourcemanager.stubs.reporting.ReportingPortType;
import org.gcube.vremanagement.resourcemanager.stubs.reporting.service.ReportingServiceAddressingLocator;
import org.gcube.vremanagement.resourcemanager.stubs.scontroller.ScopeControllerPortType;
import org.gcube.vremanagement.resourcemanager.stubs.scontroller.service.ScopeControllerServiceAddressingLocator;
import org.w3c.dom.Document;
/**
 * 
 * Deploy class
 * @author Andrea Manzi (CERN)
 *
 */
public class Deploy {

	protected static Properties resources = new Properties();

	public static String SCOPE = null;

	public static GCUBELog log;

	private long deploymentWaitingTime = 900*1000;

	private static QueryInformationSystem isClient = null;

	private static QueryDeploymentInformation deployInfo = null;




	/**
	 * base constructor
	 */
	public Deploy (String scope) {
		new Deploy(scope,deploymentWaitingTime);
	} 

	/**
	 * base constructor
	 */
	public Deploy (String scope,long deploymentWaitingTime) {

		try {
			resources.load(Deploy.class.getResourceAsStream("/DeployLogger.properties"));
			SCOPE =scope;
			this.deploymentWaitingTime=deploymentWaitingTime;

			isClient = null;

			try {
				isClient = new QueryInformationSystem();
			} catch (Exception e) {

				e.printStackTrace();
			}

			deployInfo = new QueryDeploymentInformation();

		} catch (IOException e) {

			e.printStackTrace();
		}

		log = new GCUBEClientLog(Deploy.class,resources);
	} 

	/**
	 * local logger 
	 * @param logString
	 */
	public static void logInfo(String logString)
	{
		log.info(logString);
	}

	/**
	 * local logger
	 * @param logString
	 */
	public static void logDebug(String logString)
	{
		log.debug(logString);
	}

	/**
	 * local logger

	 * 
	 * @param logString
	 */
	public static void logError(String logString)
	{
		log.error(logString);
	}

	/**
	 * local logger

	 * 
	 * @param logString
	 */
	public static void logError(String logString,Exception e)
	{
		log.error(logString,e);
	}

	/**
	 * Create a ServiceInfo object given the Etics testName
	 * @param testName the Etics test name
	 * @return a Service Info 
	 */
	public static ServiceInfo getServiceNameAndClass(String testName) {
		return new ServiceInfo(testName.substring(testName.indexOf("_") + 1,testName.lastIndexOf("_")),
				testName.substring(testName.lastIndexOf("_") + 1));

	}

	/**
	 * Create the ResourceManager input for Deployment
	 * 
	 * @param testName the Etics test name
	 * @param GHNId the target GHN id
	 * @return the VREManager Input
	 */
	public static String prepareDeploymentServiceTemplate(String serviceName, String serviceClass,String ghnName,String version) {
		return "numOfServicesToAdd=1\n" +
				"numOfResourcesToAdd=0\n" +
				"targetScope="+SCOPE+"\n" +
				"callerScope="+SCOPE+"\n" +
				"service.1.name="+serviceName+"\n" +
				"service.1.class="+serviceClass+"\n" +
				"service.1.version="+version+"\n" +
				"service.1.GHN="+ghnName+"\n";
	}

	/**
	 * Create the ResourceManager input for Deployment
	 * 
	 * @param testName the Etics test name
	 * @param GHNId the target GHN id
	 * @return the VREManager Input
	 */
	public static String prepareUnDeploymentServiceTemplate(String serviceName, String serviceClass,String version,String ghnName) {
		return "numOfServicesToRemove=1\n" +
				"numOfResourcesToRemove=0\n" +
				"targetScope="+SCOPE+"\n" +
				"callerScope="+SCOPE+"\n" +
				"service.1.name="+serviceName+"\n" +
				"service.1.class="+serviceClass+"\n" +
				"service.1.version="+version+"\n"+
				"service.1.GHN="+ghnName+"\n";
	}

	

	/**
	 * Check if the given SA contains a WSRF Service
	 * @param service the Service profile related to the SA
	 * @return true/false
	 */
	public boolean isWSRFService(GCUBEService service) {
		for ( org.gcube.common.core.resources.service.Package pack : service.getPackages())
		{
			if (pack instanceof org.gcube.common.core.resources.service.MainPackage)
				return true;
		}
		return false;
	}


	/**
	 * Contacts the VREManager to get a deployment report
	 * @param vreGhnName the VREManager host name
	 * @param vreGhnPort the VREManager host port
	 * @param scope the VREManager scope
	 * @param deploymentId the related deployment id
	 * @return the Deployment report
	 */
	public String getDeploymentReport(String vreGhnName,String vreGhnPort, String scope,String deploymentId) {

		EndpointReferenceType endpoint = new EndpointReferenceType();
		ReportingPortType pt;
		String report = null;
		try {
			endpoint.setAddress(new Address("http://"+ vreGhnName+":"+ vreGhnPort +"/wsrf/services/gcube/vremanagement/resourcemanager/reporting"));


			GCUBESecurityManagerImpl managerSec = new GCUBESecurityManagerImpl() {  
				public boolean isSecurityEnabled() {return false;}};
				pt = GCUBERemotePortTypeContext.getProxy(new ReportingServiceAddressingLocator().getReportingPortTypePort(endpoint), 
						GCUBEScope.getScope(scope),managerSec);

				report = pt.getReport(deploymentId);
				log.info("REPORT");
				log.debug("************");

				// Get DOM Parser Factory
				DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance( );

				DocumentBuilder builder = factory.newDocumentBuilder( );
				Document doc = builder.parse(new StringBufferInputStream(report));
				TransformerFactory ft = TransformerFactory.newInstance();
				Transformer transformer = ft.newTransformer();
				transformer.setOutputProperty(OutputKeys.INDENT, "yes");
				transformer.setParameter("indent-number", new Integer(4));
				//initialize StreamResult with File object to save to file
				StreamResult result = new StreamResult(new StringWriter());
				transformer.transform(new DOMSource(doc), result);
				log.info(result.getWriter().toString());
				log.debug("************");
		} catch (Exception e) {
			e.printStackTrace();
		}
		return report;
	}


	/**
	 * Contact the ResourceManager to deploy one/more services
	 * 
	 * @param vreGhnName the ResourceManager host name
	 * @param vreGhnPort the ResourceManager port 
	 * @param serviceTemplate the service template to use
	 * @return the deployment id
	 * @throws GCUBEFault in case of error to contact the ResourceManager
	 * @throws RemoteException in case of error to contact the ResourceManager
	 */
	public String deployServices(String vreGhnName, String vreGhnPort,String serviceTemplate) throws GCUBEFault, RemoteException {

		Properties resources = new Properties();		
		EndpointReferenceType endpoint = new EndpointReferenceType();
		String deploymentId = null;

		try {
			resources.load(new StringBufferInputStream (serviceTemplate));
		} catch (IOException e1) {			
			e1.printStackTrace();
			Runtime.getRuntime().exit(1);

		}

		ResourceBinderPortType pt = null;
		AddResourcesParameters add = new AddResourcesParameters();
		try {
			endpoint.setAddress(new Address("http://"+vreGhnName+":"+ vreGhnPort +"/wsrf/services/gcube/vremanagement/resourcemanager/binder"));

			GCUBESecurityManagerImpl managerSec = new GCUBESecurityManagerImpl() {  
				public boolean isSecurityEnabled() {return false;}};
				pt = GCUBERemotePortTypeContext.getProxy(new ResourceBinderServiceAddressingLocator().getResourceBinderPortTypePort(endpoint), 
						GCUBEScope.getScope(resources.getProperty("callerScope")),240000,managerSec);

				// prepare the list of service to ask to deploy
				PackageItem[] packagelist = new PackageItem[new Integer(resources.getProperty("numOfServicesToAdd"))];
				for (int i = 1 ; i < (packagelist.length +1); i++) {
					packagelist[i-1] = new PackageItem();
					packagelist[i-1].setServiceClass(resources.getProperty("service." + i + ".class"));
					packagelist[i-1].setServiceName(resources.getProperty("service." + i + ".name"));
					packagelist[i-1].setServiceVersion(resources.getProperty("service." + i + ".version"));
					//packagelist[i-1].setPackageName("<packagename>");
					//packagelist[i-1].setPackageVersion("<packageversion>");
					if (resources.getProperty("service." + i + ".GHN") != null)
						packagelist[i-1].setTargetGHNName(resources.getProperty("service." + i + ".GHN"));
				}
				SoftwareList l = new SoftwareList();
				l.setSoftware(packagelist);

				//set the set of GHN, if any
				//if (resources.getProperty("GHNSet") != null) {
				//	String[] ghns = resources.getProperty("GHNSet").split(",");
				//	l.setGHN(ghns);
				//
				//}			
				add.setSoftware(l);			
				ResourceItem[] resourcelist = new ResourceItem[new Integer(resources.getProperty("numOfResourcesToAdd"))];
				for (int i = 1 ; i < (resourcelist.length +1); i++) {
					resourcelist[i-1] = new ResourceItem();
					resourcelist[i-1].setID(resources.getProperty("resource." + i + ".id"));
					resourcelist[i-1].setType(resources.getProperty("resource." + i + ".type"));
				}
				ResourceList r = new ResourceList();
				r.setResource(resourcelist);
				add.setResources(r);
				add.setTargetScope(resources.getProperty("targetScope"));


		} catch (Exception e) {
			e.printStackTrace();
			log.error(e);
			return null;
		}

		long time = System.currentTimeMillis();

		do {
			try {

				deploymentId = pt.addResources(add);
			}catch (Exception e) {
				e.printStackTrace();
				log.error(e);
			}
			if (deploymentId== null)
			{
				try {
					Thread.sleep(60000);
				} catch (InterruptedException e) {
					e.printStackTrace();
				}
			}
		} while (deploymentId== null && (System.currentTimeMillis() - time) < deploymentWaitingTime) ;

		if ((System.currentTimeMillis() - time) > deploymentWaitingTime)
		{
			log.error("Failed to contact the ResourceManager");
		}

		return deploymentId;
	}

	/**
	 * Contact the ResourceManager to undeploy one/more libraries
	 * 
	 * @param vreGhnName the ResourceManager host name
	 * @param vreGhnPort the ResourceManager port 
	 * @param serviceTemplate the service template to use
	 * @return the deployment id
	 * @throws GCUBEFault in case of error to contact the ResourceManager
	 * @throws RemoteException in case of error to contact the ResourceManager
	 */
	public String undeployLibraries(String vreGhnName, String vreGhnPort,String serviceTemplate) throws GCUBEFault, RemoteException {

		Properties resources = new Properties();		
		EndpointReferenceType endpoint = new EndpointReferenceType();
		String undeploymentId = null;

		try {
			resources.load(new StringBufferInputStream (serviceTemplate));
		} catch (IOException e1) {			
			e1.printStackTrace();
			Runtime.getRuntime().exit(1);

		}

		ResourceBinderPortType pt = null;
		RemoveResourcesParameters remove = new RemoveResourcesParameters();
		try {
			endpoint.setAddress(new Address("http://"+vreGhnName+":"+ vreGhnPort +"/wsrf/services/gcube/vremanagement/resourcemanager/binder"));

			GCUBESecurityManagerImpl managerSec = new GCUBESecurityManagerImpl() {  
				public boolean isSecurityEnabled() {return false;}};
				pt = GCUBERemotePortTypeContext.getProxy(new ResourceBinderServiceAddressingLocator().getResourceBinderPortTypePort(endpoint),
						GCUBEScope.getScope(resources.getProperty("callerScope")),240000,managerSec);

				// prepare the list of service to ask to deploy
				PackageItem[] servicelist = new PackageItem[new Integer(resources.getProperty("numOfServicesToRemove"))];
				for (int i = 1 ; i < (servicelist.length +1); i++) {
					servicelist[i-1] = new PackageItem();
					servicelist[i-1].setServiceClass(resources.getProperty("service." + i + ".class"));
					servicelist[i-1].setServiceName(resources.getProperty("service." + i + ".name"));
					servicelist[i-1].setServiceVersion(resources.getProperty("service." + i + ".version"));
					if (resources.getProperty("service." + i + ".GHN") != null)
						servicelist[i-1].setTargetGHNName(resources.getProperty("service." + i + ".GHN"));
				}

				SoftwareList l = new SoftwareList();
				l.setSoftware(servicelist);

				remove.setSoftware(l);			
				remove.setTargetScope(resources.getProperty("targetScope"));


		} catch (Exception e) {
			e.printStackTrace();
			log.error("Error creating RemoveLibrary request",e);
			return null;
		}

		long time = System.currentTimeMillis();

		do {
			try {

				undeploymentId = pt.removeResources(remove);
			}catch (Exception e) {
				e.printStackTrace();
				log.error(e);
			}
			if (undeploymentId== null)
			{
				try {
					Thread.sleep(60000);
				} catch (InterruptedException e) {
					e.printStackTrace();
				}
			}
		} while (undeploymentId== null && (System.currentTimeMillis() - time) < deploymentWaitingTime) ;

		if ((System.currentTimeMillis() - time) > deploymentWaitingTime)
		{
			log.error("Failed to contact the ResourceManager");
		}

		return undeploymentId;
	}

	/**
	 * Contact the ResourceManager to undeploy one/more services
	 * 
	 * @param vreGhnName the ResourceManager host name
	 * @param vreGhnPort the ResourceManager port 
	 * @param serviceTemplate the service template to use
	 * @return the deployment id
	 * @throws GCUBEFault in case of error to contact the ResourceManager
	 * @throws RemoteException in case of error to contact the ResourceManager
	 */
	public String removeVRE(String vreGhnName, String vreGhnPort,String scope) throws GCUBEFault, RemoteException {

		EndpointReferenceType endpoint = new EndpointReferenceType();
		String undeploymentId = null;

		ScopeControllerPortType pt = null;
		RemoveResourcesParameters remove = new RemoveResourcesParameters();
		try {
			endpoint.setAddress(new Address("http://"+vreGhnName+":"+ vreGhnPort +"/wsrf/services/gcube/vremanagement/resourcemanager/scopecontroller"));

			GCUBESecurityManagerImpl managerSec = new GCUBESecurityManagerImpl() {  
				public boolean isSecurityEnabled() {return false;}};
				pt = GCUBERemotePortTypeContext.getProxy(new ScopeControllerServiceAddressingLocator().getScopeControllerPortTypePort(endpoint),
						GCUBEScope.getScope(Deploy.SCOPE),240000,managerSec);

				//prepare the parameters


		} catch (Exception e) {
			e.printStackTrace();
			log.error("ERROR creating  ResourceManager request for undeployment",e);
			return null;
		}

		long time = System.currentTimeMillis();

		do {
			try {

				undeploymentId = pt.disposeScope(scope);

			}catch (Exception e) {
				e.printStackTrace();
				log.error(e);
			}
			if (undeploymentId== null)
			{
				try {
					Thread.sleep(60000);
				} catch (InterruptedException e) {
					e.printStackTrace();
				}
			}
		} while (undeploymentId== null && (System.currentTimeMillis() - time) < deploymentWaitingTime) ;

		if ((System.currentTimeMillis() - time) > deploymentWaitingTime)
		{
			log.error("Failed to contact the ResourceManager");
		}

		return undeploymentId;
	}


	/**
	 * Contact the ResourceManager to undeploy one/more services
	 * 
	 * @param vreGhnName the ResourceManager host name
	 * @param vreGhnPort the ResourceManager port 
	 * @param serviceTemplate the service template to use
	 * @return the deployment id
	 * @throws GCUBEFault in case of error to contact the ResourceManager
	 * @throws RemoteException in case of error to contact the ResourceManager
	 */
	
	public String undeployServices(String vreGhnName, String vreGhnPort, String serviceTemplate) throws GCUBEFault, RemoteException {

		Properties resources = new Properties();		
		EndpointReferenceType endpoint = new EndpointReferenceType();
		String undeploymentId = null;

		try {
			resources.load(new StringBufferInputStream (serviceTemplate));
		} catch (IOException e1) {			
			e1.printStackTrace();
			Runtime.getRuntime().exit(1);

		}

		ResourceBinderPortType pt = null;
		RemoveResourcesParameters remove = new RemoveResourcesParameters();
		try {
			endpoint.setAddress(new Address("http://"+vreGhnName+":"+ vreGhnPort +"/wsrf/services/gcube/vremanagement/resourcemanager/binder"));

			GCUBESecurityManagerImpl managerSec = new GCUBESecurityManagerImpl() {  
				public boolean isSecurityEnabled() {return false;}};
				pt = GCUBERemotePortTypeContext.getProxy(new ResourceBinderServiceAddressingLocator().getResourceBinderPortTypePort(endpoint),
						GCUBEScope.getScope(Deploy.SCOPE),240000,managerSec);

				//prepare the parameters

				PackageItem[] servicelist = new PackageItem[new Integer(resources.getProperty("numOfServicesToRemove"))];
				for (int i = 1 ; i < (servicelist.length +1); i++) {
					servicelist[i-1] = new PackageItem();
					servicelist[i-1].setServiceClass(resources.getProperty("service." + i + ".class"));
					servicelist[i-1].setServiceName(resources.getProperty("service." + i + ".name"));
					servicelist[i-1].setServiceVersion(resources.getProperty("service." + i + ".version"));
					if (resources.getProperty("service." + i + ".GHN") != null)
						servicelist[i-1].setTargetGHNName(resources.getProperty("service." + i + ".GHN"));
				}

				SoftwareList l = new SoftwareList();
				l.setSoftware(servicelist);

				remove.setSoftware(l);			
				remove.setTargetScope(resources.getProperty("targetScope"));


		} catch (Exception e) {
			e.printStackTrace();
			log.error("ERROR creating  ResourceManager request for undeployment",e);
			return null;
		}

		long time = System.currentTimeMillis();

		do {
			try {

				undeploymentId = pt.removeResources(remove);

			}catch (Exception e) {
				e.printStackTrace();
				log.error(e);
			}
			if (undeploymentId== null)
			{
				try {
					Thread.sleep(60000);
				} catch (InterruptedException e) {
					e.printStackTrace();
				}
			}
		} while (undeploymentId== null && (System.currentTimeMillis() - time) < deploymentWaitingTime) ;

		if ((System.currentTimeMillis() - time) > deploymentWaitingTime)
		{
			log.error("Failed to contact the ResourceManager");
		}

		return undeploymentId;
	}

	public void undeploy(GCUBEService service, String ghnID, String ghnName, String rmGHNName, String rmGHNPort, String unDeploymentId) throws Exception{

		log.info("Starting undeployment");
		boolean depResolution = false;
		long time;
		String unDeploymentReport = null;

		//testing undeployment
		if (service != null && this.isWSRFService(service)) {

			//getting RI ID
			log.debug("Getting RI ID from IS ("+service.getServiceName() +","+service.getServiceClass()+","+ghnID);
			String id = isClient.getRunningInstanceID(service.getServiceName(), service.getServiceClass(), ghnID); 
			log.debug("id="+id);

			try { 
				unDeploymentId = this.undeployServices(rmGHNName, rmGHNPort, 
						Deploy.prepareUnDeploymentServiceTemplate(service.getServiceName(),service.getServiceClass(), service.getVersion(), ghnName));
			} catch (GCUBEFault e) {
				e.printStackTrace();
				deployInfo.writeFile("FAILED\n");
				System.exit(1);
			} catch (RemoteException e) {
				e.printStackTrace();
				deployInfo.writeFile("FAILED\n");
				System.exit(1);

			} catch (Exception e) {
				e.printStackTrace();
				deployInfo.writeFile("FAILED\n");
				System.exit(1);
			}
		} else
		{
			try { 
				unDeploymentId = this.undeployLibraries(rmGHNName, rmGHNPort, 
						Deploy.prepareUnDeploymentServiceTemplate(service.getServiceName(),service.getServiceClass(), service.getVersion(), ghnName));
			} catch (GCUBEFault e) {
				e.printStackTrace();
				deployInfo.writeFile("FAILED\n");
				System.exit(1);
			} catch (RemoteException e) {
				e.printStackTrace();
				deployInfo.writeFile("FAILED\n");
				System.exit(1);

			} catch (Exception e) {
				e.printStackTrace();
				deployInfo.writeFile("FAILED\n");
				System.exit(1);
			}

		}


		time = System.currentTimeMillis();

		while ((depResolution == false) && 
				((System.currentTimeMillis() - time) < deploymentWaitingTime))
		{
			unDeploymentReport = this.getDeploymentReport(rmGHNName,rmGHNPort,SCOPE,unDeploymentId);

			deployInfo.parseXml(unDeploymentReport);

			depResolution = deployInfo.checkUnDeploymentInfo(isClient);

			if (depResolution == false)
				Thread.sleep(60000);
		}

		if ((System.currentTimeMillis() - time) > deploymentWaitingTime)
		{
			log.error("Unable to get a Final UnDeploymentReport after " + deploymentWaitingTime/1000+" sec");
			deployInfo.writeFile("FAILED\n");
			System.exit(1);
		}


	}


	public String deploy(GCUBEService service, String ghnID, String ghnName,String rmGHNName, String rmGHNPort) throws Exception{

		EndpointReferenceType epr = null;

		boolean depResolution = false;
		long time;
		String deploymentId = null;
		String deploymentReport = null;


		log.info("Starting the deployment of the SA ( " 
				+ service.getServiceClass() +
				" / " +service.getServiceName()+
				" ) on GHN " +  ghnName);

		try {
			deploymentId = this.deployServices(rmGHNName, rmGHNPort,
					Deploy.prepareDeploymentServiceTemplate(service.getServiceName(),service.getServiceClass(), ghnName,service.getVersion() ));

		} catch (GCUBEFault e) {
			e.printStackTrace();
			deployInfo.writeFile("FAILED\n");
			System.exit(1);
		} catch (RemoteException e) {
			e.printStackTrace();
			deployInfo.writeFile("FAILED\n");
			System.exit(1);
		}

		if (deploymentId == null)
		{
			log.error("Failed to deploy the service due to an error contacting the ResourceManager..");
			deployInfo.writeFile("FAILED\n");
			System.exit(1);
		}
		//query the informationSystem until the RI appears


		if (service != null && this.isWSRFService(service))
		{
			log.info("Waiting for the Running Instance publication");
			//wait until the IS publishes the profile
			time = System.currentTimeMillis();

			while(((epr = isClient.getEndpoint(service.getServiceName(),service.getServiceClass(), ghnID)) == null) && 
					((System.currentTimeMillis() - time) < deploymentWaitingTime))
			{	
				Thread.sleep(10000);
				log.debug("Waiting for the Running Instance publication");
			}

			if ((System.currentTimeMillis() - time) > deploymentWaitingTime)
				log.error("Running Instance not yet published on the IS after " + deploymentWaitingTime/1000+" sec");


			log.info("Running Instance retrieved from IS, waiting for deployment report");
			Thread.sleep(60000);
		} 	else Thread.sleep(120000);


		time = System.currentTimeMillis();

		while ((depResolution == false) && 
				((System.currentTimeMillis() - time) < deploymentWaitingTime))
		{
			try {

				deploymentReport = this.getDeploymentReport(rmGHNName,rmGHNPort,SCOPE,deploymentId);

				deployInfo.parseXml(deploymentReport);

				depResolution = deployInfo.checkDeploymentInfo(isClient);

			}
			catch (Exception e){
				e.printStackTrace();
				log.error("Unable to parse a  DeploymentReport",e);
			}
			if (depResolution == false)
				Thread.sleep(60000);
		}

		if ((System.currentTimeMillis() - time) > deploymentWaitingTime)
		{
			log.error("Unable to get a Final DeploymentReport after " + deploymentWaitingTime/1000+" sec");
			deployInfo.writeFile("FAILED\n");
			System.exit(1);
		}

		return deploymentId;
	}


	/**
	 * Main
	 * 
	 * @param args Client input parameters (testName, GHNID, ResourceManager host, ResourceManager port)
	 * @throws InterruptedException
	 */
	public static void main(String[] args) throws Exception{

		String scope = args[0];
		long deploymentWaitingTime = Long.parseLong(args[5])*1000;

		Deploy deploy = new Deploy(scope,deploymentWaitingTime);

		String testName = args[1];
		String ghnID = args[2];
		String rmGHNName = args[3];
		String rmGHNPort = args[4];

		String unDeploymentId = null;
		String ghnName = null;

		GCUBEService service = null;


		try {
			service = isClient.getServiceProfile(Deploy.getServiceNameAndClass(testName)); 
		} catch (Exception e) {
			e.printStackTrace();
			log.error("Error querying the IS to get SA information",e);
			deployInfo.writeFile("FAILED\n");
			System.exit(1); 
		}

		if (service == null){
			log.error("The Service Profile related to the SA "
					+ Deploy.getServiceNameAndClass(testName).get_serviceClass() +
					" / " + Deploy.getServiceNameAndClass(testName).get_serviceName() +", is not published on the IS\n"
					+ " The deployment cannot be performed\n");
			deployInfo.writeFile("FAILED\n");
			System.exit(1); 
		}

		
		log.debug("ghnID is: \""+ghnID+"\"");
		try {
			ghnName = isClient.getGHNName(ghnID); 
		} catch (Exception e) {
			e.printStackTrace();
			log.error("Error querying the IS to get GHN information",e);
			deployInfo.writeFile("FAILED\n");
			System.exit(1); 
		}



		unDeploymentId = deploy.deploy(service, ghnID, ghnName, rmGHNName, rmGHNPort);

		//undeployment
		deploy.undeploy(service, ghnID, ghnName, rmGHNName, rmGHNPort,unDeploymentId);


	}

}