package org.gcube.social_networking.social_networking_client_library;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.ws.rs.core.GenericType;

import org.apache.commons.lang.Validate;
import org.gcube.portal.databook.shared.Notification;
import org.gcube.portal.databook.shared.NotificationChannelType;
import org.gcube.portal.databook.shared.NotificationType;
import org.gcube.portal.databook.shared.ex.ColumnNameNotFoundException;
import org.gcube.portal.databook.shared.ex.NotificationChannelTypeNotFoundException;
import org.gcube.portal.databook.shared.ex.NotificationIDNotFoundException;
import org.gcube.portal.databook.shared.ex.NotificationTypeNotFoundException;
import org.gcube.social_networking.social_networking_client_library.utils.HttpClient;
import org.gcube.social_networking.socialnetworking.model.beans.JobNotificationBean;
import org.gcube.social_networking.socialnetworking.model.beans.catalogue.CatalogueEvent;
import org.gcube.social_networking.socialnetworking.model.beans.catalogue.CatalogueEventType;
import org.gcube.social_networking.socialnetworking.model.beans.workspace.WorkspaceEvent;
import org.gcube.social_networking.socialnetworking.model.beans.workspace.WorkspaceEventType;
import org.gcube.social_networking.socialnetworking.model.output.ResponseBean;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Notifications client.
 * @author Costantino Perciante at ISTI-CNR (costantino.perciante@isti.cnr.it)
 * @author Ahmed Ibrahim at ISTI-CNR (ahmed.ibrahim@isti.cnr.it)
 */
public class NotificationClient extends BaseClient{

	private static final String SUB_SERVICE_PATH = "2/notifications/";
	private static Logger logger = LoggerFactory.getLogger(NotificationClient.class);
	
	
	public NotificationClient() throws Exception {
		super(SUB_SERVICE_PATH);
	}
	
	/**
	 * Get range notifications
	 * @param from greater or equal to one
	 * @param quantity
	 * @return
	 */
	public List<Notification> getNotifications(int from, int quantity){
		
		Validate.isTrue(from >= 1, "From cannot be negative");
		Validate.isTrue(quantity >= 0, "Quantity cannot be negative");
		
		logger.debug("Request for getting notifications");
		String thisMethodSignature = "get-range-notifications";
		String request =  getServiceEndpoint()  + thisMethodSignature + "?from=" + from + "&quantity=" +quantity;
		return HttpClient.get(new GenericType<ResponseBean<ArrayList<Notification>>>(){}, request);
		
	}
	
	/**
	 * Notify job status
	 * @param notification
	 * @return
	 */
	public void sendJobNotification(JobNotificationBean notification){
		
		Validate.isTrue(notification != null, "Notification cannot be null");
		logger.debug("Request for getting notifications");
		String thisMethodSignature = "notify-job-status";
		String request =  getServiceEndpoint()  + thisMethodSignature;
		HttpClient.post(new GenericType<ResponseBean<String>>(){}, request, notification);
		
	}
	
	/**
	 * Notify a catalogue event, types available see {@link WorkspaceEventType}
	 * @param event an instance of {@link WorkspaceEvent}
	 */
	public void sendWorkspaceEvent(WorkspaceEvent event) {		
		Validate.isTrue(event != null, "WorkspaceEvent cannot be null");
		logger.debug("Request for sending workdspace notifications");
		String thisMethodSignature = "workspace";
		String request =  getServiceEndpoint()  + thisMethodSignature;
		HttpClient.post(new GenericType<ResponseBean<String>>(){}, request, event);
		
	}
	
	/**
	 * Notify a catalogue event, types availeble see {@link CatalogueEventType}
	 * @param event an instance of {@link CatalogueEvent}
	 */
	public void sendCatalogueEvent(CatalogueEvent event) {		
		Validate.isTrue(event != null, "CatalogueEvent cannot be null");
		logger.debug("Request for sending CatalogueEvent notifications");
		String thisMethodSignature = "catalogue";
		String request =  getServiceEndpoint()  + thisMethodSignature;
		HttpClient.post(new GenericType<ResponseBean<String>>(){}, request, event);
		
	}



	//library api

	public boolean saveNotificationLib(Notification n){
		Validate.isTrue(n != null, "Notification cannot be null");
		logger.debug("Request for saving notification");
		String thisMethodSignature = "save-notification-lib";
		String request =  getServiceEndpoint()  + thisMethodSignature;
		return HttpClient.post(new GenericType<ResponseBean<Boolean>>(){}, request, n);
	}
	public Notification readNotificationLib(String notificationid){
		logger.debug("Request for getting notification by id");
		String thisMethodSignature = "read-notification-lib";
		String request =  getServiceEndpoint()  + thisMethodSignature + "?notid=" + notificationid;
		return HttpClient.get(new GenericType<ResponseBean<Notification>>(){}, request);
	}
	public boolean setNotificationReadLib(String notificationid){
		Validate.isTrue(notificationid != null, "Notificationid cannot be null");
		logger.debug("Request for setting notification read");
		String thisMethodSignature = "set-notification-read-lib";
		String request =  getServiceEndpoint()  + thisMethodSignature + "?notid=" + notificationid;;
		return HttpClient.post(new GenericType<ResponseBean<Boolean>>(){}, request, notificationid);
	}

	public List<Notification> getAllNotificationByUserLib(String userid, int limit) {
		logger.debug("Request for getting notification by user");
		String thisMethodSignature = "get-all-notifications-user";
		String request =  getServiceEndpoint()  + thisMethodSignature + "?userid=" + userid + "&limit=" + limit;
		return HttpClient.get(new GenericType<ResponseBean<ArrayList<Notification>>>(){}, request);
	}

	public List<Notification> getUnreadNotificationsByUserLib(String userid) {
		logger.debug("Request for getting unread notification by user");
		String thisMethodSignature = "get-unread-notifications-user";
		String request =  getServiceEndpoint()  + thisMethodSignature + "?userid=" + userid;
		return HttpClient.get(new GenericType<ResponseBean<ArrayList<Notification>>>(){}, request);
	}

	public List<Notification> getRangeNotificationsByUserLib(String userid,int from, int quantity) {
		logger.debug("Request for getting range notification by user");
		String thisMethodSignature = "get-range-notifications-user";
		String request =  getServiceEndpoint()  + thisMethodSignature + "?userid=" + userid + "&from=" + from + "&quantity=" + quantity;
		return HttpClient.get(new GenericType<ResponseBean<ArrayList<Notification>>>(){}, request);
	}

	public boolean setAllNotificationReadByUserLib(String userid) {
		Validate.isTrue(userid != null, "userid cannot be null");
		logger.debug("Request for setting all notification read");
		String thisMethodSignature = "set-all-notification-read-lib";
		String request =  getServiceEndpoint()  + thisMethodSignature + "?userid=" + userid;;
		return HttpClient.post(new GenericType<ResponseBean<Boolean>>(){}, request, userid);
	}
	public boolean checkUnreadNotificationsLib(String userid) {
		Validate.isTrue(userid != null, "userid cannot be null");
		logger.debug("Request for check unread notifications");
		String thisMethodSignature = "check-unread-notification-lib";
		String request =  getServiceEndpoint()  + thisMethodSignature + "?userid=" + userid;;
		return HttpClient.post(new GenericType<ResponseBean<Boolean>>(){}, request, userid);
	}
	public boolean checkUnreadMessagesNotificationsLib(String userid) {
		Validate.isTrue(userid != null, "userid cannot be null");
		logger.debug("Request for check unread messages notifications");
		String thisMethodSignature = "check-unread-messages-notification-lib";
		String request =  getServiceEndpoint()  + thisMethodSignature + "?userid=" + userid;;
		return HttpClient.post(new GenericType<ResponseBean<Boolean>>(){}, request, userid);

	}

	public List<NotificationChannelType> getUserNotificationChannelsLib(String userid, NotificationType notificationType) {
		Validate.isTrue(userid != null, "userid cannot be null");
		logger.debug("Request for get user notification channels");
		String thisMethodSignature = "get-user-notification-channels-lib";
		String request =  getServiceEndpoint()  + thisMethodSignature + "?userid=" + userid;
		return HttpClient.post(new GenericType<ResponseBean<ArrayList<NotificationChannelType>>>(){}, request, notificationType);

	}

	public boolean setUserNotificationPreferencesLib(String userid, Map<NotificationType, NotificationChannelType[]> enabledChannels){
		Validate.isTrue(userid != null, "userid cannot be null");
		logger.debug("Request for set user notification preferences");
		String thisMethodSignature = "set-notification-preference-lib";
		String request =  getServiceEndpoint()  + thisMethodSignature + "?userid=" + userid;;
		return HttpClient.post(new GenericType<ResponseBean<Boolean>>(){}, request, enabledChannels);

	}

	public Map<NotificationType, NotificationChannelType[]> getUserNotificationPreferencesLib(String userid){
		Validate.isTrue(userid != null, "userid cannot be null");
		logger.debug("Request for getting notification preferences");
		String thisMethodSignature = "get-notification-preference-lib";
		String request =  getServiceEndpoint()  + thisMethodSignature + "?userid=" + userid;;
		return HttpClient.get(new GenericType<ResponseBean<HashMap<NotificationType, NotificationChannelType[]>>>(){}, request);
	}
}
