package org.gcube.smartgears.handler.resourceregistry.resourcemanager;

import java.net.URI;
import java.util.Arrays;
import java.util.List;
import java.util.Set;
import java.util.UUID;

import javax.servlet.ServletRegistration;

import org.gcube.informationsystem.model.impl.properties.HeaderImpl;
import org.gcube.informationsystem.model.impl.properties.PropagationConstraintImpl;
import org.gcube.informationsystem.model.reference.properties.Header;
import org.gcube.informationsystem.model.reference.properties.PropagationConstraint;
import org.gcube.informationsystem.model.reference.properties.PropagationConstraint.AddConstraint;
import org.gcube.informationsystem.model.reference.properties.PropagationConstraint.RemoveConstraint;
import org.gcube.informationsystem.resourceregistry.api.contexts.ContextCache;
import org.gcube.informationsystem.resourceregistry.api.exceptions.AvailableInAnotherContextException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.NotFoundException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.ResourceRegistryException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.context.ContextNotFoundException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.entity.resource.ResourceNotFoundException;
import org.gcube.informationsystem.resourceregistry.client.ResourceRegistryClient;
import org.gcube.informationsystem.resourceregistry.client.ResourceRegistryClientFactory;
import org.gcube.informationsystem.resourceregistry.publisher.ResourceRegistryPublisher;
import org.gcube.informationsystem.resourceregistry.publisher.ResourceRegistryPublisherFactory;
import org.gcube.resourcemanagement.model.impl.entities.facets.AccessPointFacetImpl;
import org.gcube.resourcemanagement.model.impl.entities.facets.ServiceStateFacetImpl;
import org.gcube.resourcemanagement.model.impl.entities.facets.SoftwareFacetImpl;
import org.gcube.resourcemanagement.model.impl.entities.resources.EServiceImpl;
import org.gcube.resourcemanagement.model.impl.properties.ValueSchemaImpl;
import org.gcube.resourcemanagement.model.impl.relations.consistsof.IsIdentifiedByImpl;
import org.gcube.resourcemanagement.model.impl.relations.isrelatedto.ActivatesImpl;
import org.gcube.resourcemanagement.model.reference.entities.facets.AccessPointFacet;
import org.gcube.resourcemanagement.model.reference.entities.facets.ServiceStateFacet;
import org.gcube.resourcemanagement.model.reference.entities.facets.SoftwareFacet;
import org.gcube.resourcemanagement.model.reference.entities.resources.EService;
import org.gcube.resourcemanagement.model.reference.entities.resources.HostingNode;
import org.gcube.resourcemanagement.model.reference.properties.ValueSchema;
import org.gcube.resourcemanagement.model.reference.relations.consistsof.IsIdentifiedBy;
import org.gcube.resourcemanagement.model.reference.relations.isrelatedto.Activates;
import org.gcube.smartgears.configuration.application.ApplicationConfiguration;
import org.gcube.smartgears.configuration.container.ContainerConfiguration;
import org.gcube.smartgears.context.application.ApplicationContext;
import org.gcube.smartgears.handler.resourceregistry.Constants;
import org.gcube.smartgears.handler.resourceregistry.ContextUtility;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author lucafrosini
 *
 */
/**
 * @author lucafrosini
 *
 */
public class EServiceManager {

	private static Logger logger = LoggerFactory.getLogger(HostingNodeManager.class);

	private static List<String> servletExcludes = Arrays.asList("default", "jsp");

	private ResourceRegistryPublisher resourceRegistryPublisher;

	// private Activates<HostingNode, EService> activates;
	private EService eService;
	private ServiceStateFacet serviceStateFacet;

	private ApplicationContext applicationContext;

	public EServiceManager(ApplicationContext applicationContext) {
		this.applicationContext = applicationContext;
		this.resourceRegistryPublisher = ResourceRegistryPublisherFactory.create();
	}

	public EService getEService() {
		return eService;
	}

	public void addToContext() throws ResourceNotFoundException, ContextNotFoundException, ResourceRegistryException {
		HostingNode hostingNode = applicationContext.container().properties()
				.lookup(Constants.HOSTING_NODE_MANAGER_PROPERTY).value(HostingNodeManager.class).getHostingNode();
		boolean added = resourceRegistryPublisher.addResourceToCurrentContext(hostingNode);
		if (added) {
			logger.info("{} successfully added to current context ({})", eService,
					ContextUtility.getCurrentContextName());
		} else {
			logger.error("Unable to add {} to current context ({})", eService, ContextUtility.getCurrentContextName());
		}
	}

	public void removeFromContext()
			throws ResourceNotFoundException, ContextNotFoundException, ResourceRegistryException {
		boolean removed = false;
		removed = resourceRegistryPublisher.removeResourceFromCurrentContext(eService);
		if (removed) {
			logger.info("{} successfully removed from current context ({})", eService,
					ContextUtility.getCurrentContextName());
		} else {
			logger.error("Unable to remove {} from current context ({})", eService,
					ContextUtility.getCurrentContextName());
		}
	}
	
	public void removeFromContext(UUID contextUUID)
			throws ResourceNotFoundException, ContextNotFoundException, ResourceRegistryException {
		boolean removed = false;
		String contextFullName = ContextCache.getInstance().getContextFullNameByUUID(contextUUID);
		removed = resourceRegistryPublisher.removeResourceFromContext(eService, contextUUID);
		if (removed) {
			logger.info("{} successfully removed from context ({})", eService, contextFullName);
		} else {
			logger.error("Unable to remove {} from current context ({})", eService, contextFullName);
		}
	}

	private String getBaseAddress() {
		ApplicationConfiguration configuration = applicationContext.configuration();
		ContainerConfiguration container = applicationContext.container().configuration();
		String baseAddress;
		if (configuration.proxied()) {
			String protocol = configuration.proxyAddress().protocol();
			String port = configuration.proxyAddress().port() != null ? ":" + configuration.proxyAddress().port() : "";

			baseAddress = String.format("%s://%s%s%s", protocol, configuration.proxyAddress().hostname(), port,
					applicationContext.application().getContextPath());
		} else {
			String protocol = container.protocol();
			int port = container.port();

			baseAddress = String.format("%s://%s:%d%s", protocol, container.hostname(), port,
					applicationContext.application().getContextPath());
		}
		return baseAddress;
	}

	public String getState() {
		return applicationContext.lifecycle().state().remoteForm().toLowerCase();
	}

	private EService instantiateEService() {
		logger.info("Creating {} for {}", EService.NAME, applicationContext.name());

		ApplicationConfiguration applicationConfiguration = applicationContext.configuration();
		String id = applicationContext.id();

		UUID uuid = UUID.fromString(id);
		EService eService = new EServiceImpl();
		Header header = new HeaderImpl(uuid);
		eService.setHeader(header);

		SoftwareFacet softwareFacet = new SoftwareFacetImpl();
		softwareFacet.setDescription(applicationConfiguration.description());
		softwareFacet.setGroup(applicationConfiguration.serviceClass());
		softwareFacet.setName(applicationConfiguration.name());
		softwareFacet.setVersion(applicationConfiguration.version());

		IsIdentifiedBy<EService, SoftwareFacet> isIdentifiedBy = new IsIdentifiedByImpl<EService, SoftwareFacet>(
				eService, softwareFacet);
		eService.addFacet(isIdentifiedBy);

		String baseAddress = getBaseAddress();
		for (ServletRegistration servlet : applicationContext.application().getServletRegistrations().values()) {
			if (!servletExcludes.contains(servlet.getName())) {
				for (String mapping : servlet.getMappings()) {

					String address = baseAddress
							+ (mapping.endsWith("*") ? mapping.substring(0, mapping.length() - 2) : mapping);

					AccessPointFacet accessPointFacet = new AccessPointFacetImpl();
					accessPointFacet.setEntryName(servlet.getName());
					accessPointFacet.setEndpoint(URI.create(address));
					ValueSchema valueSchema = new ValueSchemaImpl();
					valueSchema.setValue("gcube-token");

					accessPointFacet.setAuthorization(valueSchema);

					eService.addFacet(accessPointFacet);
				}
			}
		}

		serviceStateFacet = new ServiceStateFacetImpl();
		String state = getState();
		serviceStateFacet.setValue(state);
		eService.addFacet(serviceStateFacet);

		return eService;
	}

	public EService createEService() throws ResourceRegistryException {
		ResourceRegistryClient resourceRegistryClient = ResourceRegistryClientFactory.create();
		UUID eServiceUUID = UUID.fromString(applicationContext.id());
		try {
			ResourceRegistryClientFactory.includeContextsInInstanceHeader(true);
			eService = resourceRegistryClient.getInstance(EService.class, eServiceUUID);
			serviceStateFacet = eService.getFacets(ServiceStateFacet.class).get(0);
			if(serviceStateFacet==null) {
				serviceStateFacet = new ServiceStateFacetImpl();
				String state = getState();
				serviceStateFacet.setValue(state);
				eService.addFacet(serviceStateFacet);
				resourceRegistryPublisher.update(eService);
			}else{
				updateServiceStateFacet();
			}
		} catch (NotFoundException e) {
			eService = instantiateEService();
			createActivatesRelation(eService);
			
		} catch (AvailableInAnotherContextException e) {
			addToContext();
			try {
				eService = resourceRegistryClient.getInstance(EService.class, eServiceUUID);
			} catch (AvailableInAnotherContextException ex) {
				resourceRegistryPublisher.addResourceToCurrentContext(eService);
				// addToContext() is executed on HostingNode.
				// If the EService is still not available we need to create activates
				// relation because does not exists otherwise the EService should
				// already be in the context due to propagation constraint.
				createActivatesRelation(eService);
			}
			serviceStateFacet = eService.getFacets(ServiceStateFacet.class).get(0);
			if(serviceStateFacet==null) {
				serviceStateFacet = new ServiceStateFacetImpl();
				String state = getState();
				serviceStateFacet.setValue(state);
				eService.addFacet(serviceStateFacet);
				resourceRegistryPublisher.update(eService);
			}else{
				updateServiceStateFacet();
			}
		} catch (ResourceRegistryException e) {
			throw e;
		}
		return eService;
	}

	public void updateServiceStateFacet() throws ResourceRegistryException {
		String state = getState();
		serviceStateFacet.setValue(state);
		serviceStateFacet = resourceRegistryPublisher.updateFacet(serviceStateFacet);
	}

	private Activates<HostingNode, EService> createActivatesRelation(EService eService)
			throws ResourceRegistryException {

		HostingNode hostingNode = applicationContext.container().properties()
				.lookup(Constants.HOSTING_NODE_MANAGER_PROPERTY).value(HostingNodeManager.class).getHostingNode();

		PropagationConstraint propagationConstraint = new PropagationConstraintImpl();
		propagationConstraint.setRemoveConstraint(RemoveConstraint.cascade);
		propagationConstraint.setAddConstraint(AddConstraint.propagate);
		Activates<HostingNode, EService> activates = new ActivatesImpl<>(hostingNode, eService, propagationConstraint);

		try {
			activates = resourceRegistryPublisher.createIsRelatedTo(activates);
		} catch (NotFoundException e) {
			logger.error("THIS IS REALLY STRANGE. YOU SHOULD NE BE HERE. Error while creating {}.", activates, e);
			throw e;
		} catch (ResourceRegistryException e) {
			logger.error("Error while creating {}", activates, e);
			throw e;
		}

		hostingNode.attachResource(activates);

		return activates;

	}

	public void removeEService() throws ResourceRegistryException {
		try {
			resourceRegistryPublisher.delete(eService);
		} catch (ResourceRegistryException e) {
			logger.error("Unable to delete {}. Going to set the state to {}", applicationContext.name(), getState());
			updateServiceStateFacet();
		}
	}
	
	public Set<UUID> getContextsUUID() throws Exception {
		return resourceRegistryPublisher.getResourceContexts(eService);
	}

}
