package org.gcube.resources.discovery.icclient;

import static org.gcube.common.clients.stubs.jaxws.StubFactory.stubFor;
import static org.gcube.resources.discovery.icclient.stubs.CollectorConstants.collector;
import static org.gcube.resources.discovery.icclient.stubs.CollectorConstants.localname;

import java.io.StringReader;
import java.io.StringWriter;
import java.net.URI;
import java.util.ArrayList;
import java.util.List;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import javax.xml.ws.soap.SOAPFaultException;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathFactory;

import org.gcube.common.clients.stubs.jaxws.JAXWSUtils;
import org.gcube.common.scope.api.ServiceMap;
import org.gcube.resources.discovery.client.api.DiscoveryClient;
import org.gcube.resources.discovery.client.api.DiscoveryException;
import org.gcube.resources.discovery.client.queries.api.Query;
import org.gcube.resources.discovery.icclient.stubs.CollectorStub;
import org.gcube.resources.discovery.icclient.stubs.MalformedQueryException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;

/**
 * An {@link DiscoveryClient} that submits queries to the Information Collector, without parsing the results.
 * 
 */
public class ICClient implements DiscoveryClient<String> {

	private static final Logger log = LoggerFactory.getLogger(ICClient.class);

	//result split pattern
	private static final XPath xpath = XPathFactory.newInstance().newXPath();
	private static final String recordXpath = "/Resultset/Record";
	private static final String START_RECORD = "^\\p{Space}*<Record>\\p{Space}*";
	private static final String END_RECORD = "\\p{Space}*</Record>\\p{Space}*$";
	
	
	public List<String> submit(Query query) throws DiscoveryException {
		try {
			CollectorStub stub = getStub();
			String results = callService(query, stub);
			return splitIntoList(results);	
		}
		catch(MalformedQueryException e) {
			throw new DiscoveryException("malformed query "+query.expression(),e);
		}
		catch(SOAPFaultException e) {
			throw new RuntimeException(JAXWSUtils.remoteCause(e));
		}

	}

	//helper
	private String callService(Query query, CollectorStub stub) {

		String expression = query.expression();

		log.info("executing query {}",expression);

		long time = System.currentTimeMillis();

		String submittedExpression = Helper.queryAddAuthenticationControl(expression);

		String response = stub.execute(submittedExpression);

		log.info("executed query {} in {} ms",expression,System.currentTimeMillis()-time);

		return response;
	}

	//helper
	private CollectorStub getStub() {

		//find endpoint address in service map currently in scope
		String address = ServiceMap.instance.endpoint(localname);

		//obtain a JAXWS stub configured for gCube calls
		return stubFor(collector).at(URI.create(address));
	}

	
	
	//helper
	private List<String> splitIntoList(String response) {

		List<String> results = new ArrayList<String>();
		
		try{
			DocumentBuilder builder = DocumentBuilderFactory.newInstance().newDocumentBuilder();
			Document document = builder.parse(new InputSource(new StringReader(response)));
						
			NodeList nodes = (NodeList)xpath.evaluate(recordXpath, document, XPathConstants.NODESET);
			for (int i = 0; i < nodes.getLength(); i++){
				String ret = nodeToString(nodes.item(i));
				ret = ret.replaceFirst(START_RECORD, "");
				ret = ret.replaceFirst(END_RECORD, "");
				results.add(ret);
			}
		}catch(Exception e){
			log.error("error parsing result",e);
		}
		return results;
	}

	private String nodeToString(Node node) {
		StringWriter sw = new StringWriter();
		try {
			Transformer t = TransformerFactory.newInstance().newTransformer();
			t.setOutputProperty(OutputKeys.OMIT_XML_DECLARATION, "yes");
			t.setOutputProperty(OutputKeys.INDENT, "yes");
			t.transform(new DOMSource(node), new StreamResult(sw));
		} catch (TransformerException te) {
			System.out.println("nodeToString Transformer Exception");
		}
		return sw.toString();
	}

}
