package org.gcube.vremanagement.executor.state;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.Calendar;
import java.util.HashMap;
import java.util.Map;

import org.gcube.vremanagement.executor.plugin.ExecutorPluginContext;

/**
 * The runtime environment of the task.
 * @author Fabio Simeoni (University of Strathclyde)
 **/
public class TaskRuntime {

	/**The {@link TaskResource} corresponding to the execution of the task.*/
	TaskResource resource;
	
	/**Creates an instance.*/
	public TaskRuntime() {} //simplifies subclassing for mock testing (there is no resource)
	
	/**Creates an instance with an associated {@link TaskResource}
	 * @param resource*/
	public TaskRuntime(TaskResource resource) {this.resource=resource;}
	
	/**Returns the context of the plugin's task.
	 * @return the context.*/
	public ExecutorPluginContext getContext() {return resource.getContext();}
	
	/**
	 * Returns the start time of the task.
	 * @return the start time.
	 */
	public Calendar getStartTime() {return resource.getStartTime();}

	/**Returns the task inputs.
	 * @return the inputs.*/
	public Map<String,Object> getInputs() {
		Map<String,Object> inputs = resource.getInputs(); 
		return inputs==null?new HashMap<String,Object>():inputs;
	}

	/**Returns an input given its name.
	 * @return the input.*/
	public Object getInput(String name) {
		return getInputs().get(name);
	}
	
	/**Used internally to hide null outputs from task.
	 * @return the outputs.*/
	private Map<String,Object> getOutputs() {
		Map<String,Object> outputs = resource.getOutputs(); 
		return outputs==null?new HashMap<String,Object>():outputs;
	}
	
	/**
	 * Adds an output.
	 * @param name the name of the output.
	 * @param value the output.
	 * @return any output previously stored with the same name.
	 */
	public synchronized Object addOutput(String name,Object value) {
		Map<String,Object> outputs = this.getOutputs(); 
		Object o = outputs.put(name, value);
		resource.setOutputs(outputs);//not the most performant addition..
		return o;
	}
	
	/**
	 * A wrapper for named output to be used for bulk addition.
	 */
	//note cannot use AnyPair as that would induce a dependency on stubs during plugin's development
	public static class Entry {
		/** The entry's name.*/public String name;
		/** The entry's value.*/public Object value;
		/**
		 * Creates an instance from a name and a value.
		 * @param name the name.
		 * @param value the value.
		 */
		public Entry(String name,Object value) {this.name=name;this.value=value;}
	}
	
	/**
	 * Adds one or more outputs.
	 * @param entries the outputs.
	 */
	public synchronized void addOutput(Entry ...entries) {
		if (entries==null || entries.length==0) return;
		//given poor addition performance, better not delegate to overloaded method
		Map<String,Object> outputs = this.getOutputs();
		for (Entry entry : entries) outputs.put(entry.name, entry.value);
		resource.setOutputs(outputs);
	}
	
	/**
	 * Removes one or more outputs.
	 * @param names the name of the outputs
	 */
	public synchronized void removeOutput(String ... names) {
		if (names==null) return;
		Map<String,Object> outputs = this.getOutputs();
		for (String name : names) outputs.remove(name);
		resource.setOutputs(outputs);
	}
	
	/**
	 * Throws an exception shows as the value of the error RP.
	 * @param e the exception.
	 */
	public synchronized void throwException(Exception e) {
		StringWriter writer = new StringWriter();
		e.printStackTrace(new PrintWriter(writer));
		resource.setError("\n"+writer.getBuffer().toString());
	}
}

