package org.gcube.vremanagement.executor.state;

import java.util.Calendar;
import java.util.Map;

import javax.naming.OperationNotSupportedException;

import org.gcube.common.core.plugins.GCUBEPluginManager;
import org.gcube.common.core.plugins.GCUBEPluginManager.PluginTopic;
import org.gcube.common.core.scope.GCUBEScope;
import org.gcube.common.core.state.GCUBEWSResource;
import org.gcube.common.core.utils.events.GCUBEEvent;
import org.gcube.common.core.utils.handlers.GCUBEScheduledHandler;
import org.gcube.common.core.utils.handlers.events.Event;
import org.gcube.common.core.utils.handlers.events.Event.Done;
import org.gcube.common.core.utils.handlers.events.Event.Failed;
import org.gcube.common.core.utils.handlers.events.Event.LifetimeEvent;
import org.gcube.common.core.utils.handlers.events.Monitor;
import org.gcube.common.core.utils.logging.GCUBELog;
import org.gcube.vremanagement.executor.contexts.ServiceContext;
import org.gcube.vremanagement.executor.plugin.ExecutorPluginContext;
import org.gcube.vremanagement.executor.plugin.ExecutorTask;
import org.gcube.vremanagement.executor.stubs.AnyMap;
import org.gcube.vremanagement.executor.stubs.TaskDescription;
import org.gcube.vremanagement.executor.stubs.Utils;
import org.globus.wsrf.ResourceProperty;

/**
 * An extension of {@link GCUBEWSResource} that provides and publishes a runtime environment for a {@link ExecutorTask}.
 * @author Fabio Simeoni (University of Strathclyde)
 */
public class TaskResource extends GCUBEWSResource{

	/** Name of the task descriptions RP.*/
 	static final String RP_TYPE="Type";
 	/** Name of the start-time RP.*/
 	static final String RP_START_TIME="Started";
	/** Name of the task state RP.*/
 	static final String RP_STATE="State";
	/** Name of the task state RP.*/
 	static final String RP_PARAMS="Inputs";
	/** Name of the task state RP.*/
 	static final String RP_OUTPUT="Outputs";
 	/** Name of the log RP.*/
 	static final String RP_LOG="Log";
 	/** Name of the log RP.*/
 	static final String RP_ERROR="Error";
 	
	/** RP names. */
	private static String[] RPNames = {RP_TYPE,RP_START_TIME,RP_PARAMS,RP_LOG,RP_ERROR};
	/** RP names. */
	private static String[] TopicNames = {RP_OUTPUT,RP_STATE};

	/**Maximum size of log RP.*/
	private static final int LOG_LINE_SIZE = 10;
	
	//using transient here only for documentation(no standard serialisation is used)
	/**The context of the task's plugin.*/
	transient ExecutorPluginContext context;
	/**A flag that indicates whether the task is scheduled.*/
	transient boolean isScheduled=false;  //will find out at launch time.
	/**The task.*/
	transient ExecutorTask task;
	/**The task's runtime.*/
	transient TaskRuntime runtime;
	
	/** {@inheritDoc} */ public String[] getPropertyNames() {return RPNames;}
	
	/** {@inheritDoc} */ protected String[] getTopicNames() {return TopicNames;}
	
	/**{@inheritDoc}*/
	protected void initialise(Object... args) throws Exception {
		
		if (args.length!=2) throw new IllegalArgumentException("plugin context and task inputs expected");
		
		//initialise state
		setContext((ExecutorPluginContext) args[0]);
		setInputs(Utils.intern((AnyMap) args[1]));
		setStartTime(Calendar.getInstance());

		//launch task
		launch(getServiceContext().getScope());//note:scope not set on ws-resource yet		
	}
	
	/**Returns the context of the task's plugin.
	 * @return the context.*/
	public ExecutorPluginContext getContext() {return this.context;} //used internally and by the delegate
	
	/**Sets the context of the task's plugin.
	 * @param context the context.*/
	public void setContext(ExecutorPluginContext context) {//exposed for the delegate
		this.context=context;
		this.setType(context.getDescription()); //set task description from context
	} 
	
	/**
	 * Returns the task.
	 * @return the task.
	 */
	public ExecutorTask getTask(){return this.task;} //exposed for the delegate

	/**
	 * Sets the value of the start time RP.
	 * @param value the value.
	 */
	public void setStartTime(Calendar value) {//set at init/load points, no need for synchronisation
		ResourceProperty prop = this.getResourcePropertySet().get(RP_START_TIME);
		prop.clear();
		prop.add(value);
	}
	
	/**
	 * Gets the value of the start time RP.
	 * @return value the value.
	 */
	public Calendar getStartTime() {//set at init/load points, no need for synchronisation
		ResourceProperty prop = this.getResourcePropertySet().get(RP_START_TIME);
		return (Calendar) prop.get(0);
	}
	
	/**Sets the value of the type RP.
	 * @param value the value.*/
	protected void setType(TaskDescription value) {//set at init/load time, no need for synchronisation
		ResourceProperty prop = getResourcePropertySet().get(RP_TYPE);
		prop.clear();
		prop.add(value); 
	}
	
	/**Sets the value of the state RP.
	 * @param value the value.*/
	public synchronized void setState(String value) {
		ResourceProperty prop = this.getResourcePropertySet().get(RP_STATE);
		prop.clear();
		prop.add(value);
	}
	
	//////////////////////////////////////////////////////////////////////I/O MANAGEMENT
	
	/**
	 * Sets the task inputs.
	 * @param inputs the inputs.
	 */
	public void setInputs(Map<String,Object> inputs) {//set at init/load time, no need for synchronisation
		if (inputs==null) return; //we don't write into an RP at all when there are no inputs
		ResourceProperty prop = this.getResourcePropertySet().get(RP_PARAMS);
		prop.clear();
		prop.add(Utils.extern(inputs));
	}
	

	/**Returns (a copy of) the task inputs.
	 * @return the inputs.*/
	public Map<String,Object> getInputs() {//use local map rather than RP
		ResourceProperty prop = this.getResourcePropertySet().get(RP_PARAMS);
		return (prop.size()==0)?null:Utils.intern(((AnyMap) prop.get(0)));
	}
	
	/**Sets the taks's outputs.
	 * @param outputs the outputs.*/
	public synchronized void setOutputs(Map<String,Object> outputs) {
		if (outputs==null) return; //we don't write null values into an RP
		ResourceProperty prop = getResourcePropertySet().get(RP_OUTPUT);
		prop.clear();
		prop.add(Utils.extern(outputs));
		store(); //state has changed
	}
	
	/**Returns the task outputs.
	 * @return the inputs.*/
	public synchronized Map<String,Object> getOutputs() {
		ResourceProperty prop = getResourcePropertySet().get(RP_OUTPUT);
		return prop.size()==0?null:Utils.intern(((AnyMap) prop.get(0)));
	}
	
	//////////////////////////////////////////////////////////////////////ERROR MANAGEMENT
	
	/**
	 * Sets the value of the error RP.
	 * @param value the value.
	 */
	public synchronized void setError(String value) {
		if (value==null) return;//we dont write null value into RP	
		ResourceProperty prop = getResourcePropertySet().get(RP_ERROR);
		prop.clear();
		prop.add(value);
		store(); //state has changed
	}
	
	/**
	 * Returns the value of the error RP.
	 * @return the value.
	 */
	public String getError() {
		ResourceProperty prop = getResourcePropertySet().get(RP_ERROR);
		return (prop.size()>0)?(String) prop.get(0):null; //we may have nothing to return
	}
	

	////////////////////////////////////////////////////////////////////// LOG MANAGEMENT
	
	/**
	 * Adds a message to the current contents of the log RP.
	 * @param msg the message.
	 */
	private synchronized void log(String msg) {
		msg = "["+Calendar.getInstance().getTime()+"] "+msg;
		//build buffer from RP value
		
		ResourceProperty prop = getResourcePropertySet().get(RP_LOG);
		StringBuilder log = new StringBuilder(getLog()==null?"":(String) prop.get(0));
		log.append(msg+"\n");
		//roll if needed
		if (log.toString().split("\n").length>LOG_LINE_SIZE) log.delete(0,log.indexOf("\n")+1);
		setLog(log.toString());
		store(); //state has changed.
	}
	
	
	/**
	 * Sets the value of the log RP.
	 * @param value the contents.
	 */
	public synchronized void setLog(String value) {
		ResourceProperty prop = getResourcePropertySet().get(RP_LOG);
		prop.clear();
		prop.add("\n"+value.toString());//adds a leading newline
	}
	
	
	/**
	 * Returns the contents of the log.
	 * @return the contents.
	 */
	public synchronized String getLog() {//exposes for delegate
		ResourceProperty prop = getResourcePropertySet().get(RP_LOG);
		return prop.size()==0?null:(String) prop.get(0);
	}
	
	private class ScheduledTaskConsumer extends GCUBEPluginManager.PluginConsumer<ExecutorPluginContext> {
		/**{@inheritDoc}*/
		protected void onDeregistration(GCUBEEvent<? extends PluginTopic, ? extends ExecutorPluginContext> event) {//stop scheduled event
			if (event.getPayload()==TaskResource.this.getContext()) ((GCUBEScheduledHandler<?>) TaskResource.this.getTask()).stop();
		}
	}
		
	//////////////////////////////////////////////////////////////////////
	
	/**
	 * Attempts to stop execution.
	 * @throws UnsupportedOperationException if the task cannot be stopped.
	 * @throws Exception if the task could not be stopped.
	 **/
	public void stop() throws OperationNotSupportedException,Exception {
		getTask().stop();
	}
	
	/**
	 * Launches the task in a given scope.
	 * @param scope the scope.
	 * @throws Exception if the task could not launched.
	 */
	public void launch(GCUBEScope scope) throws Exception {
		
		task = context.getTaskClass().newInstance(); //create a task instance
		task.setName(context.getPlugin().getServiceName()); //set task's name.
		//inject runtime
		runtime = new TaskRuntime(this); //create a new runtime for this execution.
		task.setHandled(runtime); 
		//inject logger
		TaskLog runtimeLogger = new TaskLog(this); //create a logger for the task.
		task.setLogger(runtimeLogger); 
		//inject managers
		task.setScopeManager(getServiceContext());
		task.setSecurityManager(getServiceContext());
		
		//scheduled? if so remember it for persistence and inject into inner task.
		if (GCUBEScheduledHandler.class.isAssignableFrom(task.getClass())){//propagates task to scheduled task;
			isScheduled=true;
			@SuppressWarnings({"rawtypes","unchecked"}) 
			GCUBEScheduledHandler<TaskRuntime> scheduler = (GCUBEScheduledHandler) task;
			scheduler.getScheduled().setHandled(runtime);//inject in
			scheduler.getScheduled().setLogger(scheduler.getLogger());
			
			ServiceContext.getContext().getPluginManager().subscribe(new ScheduledTaskConsumer(), GCUBEPluginManager.PluginTopic.DEREGISTRATION);
		}
		
		//listen to task's events
		task.subscribe(new TaskMonitor());
		
		//set initial state (will be monitored)
		this.setState(task.getState().toString());
		
		//launch task so prepared in dedicated thread
		Thread t = new Thread("task-"+TaskResource.this.getID().getValue()+"-execution"){
			public void run(){
				try {task.run();}
				catch(Exception e) { //any synchronous exception?
					logger.trace("task "+task.getName()+" failed",e);
					runtime.throwException(e); //publish problem
					//set failed state (will be monitored)
					setState(org.gcube.common.core.utils.handlers.lifetime.State.Failed.INSTANCE.toString());
				}
			}
		};
		
		//propagate scope and credentials to thread
		getServiceContext().setScope(t,scope);
		getServiceContext().useCallerCredentials(t);
		//launch!
		t.start();
	}

	/** {@inheritDoc} */
	public void store() {//only scheduled tasks are persisted
		if (!isScheduled) return; else super.store();
	}
	

	//////////////////////////////////////////////////////////////////////

	/**Extends {@link GCUBELog} to forward a selection of log messages to the log RP.*/
	public class TaskLog extends GCUBELog {

	  	 /**Creates an instance for a given object.
	     * @param obj the object.*/
	    public TaskLog(Object obj) {super(obj);}

	    /**{@inheritDoc}*/public void error(Object arg0) {super.error(arg0);log("ERROR:"+arg0.toString());}
	    /**{@inheritDoc}*/public void error(Object arg0, Throwable arg1) {super.error(arg0,arg1);log("ERROR:"+arg0.toString());}
	    /**{@inheritDoc}*/public void fatal(Object arg0) {super.fatal(arg0);log("FATAL:"+arg0.toString());}
	    /**{@inheritDoc}*/public void fatal(Object arg0, Throwable arg1) {super.fatal(arg0,arg1);log("FATAL:"+arg0.toString());}
	    /**{@inheritDoc}*/public void info(Object arg0) {super.info(arg0);log("INFO:"+arg0.toString());}
	    /**{@inheritDoc}*/public void info(Object arg0, Throwable arg1) {super.info(arg0,arg1);log("INFO:"+arg0.toString());}
	    /**{@inheritDoc}*/public void warn(Object arg0) {super.warn(arg0);log("WARN:"+arg0.toString());}
	    /**{@inheritDoc}*/public void warn(Object arg0, Throwable arg1) {super.warn(arg0,arg1);log("WARN:"+arg0.toString());}

	}
	
	/**Extends {@link Monitor} to consumer task lifetime events.*/
	private class TaskMonitor extends Monitor {
		/**{@inheritDoc}*/protected void onCompletion(Done e) {setExpirationTime();}
		/**{@inheritDoc}*/protected void onFailure(Failed e) {setExpirationTime();}
		/**Used internally to set the expiration time of a completed or failed task*/
		private void setExpirationTime() {
			Calendar time = Calendar.getInstance(); 
			time.add(Calendar.MINUTE, Math.min(getContext().getTimeToLive(),60));
			setTerminationTime(time);
			store();//we need to store the termination time. if the container goes down now, the task may be restarted! 
		}
		/**{@inheritDoc}*/protected void onAnyEvent(Event<?, ?> e) {
			super.onAnyEvent(e);
			//for all lifetime events publish new state
			if (e instanceof LifetimeEvent) setState(((LifetimeEvent) e).getPayload().getState().toString()); 
		}
	}


}
