package org.gcube.resourcemanagement.analyser;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.Callable;

import org.gcube.com.fasterxml.jackson.databind.JsonNode;
import org.gcube.com.fasterxml.jackson.databind.ObjectMapper;
import org.gcube.com.fasterxml.jackson.databind.node.ObjectNode;
import org.gcube.common.security.AuthorizedTasks;
import org.gcube.common.security.secrets.Secret;
import org.gcube.informationsystem.base.reference.AccessType;
import org.gcube.informationsystem.base.reference.IdentifiableElement;
import org.gcube.informationsystem.model.reference.ERElement;
import org.gcube.informationsystem.model.reference.entities.Resource;
import org.gcube.informationsystem.queries.templates.impl.properties.QueryTemplateReferenceImpl;
import org.gcube.informationsystem.queries.templates.reference.entities.QueryTemplate;
import org.gcube.informationsystem.queries.templates.reference.properties.QueryTemplateReference;
import org.gcube.informationsystem.resourceregistry.api.exceptions.ResourceRegistryException;
import org.gcube.informationsystem.resourceregistry.api.rest.SharingPath.SharingOperation;
import org.gcube.informationsystem.resourceregistry.client.ResourceRegistryClient;
import org.gcube.informationsystem.resourceregistry.client.ResourceRegistryClientFactory;
import org.gcube.informationsystem.resourceregistry.publisher.ResourceRegistryPublisher;
import org.gcube.informationsystem.resourceregistry.publisher.ResourceRegistryPublisherFactory;
import org.gcube.informationsystem.resourceregistry.queries.templates.ResourceRegistryQueryTemplateClient;
import org.gcube.informationsystem.resourceregistry.queries.templates.ResourceRegistryQueryTemplateClientFactory;
import org.gcube.informationsystem.serialization.ElementMapper;
import org.gcube.informationsystem.utils.TypeUtility;
import org.gcube.resourcemanagement.context.TargetContext;
import org.gcube.resourcemanagement.resource.DerivatedRelatedResourceGroup;
import org.gcube.resourcemanagement.resource.Instance;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import jakarta.ws.rs.BadRequestException;
import jakarta.ws.rs.InternalServerErrorException;
import jakarta.ws.rs.WebApplicationException;

/**
 * @author Luca Frosini (ISTI - CNR)
 */
public abstract class InstanceAnalyser<R extends Resource, I extends Instance> {

    /**
     * Logger
     */
    protected Logger logger = LoggerFactory.getLogger(this.getClass());

    protected String type;
    protected UUID instanceUUID;
    
    protected ResourceRegistryClient client;
    protected ResourceRegistryPublisher publisher;
    protected ResourceRegistryQueryTemplateClient templateClient;

    protected int level;
    protected boolean mandatory;

    protected Resource ancestor;
    protected R resource;
    protected I instance;

    protected List<Resource> mandatoryRelatedResources;
    protected Set<DerivatedRelatedResourceGroup<I>> derivatedRelatedResourceGroups;   

    protected ObjectMapper mapper;

    protected TargetContext targetContext;

    public InstanceAnalyser() {    
        this.client = ResourceRegistryClientFactory.create();
        this.mapper = new ObjectMapper();
        this.level = 0;
    }

    public InstanceAnalyser(String type, UUID instanceUUID) {
        this();
        this.type = type;
        this.instanceUUID = instanceUUID;
    }

    /***
     * @return the name of the analyser 
     */
    public String getName() {
        return this.getClass().getSimpleName();
    }

    /**
     * @return the supported type of the analyser
     */
    public String getType() {
        if(type==null) {
            type =  TypeUtility.getTypeName(getTypeClass());
        }
        return type;
    }

    public void setType(String type) {
        this.type = type;
    }

    public abstract Class<R> getTypeClass();

    public abstract Class<I> getInstanceClass();

    public UUID getInstanceUUID() {
        return instanceUUID;
    }

    public void setInstanceUUID(UUID instanceUUID) {
        this.instanceUUID = instanceUUID;
    }

    public void setTargetContext(TargetContext targetContext) {
        this.targetContext = targetContext;
    }
    
    /**
     * @return the description of the analyser
     */
    public String getDescription() {
        StringBuffer sb = new StringBuffer();
        sb.append("It analyses instances of type ");
        sb.append(getType());
        if(polymorphic()){
            sb.append(" and all its derivated types");
        }
        return sb.toString();
     }

    /**
     * @return true of the analyser supports instances of derivated type
     * aparte the one specified in the getType method.
     */
    public boolean polymorphic() {
        return false;
    }

    /**
     * @return the priority of the analyser. When two analysers are available for the same type, the one with the highest priority is used first.
     * By default the priority is 100 so we have room for other analysers to be used first.
     * It is strongly reccomended to use 50 as first option and then 25 to leave room for others with higher priority to be used first
     * in respect to the one aleady developed. This avoid to released a analyser just to change the priority.
     */
    public Integer getPriority(){
        return 100;
    }

    public Resource getAncestor() {
        return ancestor;
    }

    public void setAncestor(Resource ancestor) {
        this.ancestor = ancestor;
    }

    public int getLevel() {
        return level;
    }

    public void setLevel(int level) {
        this.level = level;
    }

    public R getResource() throws WebApplicationException, ResourceRegistryException {
        if(resource==null) {
            resource = client.getInstance(getTypeClass(), instanceUUID);
        }
        return resource;
    }

    public void setResource(R resource) {
        this.resource = resource;
    }
    
    /**
     * Return the set of Derivated Related Resources (D-RR) to be analysed
     * @return the set of Derivated Related Resources (D-RR)
     * @throws WebApplicationException
     * @throws ResourceRegistryException
     */
    protected abstract Set<DerivatedRelatedResourceGroup<I>> getDerivatedRelatedResourceGroup() throws WebApplicationException, ResourceRegistryException;

    /**
     * Return the set of new instances to create during add to context operation
     * It get the set of instance to create from client, aanlyse it and reate the real instance.
     * @param newInstances
     * @return
     * @throws WebApplicationException
     */
    protected abstract List<IdentifiableElement> validateInstancesToCreate(List<IdentifiableElement> newInstances) throws WebApplicationException, ResourceRegistryException;

    /**
     * Return the set of new instances to create to return to the client
     * which has to analyse and to provide the real instances to create 
     * @return
     * @throws WebApplicationException
     */
    protected abstract List<IdentifiableElement> getNewInstances() throws WebApplicationException, ResourceRegistryException;

    protected QueryTemplateReference getMandatoryRelatedResourcesQTR(){
        QueryTemplateReference queryTemplateReference = new QueryTemplateReferenceImpl();
        queryTemplateReference.setName("GetMandatoryRelatedResources");
        queryTemplateReference.addVariable("$id", instanceUUID.toString());
        queryTemplateReference.addVariable("$type", getType());
        return queryTemplateReference;
    }

    /**
     * Return the list of Mandatory Related Resources (M-RR)
     * @return the list of Mandatory Related Resources (M-RR)
     * @throws WebApplicationException
     */
    public List<Resource> getMandatoryRelatedResources() throws WebApplicationException, ResourceRegistryException {
        if(mandatoryRelatedResources==null){
            logger.debug("Going to discover Mandatory Related Resources (M-RRs) from {} with UUID: {}", getType(), instanceUUID);
            mandatoryRelatedResources = new ArrayList<>();
            QueryTemplateReference queryTemplateReference = getMandatoryRelatedResourcesQTR();
            mandatoryRelatedResources.addAll(executeQueryTemplate(queryTemplateReference));
        }
        return mandatoryRelatedResources;
    }

    /**
     * This function is used to discover the Derivated Related Resources (D-RR)
     * analyzing the DiscoveryFacets of the instance
     * @return the set of D-RR
     * @throws WebApplicationException 
     */
    public Set<DerivatedRelatedResourceGroup<I>> getDerivatedRelatedResources() throws WebApplicationException, ResourceRegistryException {
        if(derivatedRelatedResourceGroups==null){
            logger.debug("Going to get Derivated Related Resources (D-RRs) from {} with UUID: {}", getType(), instanceUUID);
            derivatedRelatedResourceGroups = getDerivatedRelatedResourceGroup();
        }
        return derivatedRelatedResourceGroups;
    }

    protected List<Resource> executeQuery(JsonNode query) throws WebApplicationException, ResourceRegistryException {
        logger.trace("Executing query: {}", query);
        List<Resource> resources = client.jsonQuery(query);
        return resources;
    }

    protected List<Resource> executeQueryTemplate(QueryTemplateReference queryTemplateReference) throws WebApplicationException, ResourceRegistryException {
        logger.trace("Executing query template: {}", queryTemplateReference);
        ObjectNode parameters = mapper.valueToTree(queryTemplateReference.getVariables());
        String ret = client.runQueryTemplate(queryTemplateReference.getName(), parameters);
        List<Resource> resources;
        try {
            resources = ElementMapper.unmarshalList(Resource.class, ret);
        } catch (IOException e) {
            StringBuffer error = new StringBuffer();
            error.append("Error while unmarshalling the result of the query template ");
            error.append(queryTemplateReference);
            logger.error(error.toString(), e);
            throw new InternalServerErrorException(error.toString(),e);
        }
        return resources;
    }

    public I read(SharingOperation operation) throws WebApplicationException, ResourceRegistryException {
        if(operation==SharingOperation.ADD){
            return readToAdd();
        } else if(operation==SharingOperation.REMOVE){
            return readToRemove();
        } else {
            throw new UnsupportedOperationException("Operation not supported");
        }
    }

    protected I readToAdd() throws WebApplicationException, ResourceRegistryException {
        I i = null;
        try {
            i = (I) getInstanceClass().getConstructor().newInstance();
        } catch (Exception e) {
            StringBuffer error = new StringBuffer();
            error.append("Error while creating a new instance of ");
            error.append(getInstanceClass().getName());
            logger.error(error.toString(), e);
            throw new InternalServerErrorException(error.toString(),e);
        }
        
        i.setResource(getResource());

        List<IdentifiableElement> elements = getNewInstances();
        i.addNewInstances(elements);

        for(Resource rr : getMandatoryRelatedResources()){
            InstanceAnalyser<Resource, Instance> analyser = InstanceAnalyserFactory.getInstanceAnalyser(rr.getTypeName(), rr.getID());
            if(analyser!=null){
                analyser.setAncestor(resource);
                analyser.setResource(rr);
                analyser.setLevel(this.getLevel()+1);
                Instance in = analyser.getInstance();
                i.addMandatoryRelatedResource(in);
            }
        }

        if(level==0 || ancestor==null || mandatory){
            getDerivatedRelatedResources();
            for(DerivatedRelatedResourceGroup<I> group : derivatedRelatedResourceGroups){
                @SuppressWarnings("unchecked")
                DerivatedRelatedResourceGroup<Instance> g = (DerivatedRelatedResourceGroup<Instance>) group;
                i.addDerivatedRelatedResource(g);
            }
        }

        return i;
    }

    public I getInstance() throws WebApplicationException, ResourceRegistryException {
        if(instance==null){
            instance = readToAdd();
        }
        return instance;
    }

    public void setInstance(I instance) {
        this.instance = instance;
        if(this.instanceUUID!=null && this.instanceUUID.compareTo(instance.getID())!=0){
            StringBuffer error = new StringBuffer();
            error.append("The instance UUID ");
            error.append(this.instanceUUID);
            error.append(" is different from the instance UUID ");
            error.append(instance.getID());
            logger.error(error.toString());
            throw new BadRequestException(error.toString());
        }else{
            this.instanceUUID = instance.getID();
        }
    }

    public I readToRemove() throws WebApplicationException {
        throw new UnsupportedOperationException("Method not implemented yet");
    }

    protected List<ERElement> executeAdd() throws WebApplicationException, ResourceRegistryException {
        ResourceRegistryPublisher rrp = ResourceRegistryPublisherFactory.create();
        List<ERElement> added = rrp.addResourceToContext(instance.getType(), instance.getID(), targetContext.getUUID(), false);
        return added;
    }

    protected List<IdentifiableElement> createNewInstances() throws WebApplicationException, ResourceRegistryException {
        List<IdentifiableElement> newInstancesToCreate = validateInstancesToCreate(instance.getNewInstances());
        for(IdentifiableElement newInstanceToCreate : newInstancesToCreate){
            if(newInstanceToCreate instanceof QueryTemplate){
                QueryTemplate queryTemplate = (QueryTemplate) newInstanceToCreate;
                QueryTemplate createdQueryTemplate = templateClient.create(queryTemplate);
                newInstancesToCreate.add(createdQueryTemplate);
            }else if(newInstanceToCreate instanceof ERElement){
                ERElement element = (ERElement) newInstanceToCreate;
                ERElement createdElement = publisher.create(element);
                newInstancesToCreate.add(createdElement);
            }else{
                logger.warn("It is supported creation of {} i.e. {}. It will be skipped", AccessType.getAccessType(newInstanceToCreate.getClass()), newInstanceToCreate.getTypeName());
            }
        }
        return newInstancesToCreate;
    }

    protected Set<Instance> validateMandatoryRelatedResources(I expectedInstance) throws WebApplicationException, ResourceRegistryException {
        Set<Instance> requestedMRRs = instance.getMandatoryRelatedResources();
        Set<Instance> expectedMRRs = expectedInstance.getMandatoryRelatedResources();
        
        if(requestedMRRs.containsAll(expectedMRRs)){
            logger.debug("All expected mandatory related resources for {} {} are present in the request.", getType(), getInstanceUUID());
        }else{
            StringBuffer error = new StringBuffer();
            error.append("Not all mandatory related resources for ");
            error.append(getType());
            error.append(" ");
            error.append(getInstanceUUID());
            error.append(" are present in the request. ");
            expectedMRRs.removeAll(requestedMRRs);
            error.append("Missing mandatory related resources are ");
            error.append(expectedMRRs);
            logger.error(error.toString());
            throw new WebApplicationException(error.toString());
        }

        
        if(expectedMRRs.containsAll(requestedMRRs)){
            logger.debug("Mandatory related resources in the request for {} {} are exactly the expected ones.", getType(), getInstanceUUID());
        }else{
            StringBuffer error = new StringBuffer();
            error.append("Requested mandatory related resources for ");
            error.append(getType());
            error.append(" ");
            error.append(getInstanceUUID());
            error.append(" in the request are more than the expected ones. ");
            requestedMRRs.removeAll(expectedMRRs);
            error.append("Unexpected mandatory related resources are ");
            error.append(requestedMRRs);
            logger.error(error.toString());
            throw new WebApplicationException(error.toString());
        }

        return expectedMRRs;
    }

    protected Set<Instance> addMandatoryRelatedResources(I expectedInstance) throws WebApplicationException, ResourceRegistryException {
        Set<Instance> addedMandatoryRelatedResources = new HashSet<>();
        Set<Instance> requestedMRRs = instance.getMandatoryRelatedResources();
        
        validateMandatoryRelatedResources(expectedInstance);

        for(Instance mandatory : requestedMRRs){
            logger.debug("Going to add mandatory related resource: {}", mandatory);
            InstanceAnalyser<Resource, Instance> analyser = InstanceAnalyserFactory.getInstanceAnalyser(mandatory.getType(), mandatory.getID());
            if(analyser!=null){
                analyser.setAncestor(resource);
                analyser.setLevel(this.getLevel()+1);
                analyser.setInstance(mandatory);
                analyser.setTargetContext(targetContext);
                Instance mandatoryRR = analyser.add();
                if(mandatoryRR!=null) {
                    addedMandatoryRelatedResources.add(mandatoryRR);
                }
            }
        }
        return addedMandatoryRelatedResources;
    }

    protected DerivatedRelatedResourceGroup<Instance> addDerivatedRelatedResourceGroup(DerivatedRelatedResourceGroup<Instance> requested, DerivatedRelatedResourceGroup<Instance> expected) throws WebApplicationException, ResourceRegistryException {
        
        DerivatedRelatedResourceGroup<Instance> drrgToAdd = new DerivatedRelatedResourceGroup<>(expected.getInstanceClass());
        drrgToAdd.setName(expected.getName());
        drrgToAdd.setDescription(expected.getDescription());
        drrgToAdd.setMin(expected.getMin());
        drrgToAdd.setMax(expected.getMax());
        drrgToAdd.setAncestor(getResource());
        
        /*
         * I don't want to manipulate the requested set so I create a new one
         */ 
        Set<Instance> requestedInstances = new HashSet<>(requested.getInstances());
        /*
         * I maintain only the instances that are present in the expected set
         * the others belongs to other analyser
         */
        requestedInstances.retainAll(expected.getInstances());

        int size = requestedInstances.size();
        if(size < expected.getMin()){
            StringBuffer error = new StringBuffer();
            error.append("The number of requested instances for ");
            error.append(getType());
            error.append(" ");
            error.append(getInstanceUUID());
            error.append(" is less than the minimum expected. ");
            error.append("Requested: ");
            error.append(size);
            error.append(" Minimum: ");
            error.append(expected.getMin());
            logger.error(error.toString());
            throw new WebApplicationException(error.toString());
        }

        if(size > expected.getMax()){
            StringBuffer error = new StringBuffer();
            error.append("The number of requested instances for ");
            error.append(getType());
            error.append(" ");
            error.append(getInstanceUUID());
            error.append(" is more than the maximum expected. ");
            error.append("Requested: ");
            error.append(size);
            error.append(" Maximum: ");
            error.append(expected.getMax());
            logger.error(error.toString());
            throw new WebApplicationException(error.toString());
        }

        Set<Instance> addedInstances = new HashSet<>();
        
        for(Instance instance : requestedInstances){
            InstanceAnalyser<Resource, Instance> analyser = InstanceAnalyserFactory.getInstanceAnalyser(instance.getType(), instance.getID());
            if(analyser!=null){
                analyser.setAncestor(resource);
                analyser.setLevel(this.getLevel()+1);
                analyser.setInstance(instance);
                analyser.setTargetContext(targetContext);
                Instance instanceToAdd = analyser.add();
                if(instanceToAdd!=null) {
                    addedInstances.add(instanceToAdd);
                }
            }
        }
        
        drrgToAdd.addInstances(addedInstances);
        return drrgToAdd;
    }

    protected Map<String, DerivatedRelatedResourceGroup<Instance>> getDRRMap(Collection<DerivatedRelatedResourceGroup<Instance>> derivatedRelatedResources){
        Map<String, DerivatedRelatedResourceGroup<Instance>> map = new HashMap<>();
        for(DerivatedRelatedResourceGroup<Instance> drr : derivatedRelatedResources){
            map.put(drr.getName(), drr);
        }
        return map;
    }

    protected Map<String, DerivatedRelatedResourceGroup<Instance>> validateDerivatedRelatedResources(I expectedInstance) throws WebApplicationException, ResourceRegistryException {
        Set<DerivatedRelatedResourceGroup<Instance>> requestedDRRs = instance.getDerivatedRelatedResources();
        Set<DerivatedRelatedResourceGroup<Instance>> expectedDRRs = expectedInstance.getDerivatedRelatedResources();
        
        Map<String, DerivatedRelatedResourceGroup<Instance>> expected = getDRRMap(expectedDRRs);

        if(requestedDRRs.containsAll(expectedDRRs)){
            logger.debug("All expected derivated related resources for {} {} are present in the request.", getType(), getInstanceUUID());
        }else{
            StringBuffer error = new StringBuffer();
            error.append("Not all derivated related resources for ");
            error.append(getType());
            error.append(" ");
            error.append(getInstanceUUID());
            error.append(" are present in the request. ");
            expectedDRRs.removeAll(requestedDRRs);
            error.append("Missing derivated related resources are ");
            error.append(expectedDRRs);
            logger.error(error.toString());
            throw new WebApplicationException(error.toString());
        }

        
        if(expectedDRRs.containsAll(requestedDRRs)){
            logger.debug("Derivated related resources in the request for {} {} are exactly the expected ones.", getType(), getInstanceUUID());
        }else{
            StringBuffer error = new StringBuffer();
            error.append("Requested derivated related resources for ");
            error.append(getType());
            error.append(" ");
            error.append(getInstanceUUID());
            error.append(" in the request are more than the expected ones. ");
            requestedDRRs.removeAll(expectedDRRs);
            error.append("Unexpected derivated related resources are ");
            error.append(expectedDRRs);
            logger.error(error.toString());
            throw new WebApplicationException(error.toString());
        }

        return expected;
    }

    protected Set<DerivatedRelatedResourceGroup<Instance>> addDerivatedRelatedResources(I expectedInstance) throws WebApplicationException, ResourceRegistryException {
        Set<DerivatedRelatedResourceGroup<Instance>> addedDerivatedRelatedResources = new HashSet<>();
        
        Set<DerivatedRelatedResourceGroup<Instance>> requestedDRRs = instance.getDerivatedRelatedResources();
        Map<String, DerivatedRelatedResourceGroup<Instance>> expectedDRRs = validateDerivatedRelatedResources(expectedInstance);

        for(DerivatedRelatedResourceGroup<Instance> requested : requestedDRRs){
            DerivatedRelatedResourceGroup<Instance> expected = expectedDRRs.get(requested.getName());
            DerivatedRelatedResourceGroup<Instance> drrgToAdd = addDerivatedRelatedResourceGroup(requested, expected);
            addedDerivatedRelatedResources.add(drrgToAdd);
        }
        return addedDerivatedRelatedResources;
    }

    public I add() throws WebApplicationException, ResourceRegistryException {
        I addedInstance = null;
        try {
            @SuppressWarnings("unchecked")
            I i = (I) instance.getClass().getConstructor().newInstance();
            /*
             * Why this? I don't like suppress warnings in function declaration
             * instead I prefer to suppress them in the line where they are used
             */ 
            addedInstance = i;
        } catch (Exception e) {
            StringBuffer error = new StringBuffer();
            error.append("Error while creating a new instance of ");
            error.append(getInstanceClass().getName());
            logger.error(error.toString(), e);
            throw new InternalServerErrorException(error.toString(),e);
        }

        this.publisher = ResourceRegistryPublisherFactory.create();
        this.templateClient = ResourceRegistryQueryTemplateClientFactory.create();

        I expectedInstance = readToAdd();

        Set<Instance> mandatoryRelatedResources = addMandatoryRelatedResources(expectedInstance);
        addedInstance.setMandatoryRelatedResources(mandatoryRelatedResources);

        Set<DerivatedRelatedResourceGroup<Instance>> addedDerivatedRelatedResourceGroups = addDerivatedRelatedResources(expectedInstance);
        addedInstance.setDerivatedRelatedResources(addedDerivatedRelatedResourceGroups);

        if(level==0 || ancestor==null){
            List<ERElement> addedElements = executeAdd();
            addedInstance.addAddedElements(addedElements);
        }

        /*
         * New instances must be created at the end otherwise if the analyser attempts to create
         * a relation between a new instance and another resource added to the context
         * it will fail because we don't have them in the target context.
         */
        Callable<List<IdentifiableElement>> callable = new Callable<List<IdentifiableElement>>() {
            @Override
            public List<IdentifiableElement> call() throws WebApplicationException, ResourceRegistryException {
                return createNewInstances();
            }
        };

        Secret secret = targetContext.getSecret();
        if(secret==null){
            StringBuffer error = new StringBuffer();
            error.append("Secret not found for ");
            error.append(targetContext.getContextFullPath());
            error.append(" UUID ");
            error.append(targetContext.getUUID());
            logger.error(error.toString());
            throw new InternalServerErrorException(error.toString());
        }

        try {
            List<IdentifiableElement> createdInstances = AuthorizedTasks.executeSafely(callable, secret);
            addedInstance.setNewInstances(createdInstances);
        } catch (WebApplicationException | ResourceRegistryException e) {
            StringBuffer error = new StringBuffer();
            error.append("Error while creating new instances");
            logger.error(error.toString(), e);
            throw e;
        } catch (Throwable t) {
            throw new InternalServerErrorException("Error while creating new instances", t);
        }
        
        return addedInstance;
    }

    public I remove() throws WebApplicationException {
        throw new UnsupportedOperationException("Method not implemented yet");
    }

}
