package org.gcube.resourcemanagement.analyser;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.ServiceLoader;
import java.util.Set;
import java.util.UUID;

import org.gcube.informationsystem.base.reference.AccessType;
import org.gcube.informationsystem.model.reference.entities.Resource;
import org.gcube.informationsystem.tree.Node;
import org.gcube.informationsystem.tree.Tree;
import org.gcube.informationsystem.types.knowledge.TypesKnowledge;
import org.gcube.informationsystem.types.reference.Type;
import org.gcube.resourcemanagement.resource.Instance;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author Luca Frosini (ISTI - CNR)
 */
public class InstanceMultipleAnalyserFactory {

    /**
     * Logger
     */
    protected static Logger logger = LoggerFactory.getLogger(InstanceMultipleAnalyserFactory.class);

    protected static Map<String, List<InstanceAnalyser<Resource, Instance>>> instances;
    protected static Map<String, List<InstanceAnalyser<Resource, Instance>>> polymorphicInstances;

    protected static List<InstanceAnalyser<Resource, Instance>> sortInstanceAnalyserList(List<InstanceAnalyser<Resource, Instance>> list) {
        /* 
         * This line sorts the list of InstanceAnalyser<Resource, Instance> objects in descending order of priority.
         * The comparator compares the priority of two analyser, a1 and a2.
         * a2.getPriority() - a1.getPriority() ensures that analysers with lower priority values (e.g., 0) 
         * come before those with higher priority values.
         * This means that a priority value of 0 is considered the highest priority.
         */
        list.sort((a1, a2) -> a2.getPriority() - a1.getPriority());
        return list;
    }

    protected static void addInstanceAnalyser(String type, InstanceAnalyser<Resource, Instance> analyser, Map<String, List<InstanceAnalyser<Resource, Instance>>> instancesMap){
        List<InstanceAnalyser<Resource, Instance>> list = instancesMap.get(type);
        if(list==null){
            list = new ArrayList<>();
            instancesMap.put(type, list);
        }
        list.add(analyser);
        sortInstanceAnalyserList(list);
    }
    
    protected static void addPolymorphicInstanceAnalyser(String type, InstanceAnalyser<Resource, Instance> analyser){
        if(!analyser.polymorphic()){
            return;
        }
        TypesKnowledge tk = TypesKnowledge.getInstance();
        Tree<Type> resourceTree = tk.getModelKnowledge().getTree(AccessType.RESOURCE);
        Node<Type> node = resourceTree.getNodeByIdentifier(type);
        Set<Node<Type>> descendants = node.getDescendants();
        for (Node<Type> child : descendants) {
            String childTypeName = child.getIdentifier();
            if(instances.containsKey(childTypeName)){
                /*
                 * Not adding the polymorphic analyser if there is a specific analyser for the child type
                 * and removing the the list of polymorphic analysers if any
                 */
                if(polymorphicInstances.containsKey(childTypeName)){
                    polymorphicInstances.remove(childTypeName);
                }
                continue;
            }else{
                if(!polymorphicInstances.containsKey(childTypeName)){
                    addInstanceAnalyser(childTypeName, analyser, polymorphicInstances);
                }else{
                    evaluateBestInstanceAnalyser(childTypeName, analyser, resourceTree);
                }
            }
        }
    }

    protected static void evaluateBestInstanceAnalyser(String type, InstanceAnalyser<Resource, Instance> candidatedAnalyser, Tree<Type> resourceTree) {
        List<InstanceAnalyser<Resource, Instance>> currentAnalysers = polymorphicInstances.get(type);
        String typeOfCurrentAnalyser = currentAnalysers.get(0).getType();
        String typeOfCandidatedAnalyser = candidatedAnalyser.getType();
        
        if(typeOfCurrentAnalyser.compareTo(typeOfCandidatedAnalyser) == 0){
            // Adding the new analyser to the list of polymorphic analysers for the type
            addInstanceAnalyser(type, candidatedAnalyser, polymorphicInstances);        
        } else if(resourceTree.isChildOf(typeOfCurrentAnalyser, typeOfCandidatedAnalyser)){
            /*
            * If the type of the candidated analyser is a child of the type of the current analyser, 
            * the candidated analyser is the best one because it is more specific.
            */
            // Remove the current list of analysers because we found a more specific analyser
            polymorphicInstances.remove(type);
            // Add the candidated analyser because it is more specific
            addInstanceAnalyser(type, candidatedAnalyser, polymorphicInstances);
        }
    }
    
    protected static void analyseInstanceAnalyser(InstanceAnalyser<Resource, Instance> analyser){
        try {
            String name = analyser.getName();
            logger.debug("{} {} found", name, InstanceAnalyser.class.getSimpleName());
            String type = analyser.getType();
            addInstanceAnalyser(type, analyser, instances);
            addPolymorphicInstanceAnalyser(type, analyser);
        } catch (Exception e) {
            logger.error("{} {} not initialized correctly. It will not be used", analyser.getName(), InstanceAnalyser.class.getSimpleName());
        }
    }

    static {
        InstanceAnalyserFactory.instances = new HashMap<>();
        InstanceAnalyserFactory.polymorphicInstances = new HashMap<>();

        @SuppressWarnings("rawtypes")
        ServiceLoader<InstanceAnalyser> serviceLoader = ServiceLoader.load(InstanceAnalyser.class);
        for (@SuppressWarnings("rawtypes") InstanceAnalyser analyser : serviceLoader) {
            @SuppressWarnings("unchecked")
            InstanceAnalyser<Resource, Instance> instanceAnalyser = analyser;
            analyseInstanceAnalyser(instanceAnalyser);
        }

        if(logger.isTraceEnabled()){
            logger.trace("{} initialized with the following {}s:", InstanceAnalyserFactory.class.getSimpleName(), InstanceAnalyserFactory.class.getSimpleName());
            for (String type : instances.keySet()) {
                List<InstanceAnalyser<Resource, Instance>> analysers = instances.get(type);
                for (InstanceAnalyser<Resource, Instance> analyser : analysers) {
                    logger.trace("{} will be managed managed by {}", type, analyser.getName());
                }
            }

            logger.trace("{} initialized with the following polymorphic {}s:", InstanceAnalyserFactory.class.getSimpleName(), InstanceAnalyserFactory.class.getSimpleName());
            for (String type : polymorphicInstances.keySet()) {
                List<InstanceAnalyser<Resource, Instance>> analysers = polymorphicInstances.get(type);
                for (InstanceAnalyser<Resource, Instance> analyser : analysers) {
                    logger.trace("{} will be managed managed by {} because is polymorphic", type, analyser.getName());
                }
            }
        }
        
    }

    public static List<InstanceAnalyser<Resource, Instance>> getInstanceAnalysers(String type, UUID id) {
        List<InstanceAnalyser<Resource, Instance>> instanceAnalysers = new ArrayList<>();
        if (instances.containsKey(type)) {
            List<InstanceAnalyser<Resource, Instance>> analysers = instances.get(type);
            for (InstanceAnalyser<Resource, Instance> analyser : analysers) {
                try {
                    @SuppressWarnings("unchecked")
                    Class<InstanceAnalyser<Resource, Instance>> clz = (Class<InstanceAnalyser<Resource, Instance>>) analyser.getClass();
                    InstanceAnalyser<Resource, Instance> instanceAnalyser = clz.getDeclaredConstructor(String.class, UUID.class).newInstance(type, id);
                    instanceAnalysers.add(instanceAnalyser);
                } catch (Exception e) {
                logger.error("Error while creating {} instance", InstanceAnalyser.class.getSimpleName(), e);
                }
            }
        }else{

        }
        sortInstanceAnalyserList(instanceAnalysers);
        return instanceAnalysers;
    }

}
