package org.gcube.datacatalogue.grsf_manage_widget.server.manage;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import org.gcube.common.portal.PortalContext;
import org.gcube.datacatalogue.ckanutillibrary.server.DataCatalogue;
import org.gcube.datacatalogue.ckanutillibrary.server.DataCatalogueFactory;
import org.gcube.datacatalogue.ckanutillibrary.server.utils.UtilMethods;
import org.gcube.datacatalogue.common.Constants;
import org.gcube.datacatalogue.common.enums.Product_Type;
import org.gcube.datacatalogue.common.enums.Sources;
import org.gcube.datacatalogue.common.enums.Status;
import org.gcube.datacatalogue.grsf_manage_widget.client.GRSFManageWidgetService;
import org.gcube.datacatalogue.grsf_manage_widget.shared.ConnectedBean;
import org.gcube.datacatalogue.grsf_manage_widget.shared.ManageProductBean;
import org.gcube.datacatalogue.grsf_manage_widget.shared.SimilarGRSFRecord;
import org.gcube.datacatalogue.grsf_manage_widget.shared.SourceRecord;
import org.gcube.datacatalogue.grsf_manage_widget.shared.ex.NoGRSFRecordException;
import org.gcube.vomanagement.usermanagement.RoleManager;
import org.gcube.vomanagement.usermanagement.impl.LiferayRoleManager;
import org.gcube.vomanagement.usermanagement.model.GCubeRole;
import org.gcube.vomanagement.usermanagement.model.GCubeTeam;
import org.gcube.vomanagement.usermanagement.model.GatewayRolesNames;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.gwt.user.server.rpc.RemoteServiceServlet;

import eu.trentorise.opendata.jackan.model.CkanDataset;
import eu.trentorise.opendata.jackan.model.CkanPair;
import eu.trentorise.opendata.jackan.model.CkanResource;

/**
 * Endpoint for sending update records information to GRSF KnowledgeBase.
 * @author Costantino Perciante at ISTI-CNR (costantino.perciante@isti.cnr.it)
 */
public class GRSFNotificationService extends RemoteServiceServlet implements GRSFManageWidgetService{

	private static final long serialVersionUID = -4534905087994875893L;
	//private static final Log logger = LogFactoryUtil.getLog(GRSFNotificationService.class);
	private static final Logger logger = LoggerFactory.getLogger(GRSFNotificationService.class);

	/**
	 * Instanciate the ckan util library.
	 * Since it needs the scope, we need to check if it is null or not
	 * @param discoverScope if you want to the discover the utils library in this specified scope
	 * @return DataCatalogue object
	 * @throws Exception 
	 */
	public DataCatalogue getCatalogue(String discoverScope) throws Exception{
		String currentScope = Utils.getCurrentContext(getThreadLocalRequest(), true);
		DataCatalogue instance = null;
		try{
			String scopeInWhichDiscover = discoverScope != null && !discoverScope.isEmpty() ? discoverScope : currentScope;
			logger.debug("Discovering ckan utils library into scope " + scopeInWhichDiscover);
			instance = DataCatalogueFactory.getFactory().getUtilsPerScope(scopeInWhichDiscover);
		}catch(Exception e){
			logger.error("Unable to retrieve ckan utils. Error was " + e.toString());
			throw e;
		}
		return instance;
	}

	@Override
	public ManageProductBean getProductBeanById(String productIdentifier) throws Exception {


		ManageProductBean toReturn = null;

		if(!Utils.isIntoPortal()){

			toReturn = new ManageProductBean();
			toReturn.setCatalogueIdentifier(UUID.randomUUID().toString());
			List<ConnectedBean> connectTo = new ArrayList<>();
			connectTo.add(new ConnectedBean("91f1e413-dc9f-3b4e-b1c5-0e8560177253","Stock", 
					"aksldsam asd", "asdasjnk:fas", UUID.randomUUID().toString(), "http://data.d4science.org/ctlg/GRSF_Admin/91f1e413-dc9f-3b4e-b1c5-0e8560177253"));
			toReturn.setConnectTo(connectTo);
			toReturn.setGrsfDomain("Stock");
			toReturn.setGrsfType("Assessment Unit");
			toReturn.setKnowledgeBaseIdentifier("91f1e413-dc9f-3b4e-b1c5-0e8560177253");
			toReturn.setShortName("Widow rockfish - US West Coast");
			toReturn.setShortNameUpdated("Widow rockfish - US West Coast");
			toReturn.setGrsfName("sebastes entomelas FAO 77 FAO 67");
			toReturn.setTraceabilityFlag(true);
			toReturn.setCurrentStatus(Status.Pending);
			toReturn.setSemanticIdentifier("asfis:WRO+fao:67;FAO");
			ArrayList<SourceRecord> sources = new ArrayList<SourceRecord>();
			sources.add(new SourceRecord("RAM", "http://www.google.it"));
			sources.add(new SourceRecord("FIRMS", "http://www.google.it"));
			sources.add(new SourceRecord("FishSource", "http://www.google.it"));
			toReturn.setSources(sources);
			List<SimilarGRSFRecord> similarGrsfRecords = new ArrayList<SimilarGRSFRecord>();
			similarGrsfRecords.add(new SimilarGRSFRecord("same species overlapping water areas", 
					Utils.getDatasetKnowledgeBaseIdFromUrl("http://data.d4science.org/ctlg/GRSF_Admin/1a03d6d8-002d-39b9-9255-a954c8ee2bf0")
					,"unknown:ONCORHYNCHUS GORBUSCHA+unknown:USA-AKSTATE-KELPB", 
					"Pink Salmon Kelp By (District112)", 
					"http://data.d4science.org/ctlg/GRSF_Admin/1a03d6d8-002d-39b9-9255-a954c8ee2bf0"));
			similarGrsfRecords.add(new SimilarGRSFRecord("same species overlapping water areas 2", 
					Utils.getDatasetKnowledgeBaseIdFromUrl("http://data.d4science.org/ctlg/GRSF_Admin/1a03d6d8-002d-39b9-9255-a954c8ee2bf0"),
					"unknown:ONCORHYNCHUS GORBUSCHA+unknown:USA-AKSTATE-KELPB", 
					"Pink Salmon Kelp By (District112) 2", 
					"http://data.d4science.org/ctlg/GRSF_Admin/1a03d6d8-002d-39b9-9255-a954c8ee2bf0"));
			toReturn.setSimilarGrsfRecords(similarGrsfRecords);

		}else{

			// retrieve scope per current portlet url
			String scopePerCurrentUrl = Utils.getScopeFromClientUrl(getThreadLocalRequest());
			DataCatalogue catalogue = getCatalogue(scopePerCurrentUrl);
			String username = Utils.getCurrentUser(getThreadLocalRequest()).getUsername();
			CkanDataset record = catalogue.getDataset(productIdentifier, catalogue.getApiKeyFromUsername(username));

			// it cannot be enabled in this case ...
			if(record == null)
				throw new Exception("Unable to retrieve information for the selected record, sorry");
			else{

				logger.debug("Trying to fetch record....");

				// check it is a grsf record (Source records have a different System Type)
				String systemType = record.getExtrasAsHashMap().get(Constants.SYSTEM_TYPE_CUSTOM_KEY);
				if(systemType == null || systemType.isEmpty() || systemType.equals(Constants.SYSTEM_TYPE_FOR_SOURCES_VALUE))
					throw new NoGRSFRecordException("This is not a GRSF Record");

				// get extras as hashmap and pairs
				List<CkanPair> extrasAsPairs = record.getExtras();

				// fetch map for namespaces
				Map<String, String> fieldsNamespacesMap = Utils.getFieldToFieldNameSpaceMapping(getThreadLocalRequest().getSession(),
						record.getExtrasAsHashMap().get(Constants.DOMAIN_CUSTOM_KEY).contains(Product_Type.STOCK.getOrigName()) ? Constants.GENERIC_RESOURCE_NAME_MAP_KEY_NAMESPACES_STOCK
								: Constants.GENERIC_RESOURCE_NAME_MAP_KEY_NAMESPACES_FISHERY);

				Map<String, List<String>> extrasWithoutNamespaces = Utils.replaceFieldsKey(extrasAsPairs, fieldsNamespacesMap);
				// get extras fields (wrt the mandatory ones) to show in the management panel TODO
				//			Utils.getExtrasToShow(); 
				String catalogueIdentifier = record.getId();
				String status = extrasWithoutNamespaces.get(Constants.STATUS_OF_THE_GRSF_RECORD_CUSTOM_KEY).get(0);
				String uuidKB = extrasWithoutNamespaces.get(Constants.UUID_KB_CUSTOM_KEY).get(0);
				String grsfDomain = extrasWithoutNamespaces.get(Constants.DOMAIN_CUSTOM_KEY).get(0);

				logger.debug(Constants.DOMAIN_CUSTOM_KEY + " is " + grsfDomain);

				if(status == null || uuidKB == null)
					throw new Exception("Some information is missing in this record: Status = " + status + ", knowledge base uuid = " + uuidKB + 
							", and grsf domain is = " + grsfDomain);

				String semanticId = extrasWithoutNamespaces.get(Constants.GRSF_SEMANTIC_IDENTIFIER_CUSTOM_KEY).get(0);
				String shortName = extrasWithoutNamespaces.get(Constants.SHORT_NAME_CUSTOM_KEY).get(0);
				String grsfType = extrasWithoutNamespaces.get(Constants.GRSF_TYPE_CUSTOM_KEY).get(0);
				String grsfName = extrasWithoutNamespaces.get(grsfDomain.contains(Product_Type.STOCK.getOrigName()) ? 
						Constants.STOCK_NAME_CUSTOM_KEY : Constants.FISHERY_NAME_CUSTOM_KEY).get(0);
				boolean traceabilityFlag = extrasWithoutNamespaces.get(Constants.TRACEABILITY_FLAG_CUSTOM_KEY).get(0).equalsIgnoreCase("true");

				// Get similar GRSF records, if any (each of which should have name, description, url and id(i.e semantic identifier))
				List<String> similarGrsfRecordsAsStrings = extrasWithoutNamespaces.containsKey(Constants.SIMILAR_GRSF_RECORDS_CUSTOM_KEY) ? extrasWithoutNamespaces.get(Constants.SIMILAR_GRSF_RECORDS_CUSTOM_KEY): null;

				List<SimilarGRSFRecord> similarRecords = new ArrayList<SimilarGRSFRecord>(0);
				if(similarGrsfRecordsAsStrings != null){
					for (String similarGRSFRecord : similarGrsfRecordsAsStrings) {
						similarRecords.add(Utils.similarGRSFRecordFromJson(similarGRSFRecord));
					}
				}

				logger.debug("SimilarGRSFRecords are " + similarRecords);

				// get connected records
				List<String> connectedBeansAsStrings =  extrasWithoutNamespaces.containsKey(Constants.CONNECTED_CUSTOM_KEY) ? extrasWithoutNamespaces.get(Constants.CONNECTED_CUSTOM_KEY): null;

				List<ConnectedBean> connectedBeans = new ArrayList<ConnectedBean>(0);
				if(connectedBeansAsStrings != null){
					for (String connectedBean : connectedBeansAsStrings) {
						connectedBeans.add(Utils.connectedBeanRecordFromJson(connectedBean, uuidKB, grsfDomain, catalogue));
					}
				}

				logger.debug("Connected records are " + connectedBeans);

				// Get sources
				List<CkanResource> resources = record.getResources();
				List<SourceRecord> sources = new ArrayList<SourceRecord>(3);
				for (CkanResource ckanResource : resources) {
					if(Sources.getListNames().contains(ckanResource.getName()))
						sources.add(new SourceRecord(ckanResource.getName(), ckanResource.getUrl()));
				}

				// set the values
				toReturn = new ManageProductBean(semanticId, catalogueIdentifier, uuidKB, grsfType, 
						grsfDomain, grsfName, shortName, traceabilityFlag, Status.fromString(status), null, 
						null, null, sources, similarRecords, connectedBeans);


			}
		}

		logger.info("Returning item bean " + toReturn);
		return toReturn;
	}

	@Override
	public boolean isAdminUser() {
		try{

			Boolean inSession = (Boolean)getThreadLocalRequest().getSession().getAttribute(Constants.GRSF_ADMIN_SESSION_KEY);

			if(inSession != null)
				return inSession;
			else{

				if(!Utils.isIntoPortal()){
					getThreadLocalRequest().getSession().setAttribute(Constants.GRSF_ADMIN_SESSION_KEY, true);
					return true;
				}

				PortalContext pContext = PortalContext.getConfiguration();
				RoleManager roleManager = new LiferayRoleManager();
				String username =  pContext.getCurrentUser(getThreadLocalRequest()).getUsername();
				long userId = pContext.getCurrentUser(getThreadLocalRequest()).getUserId();
				long groupId = pContext.getCurrentGroupId(getThreadLocalRequest());
				List<GCubeRole> vreRoles = roleManager.listRolesByUserAndGroup(userId, groupId);
				List<GCubeTeam> teamRoles = new LiferayRoleManager().listTeamsByUserAndGroup(userId, groupId);
				boolean toSetInSession = false;
				for (GCubeTeam team : teamRoles) {
					if(team.getTeamName().equals(Constants.GRSF_CATALOGUE_MANAGER_ROLE)){
						logger.info("User " + username + " is " + Constants.GRSF_CATALOGUE_MANAGER_ROLE);
						toSetInSession = true;
						break;
					}
				}

				if(!toSetInSession)
					for (GCubeRole gCubeTeam : vreRoles) {
						if(gCubeTeam.getRoleName().equals(GatewayRolesNames.VRE_MANAGER.getRoleName())){
							logger.info("User " + username + " is " + GatewayRolesNames.VRE_MANAGER.getRoleName());
							toSetInSession = true;
							break;
						}
					}

				getThreadLocalRequest().getSession().setAttribute(Constants.GRSF_ADMIN_SESSION_KEY, toSetInSession);
				return toSetInSession;
			}
		}catch(Exception e){
			logger.error("Failed to check if the user has team " + Constants.GRSF_CATALOGUE_MANAGER_ROLE 
					+ " or " + GatewayRolesNames.VRE_MANAGER.getRoleName() +"!", e);
		}
		return false;
	}

	@Override
	public boolean checkSemanticIdentifierExists(String semanticIdentifier)
			throws Exception {

		return getDataset(semanticIdentifier) != null;
	}

	@Override
	public boolean checkSemanticIdentifierExistsInDomain(String semanticIdentifier, String domain)
			throws Exception {


		CkanDataset dataset = getDataset(semanticIdentifier);

		// look for the right domain this time
		List<CkanPair> extrasAsPairs = dataset.getExtras();

		for (CkanPair ckanPair : extrasAsPairs) {
			if(ckanPair.getKey().contains(Constants.DOMAIN_CUSTOM_KEY)){
				return ckanPair.getValue().equalsIgnoreCase(domain);
			}
		}

		return false;
	}

	private CkanDataset getDataset(String semanticIdentifier) throws Exception{

		String scopePerCurrentUrl = Utils.getScopeFromClientUrl(getThreadLocalRequest());
		DataCatalogue catalogue = getCatalogue(scopePerCurrentUrl);
		String username = Utils.getCurrentUser(getThreadLocalRequest()).getUsername();
		CkanDataset dataset = Utils.getRecordBySemanticIdentifier(semanticIdentifier, catalogue, catalogue.getApiKeyFromUsername(username));
		return dataset;

	}


	@Override
	public boolean checkIdentifierExists(String id)
			throws Exception {
		String scopePerCurrentUrl = Utils.getScopeFromClientUrl(getThreadLocalRequest());
		DataCatalogue catalogue = getCatalogue(scopePerCurrentUrl);
		String username = Utils.getCurrentUser(getThreadLocalRequest()).getUsername();
		return catalogue.getDataset(id, catalogue.getApiKeyFromUsername(username)) != null;
	}

	@Override
	public boolean checkIdentifierExistsInDomain(String id,
			String domain) throws Exception {
		String scopePerCurrentUrl = Utils.getScopeFromClientUrl(getThreadLocalRequest());
		DataCatalogue catalogue = getCatalogue(scopePerCurrentUrl);
		String username = Utils.getCurrentUser(getThreadLocalRequest()).getUsername();
		CkanDataset dataset = catalogue.getDataset(id, catalogue.getApiKeyFromUsername(username));
		if(dataset == null)
			return false;

		List<CkanPair> extrasAsPairs = dataset.getExtras();

		for (CkanPair ckanPair : extrasAsPairs) {
			if(ckanPair.getKey().contains(Constants.DOMAIN_CUSTOM_KEY)){
				return ckanPair.getValue().equalsIgnoreCase(domain);
			}
		}

		return false;
	}

	@Override
	public String notifyProductUpdate(ManageProductBean bean) throws Exception{

		logger.info("Creating notification for the bean " + bean + " to send to the knowledge base");
		try{

			String context = Utils.getScopeFromClientUrl(getThreadLocalRequest());
			DataCatalogue catalogue = getCatalogue(context);

			// check if the base url of the service is in session
			String keyPerContext = UtilMethods.concatenateSessionKeyScope(Constants.GRSF_UPDATER_SERVICE, context);
			String baseUrl = (String)getThreadLocalRequest().getSession().getAttribute(keyPerContext);
			if(baseUrl == null ||  baseUrl.isEmpty()){
				baseUrl = Utils.discoverEndPoint(context);
				getThreadLocalRequest().getSession().setAttribute(keyPerContext, baseUrl);
			}
			return Utils.updateRecord(baseUrl, bean, catalogue, Utils.getCurrentUser(getThreadLocalRequest()).getUsername());

		}catch(Exception e){
			logger.error("Unable to update the product.." + e.getMessage());
			throw e;
		}
	}

}
