package org.gcube.portlets.user.homelibrary.jcr.workspace.folder.items;


import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.InputStream;

import javax.jcr.Binary;
import javax.jcr.Node;
import javax.jcr.PathNotFoundException;
import javax.jcr.Property;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.ValueFormatException;
import javax.jcr.lock.LockException;
import javax.jcr.nodetype.ConstraintViolationException;
import javax.jcr.version.VersionException;

import org.apache.jackrabbit.util.Text;
import org.gcube.contentmanagement.blobstorage.transport.backend.RemoteBackendException;
import org.gcube.portlets.user.homelibrary.home.exceptions.InternalErrorException;
import org.gcube.portlets.user.homelibrary.home.workspace.folder.items.File;
import org.gcube.portlets.user.homelibrary.jcr.repository.JCRRepository;
import org.gcube.portlets.user.homelibrary.jcr.repository.external.GCUBEStorage;
import org.gcube.portlets.user.homelibrary.util.MimeTypeUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JCRFile implements File {
	
	private static Logger logger = LoggerFactory.getLogger(JCRFile.class);
	
	public static final String MIME_TYPE 				= 	"jcr:mimeType";
	public static final String DATA 	  				= 	"jcr:data";
	public static final String SIZE 	 				= 	"hl:size";
	
	public static final String REMOTE_STORAGE_PATH 		=  	"hl:remotePath";
	
	protected String nodeId;
	private String mimeType;
	private long size;
	
	public JCRFile(Node node) throws RepositoryException  {
		
		this.nodeId = node.getIdentifier();
		
		//FIXME temporarily fix for file stored without mimeType
		try {
			this.mimeType = node.getProperty(MIME_TYPE).getString();
		} catch(Exception e) {
			this.mimeType = MimeTypeUtil.BINARY_MIMETYPE;
		}
			
		this.size = node.getProperty(SIZE).getLong();
		
	}
	
	public JCRFile(Node node, String mimeType, InputStream data) throws RepositoryException, RemoteBackendException {
		
//		String remotePath = Text.getRelativeParent(node.getPath(), 2)
//				+ UUID.randomUUID().toString();
		
		String remotePath = Text.getRelativeParent(node.getPath(), 1);
		
		//set public link
		
		String url = GCUBEStorage.putStream(data, remotePath);	
		
		logger.trace("GCUBEStorage URL : " + url);

		// Store url as byte stream in jcr:data binary property
		ByteArrayInputStream  binaryUrl = new ByteArrayInputStream(url.getBytes());
		Binary binary = node.getSession().getValueFactory().createBinary(binaryUrl);
		node.setProperty(DATA,binary);
		
		setProperties(node, remotePath, mimeType);
		
		node.setProperty(REMOTE_STORAGE_PATH, remotePath);
		
		this.mimeType = mimeType;
	}
	
	private void setProperties(Node node, String remotePath, String mimeType) throws ValueFormatException, VersionException, LockException, ConstraintViolationException, RepositoryException {

		this.size = GCUBEStorage.getRemoteFileSize(remotePath);		
		node.setProperty(SIZE, size);
		node.setProperty(MIME_TYPE, mimeType);

	}
	
		
	public void save(Node node) throws RepositoryException {
		this.nodeId = node.getIdentifier();
	}
	
	public JCRFile(Node node, InputStream data) throws RepositoryException, RemoteBackendException { 
		this(node,MimeTypeUtil.getMimeType(data), data);
	}

	@Override
	public String getName() throws InternalErrorException {
		Session session = JCRRepository.getSession();
		try {
			Node node = session.getNodeByIdentifier(nodeId);
			return node.getName();
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}

	@Override
	public String getMimeType() {
		return mimeType;
	}
	
		
//	protected InputStream getRemoteData(URL url) throws IOException {
//		return GCUBEStorage.getInputStream(url);
//	}

	@Override
	public InputStream getData() throws InternalErrorException {
		String remotePath = null;
		Session session = JCRRepository.getSession();
		try {
			Node node = session.getNodeByIdentifier(nodeId);
			
			
			try {
				remotePath = node.getProperty(REMOTE_STORAGE_PATH).getString();
			} catch (PathNotFoundException e) {
				logger.trace("Old retrieve content method");
			}
			
			
			// The remote data is stored on GCUBE storage.
			if (remotePath != null) {
				logger.trace("Content retrieved from remote storage...");
				return GCUBEStorage.getRemoteFile(remotePath);
			} else {
				
				// Move binary content on remote gcube-storage
				Binary data = node.getProperty(DATA).getBinary();	
				try {
					
					logger.trace("Content moved to remote storage");
//					remotePath = Text.getRelativeParent(node.getPath(), 2)
//							+ UUID.randomUUID().toString();
					
//					remotePath = Text.getRelativeParent(node.getPath(), 2)
//							+ node.getIdentifier().toString();
					
					remotePath = node.getPath();

					String url = GCUBEStorage.putStream(data.getStream(), remotePath);					
					
					logger.trace("New gcube storage url : " + url);
					
					// Convert url to byte stream in jcr:data binary property
					// mandatory for nt:resource node
					ByteArrayInputStream  binaryUrl = new ByteArrayInputStream(url.getBytes());
					Binary binary = node.getSession().getValueFactory().createBinary(binaryUrl);
					
					// Store URI and remote storage path 
					node.setProperty(DATA,binary);
					node.setProperty(REMOTE_STORAGE_PATH, remotePath);
					session.save();
					
				} catch (Exception e) {
					logger.error("The item doesn't contain " 
							+ REMOTE_STORAGE_PATH + " property");
				}
				return data.getStream();
			}
			
		} catch (Exception e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}

	@Override
	public long getLength() throws InternalErrorException {
		return size;
	}
	
	@Override
	public String getPublicLink() throws InternalErrorException {
	
		String remotePath = null;
		Session session = JCRRepository.getSession();
		try {
			Node node = session.getNodeByIdentifier(nodeId);
			try {
				remotePath = node.getProperty(REMOTE_STORAGE_PATH).getString();								
			} catch (PathNotFoundException e) {
			
				logger.trace("No public link for file: " + node.getPath());
			} 
			return GCUBEStorage.getPublicLink(remotePath);
		} catch (Exception e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
		
	}

	@Override
	public void getHardLink(String destPath) throws InternalErrorException {
		String remotePath = null;
		Session session = JCRRepository.getSession();
		try {
			Node node = session.getNodeByIdentifier(nodeId);
			try {
				remotePath = node.getProperty(REMOTE_STORAGE_PATH).getString();								
			} catch (PathNotFoundException e) {
			
				logger.trace("No public link for file: " + node.getPath());
			} 
			GCUBEStorage.createHardLink(remotePath, destPath);
		} catch (Exception e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}

	@Override
	public void updateInfo() throws InternalErrorException {
//		System.out.println("update in JCRFile");
		String remotePath = null;
		Session session = JCRRepository.getSession();
		try {
			Node node = session.getNodeByIdentifier(nodeId);
			try {
				remotePath = node.getProperty(REMOTE_STORAGE_PATH).getString();		
				
				InputStream data = new BufferedInputStream(GCUBEStorage.getRemoteFile(remotePath));							
				
				String mimeType = MimeTypeUtil.getMimeType(data);
			
				setProperties(node, remotePath, mimeType);
				session.save();
			} catch (PathNotFoundException e) {	
				logger.trace("No public link for file: " + node.getPath());
			} 
		
		} catch (Exception e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}




}
