package org.gcube.portlets.user.homelibrary.jcr.importing;

import static org.gcube.contentmanagement.gcubedocumentlibrary.projections.Projections.ALTERNATIVE;
import static org.gcube.contentmanagement.gcubedocumentlibrary.projections.Projections.ANNOTATION;
import static org.gcube.contentmanagement.gcubedocumentlibrary.projections.Projections.BYTESTREAM;
import static org.gcube.contentmanagement.gcubedocumentlibrary.projections.Projections.BYTESTREAM_URI;
import static org.gcube.contentmanagement.gcubedocumentlibrary.projections.Projections.METADATA;
import static org.gcube.contentmanagement.gcubedocumentlibrary.projections.Projections.PART;
import static org.gcube.contentmanagement.gcubedocumentlibrary.projections.Projections.alternative;
import static org.gcube.contentmanagement.gcubedocumentlibrary.projections.Projections.annotation;
import static org.gcube.contentmanagement.gcubedocumentlibrary.projections.Projections.document;
import static org.gcube.contentmanagement.gcubedocumentlibrary.projections.Projections.metadata;
import static org.gcube.contentmanagement.gcubedocumentlibrary.projections.Projections.opt;
import static org.gcube.contentmanagement.gcubedocumentlibrary.projections.Projections.part;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringWriter;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URLConnection;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import javax.jcr.Node;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import org.apache.commons.io.IOUtils;
import org.apache.jackrabbit.util.Text;
import org.gcube.common.core.scope.GCUBEScope;
import org.gcube.common.core.utils.logging.GCUBEClientLog;
import org.gcube.contentmanagement.contentmanager.stubs.model.protocol.URIs;
import org.gcube.contentmanagement.gcubedocumentlibrary.io.DocumentReader;
import org.gcube.contentmanagement.gcubedocumentlibrary.projections.AlternativeProjection;
import org.gcube.contentmanagement.gcubedocumentlibrary.projections.AnnotationProjection;
import org.gcube.contentmanagement.gcubedocumentlibrary.projections.DocumentProjection;
import org.gcube.contentmanagement.gcubedocumentlibrary.projections.MetadataProjection;
import org.gcube.contentmanagement.gcubedocumentlibrary.projections.PartProjection;
import org.gcube.contentmanagement.gcubedocumentlibrary.projections.Projection;
import org.gcube.contentmanagement.gcubedocumentlibrary.util.Collection;
import org.gcube.contentmanagement.gcubedocumentlibrary.util.Collections;
import org.gcube.contentmanagement.gcubedocumentlibrary.views.MetadataView;
import org.gcube.contentmanagement.gcubemodellibrary.elements.GCubeAlternative;
import org.gcube.contentmanagement.gcubemodellibrary.elements.GCubeAnnotation;
import org.gcube.contentmanagement.gcubemodellibrary.elements.GCubeDocument;
import org.gcube.contentmanagement.gcubemodellibrary.elements.GCubeElement;
import org.gcube.contentmanagement.gcubemodellibrary.elements.GCubeInnerElement;
import org.gcube.contentmanagement.gcubemodellibrary.elements.GCubeMetadata;
import org.gcube.contentmanagement.gcubemodellibrary.elements.GCubePart;
import org.gcube.contentmanagement.gcubemodellibrary.elements.InnerElements;
import org.gcube.portlets.user.homelibrary.home.exceptions.InternalErrorException;
import org.gcube.portlets.user.homelibrary.home.workspace.folder.items.ExternalUrl;
import org.gcube.portlets.user.homelibrary.home.workspace.folder.items.Query;
import org.gcube.portlets.user.homelibrary.home.workspace.folder.items.gcube.Document;
import org.gcube.portlets.user.homelibrary.home.workspace.folder.items.gcube.Metadata;
import org.gcube.portlets.user.homelibrary.home.workspace.folder.items.gcube.link.DocumentAlternativeLink;
import org.gcube.portlets.user.homelibrary.home.workspace.folder.items.gcube.link.DocumentPartLink;
import org.gcube.portlets.user.homelibrary.jcr.repository.JCRRepository;
import org.gcube.portlets.user.homelibrary.jcr.workspace.JCRFolderBulkCreatorManager;
import org.gcube.portlets.user.homelibrary.jcr.workspace.JCRWorkspaceFolder;
import org.gcube.portlets.user.homelibrary.jcr.workspace.folder.items.ContentType;
import org.gcube.portlets.user.homelibrary.jcr.workspace.folder.items.JCRFile;
import org.gcube.portlets.user.homelibrary.jcr.workspace.folder.items.gcube.JCRDocument;

import org.gcube.portlets.user.homelibrary.jcr.workspace.folder.items.gcube.link.JCRDocumentAlternativeLink;
import org.gcube.portlets.user.homelibrary.jcr.workspace.folder.items.gcube.link.JCRDocumentPartLink;
import org.gcube.portlets.user.homelibrary.util.WorkspaceUtil;


public class JCRWorkspaceFolderItemImporter implements Runnable{
	
	protected static final AlternativeProjection ALTERNATIVE_PROJECTION = alternative().allexcept(BYTESTREAM);
	protected static final AlternativeProjection ALTERNATIVE_WITH_CONTENT_PROJECTION = alternative().etc().with(opt(BYTESTREAM),opt(BYTESTREAM_URI));

	protected static final AnnotationProjection ANNOTATION_PROJECTION = annotation();
	protected static final AnnotationProjection ANNOTATION_WITH_CONTENT_PROJECTION = annotation().etc().with(opt(BYTESTREAM),opt(BYTESTREAM_URI));

	protected static final MetadataProjection METADATA_PROJECTION = metadata();
	protected static final MetadataProjection METADATA_WITH_CONTENT_PROJECTION = metadata().etc().with(opt(BYTESTREAM),opt(BYTESTREAM_URI));

	protected static final PartProjection PART_PROJECTION = part();
	protected static final PartProjection PART_WITH_CONTENT_PROJECTION = part().etc().with(opt(BYTESTREAM),opt(BYTESTREAM_URI));

	protected static final DocumentProjection DOCUMENT_PROJECTION = document()
	.with(opt(ALTERNATIVE, alternative().allexcept(BYTESTREAM)))
	.with(opt(ANNOTATION, annotation().allexcept(BYTESTREAM)))
	.with(opt(METADATA, metadata().allexcept(BYTESTREAM)))
	.with(opt(PART, part().allexcept(BYTESTREAM)))
	.allexcept(BYTESTREAM);

	protected static final DocumentProjection DOCUMENT_WITH_CONTENT_PROJECTION = document()
	.with(opt(ALTERNATIVE, alternative().allexcept(BYTESTREAM)))
	.with(opt(ANNOTATION, annotation().allexcept(BYTESTREAM)))
	.with(opt(PART, part().allexcept(BYTESTREAM))).etc();

	private List<ImportContentManagerItemRequest> contentItemRequests;
	private List<ImportUrlRequest> urlRequests;
	private List<ImportQueryRequest> queryRequests;
	
	private Map<String, DocumentReader> readersCache;
	private Map<String, Collection> collectionsCache;
	private List<MetadataView> metadataViewsCache;
	
	private final JCRRepository contentManager;
	private final String folderBulkCreatorId;
	private final GCUBEScope scope;
	private final JCRFolderBulkCreatorManager manager;
	
	private final String STATUS 				= "hl:status";
	private final String FAILURES				= "hl:failures";
	
	private int failures;
	private int status;
	private final int totalRequests;
	
	private final JCRWorkspaceFolder folder;
	private GCUBEClientLog logger;
	

	public JCRWorkspaceFolderItemImporter(JCRFolderBulkCreatorManager manager, String folderBulkCreatorId,
			List<ImportRequest> requests, JCRWorkspaceFolder folder) throws InternalErrorException {
		super();
		
		this.logger = new GCUBEClientLog(this);
	
		this.manager = manager;
		this.folderBulkCreatorId = folderBulkCreatorId;
		this.contentManager = folder.getWorkspace().getRepository();
		this.scope = folder.getWorkspace().getOwner().getScope();
		this.folder = folder;
		
		this.status = 0;
		this.failures = 0;
		
		this.contentItemRequests = new LinkedList<ImportContentManagerItemRequest>();
		this.urlRequests = new LinkedList<ImportUrlRequest>();
		this.queryRequests = new LinkedList<ImportQueryRequest>();
		this.collectionsCache = new LinkedHashMap<String, Collection>();
		
		this.readersCache = new HashMap<String, DocumentReader>();

		this.totalRequests = requests.size();
		for (ImportRequest request:requests){
			switch(request.getType()){
				case CONTENT_MANAGER_ITEM: contentItemRequests.add((ImportContentManagerItemRequest) request);break;
				case URL: urlRequests.add((ImportUrlRequest) request); break;
				case QUERY: queryRequests.add((ImportQueryRequest) request); break;
			}
		}
	}
	
	private void upgradeStatus() {
		Session session = null;
		try {
			session = JCRRepository.getSession();
			Node node = contentManager.getRootFolderBulkCreators(session)
			.getNode(folderBulkCreatorId);
			node.setProperty(STATUS, ++status);
			session.save();
		}  catch (Exception e) {
			logger.error("Status not set", e);
		} finally {
			if(session != null)
				session.logout();
		}
	}
	
	private void upgradeFailures()  {
		Session session = null;
		try {
			session = JCRRepository.getSession();
			Node node = contentManager.getRootFolderBulkCreators(session).
			getNode(folderBulkCreatorId);
			node.setProperty(FAILURES, ++failures);
			session.save();
		}  catch (Exception e) {
			logger.error("Failure not set", e);
		} finally {
			if (session != null)
				session.logout();
		}
	}
	

	@Override
	public void run() {
	
		
		ExecutorService executorService = Executors.newFixedThreadPool(5);	
		for (ImportContentManagerItemRequest request: contentItemRequests) {
			
			final URI uri = request.getUri();	
			boolean contentIsLocal = contentIsLocal(uri);
			
			if (request.getItemType() == ContentManagerItemType.METADATA)		
				contentIsLocal = false;
			
			DocumentReader reader;
			final GCubeElement element;
			try {
				String collectionId = URIs.collectionID(uri);
				reader = getReader(collectionId);
				
				if (request.getItemType() != ContentManagerItemType.DOCUMENT){
					Projection<? extends GCubeInnerElement, ?> projection = null;
					switch (request.getItemType()) {
					case ALTERNATIVE:projection = contentIsLocal?ALTERNATIVE_PROJECTION:ALTERNATIVE_WITH_CONTENT_PROJECTION; break;
					case ANNOTATION:projection = contentIsLocal?ANNOTATION_PROJECTION:ANNOTATION_WITH_CONTENT_PROJECTION; break;
					case METADATA:projection = contentIsLocal?METADATA_PROJECTION:METADATA_WITH_CONTENT_PROJECTION; break;
					case PART:projection = contentIsLocal?PART_PROJECTION:PART_WITH_CONTENT_PROJECTION; break;
					}
					element = reader.resolve(uri, projection);
				} else {
					DocumentProjection projection = contentIsLocal?DOCUMENT_PROJECTION:DOCUMENT_WITH_CONTENT_PROJECTION;
					element = reader.get(URIs.documentID(uri), projection);
					logger.debug("Document uri: " + element.uri());
				}

			} catch (Exception e) {
				// Upgrade failure
				upgradeFailures();
				logger.error("Reader exception", e);
				continue;
			}
			
			
			if (contentIsLocal) { 
				importGCubeElement(uri, element, null);
				continue;
			} 
				
			if (element.bytestream()!=null) {
				importGCubeElement(uri, element, new ByteArrayInputStream(element.bytestream()));
				continue;
			}  
	
			if (element.bytestreamURI()!=null){
					executorService.execute(new Runnable() {
						@Override
						public void run() {
							
							InputStream inputStream = null;
							try {
								URLConnection connection = element.bytestreamURI().toURL().openConnection();
								inputStream = connection.getInputStream();
								importGCubeElement(uri, element, inputStream);								
							} catch (IOException e) {
								upgradeFailures();
								logger.error("Content retrieving failed",e);
								return;
							} finally {
									try {
										inputStream.close();
									} catch (Exception e) {
										logger.error("InputStream not closed");
									}
							}
						}
					});
			}
		}
		executorService.shutdown();
		
		for (ImportUrlRequest request:urlRequests) {

			String name = request.getUrl();
			name = Text.escapeIllegalJcrChars(name);
			try {
				name = WorkspaceUtil.getUniqueName(name, folder);
				ExternalUrl externalUrl = folder.createExternalUrlItem(name, "", request.getUrl());
				folder.getWorkspace().fireItemImportedEvent(externalUrl);
				upgradeStatus();
			} catch (Exception e) {
				upgradeFailures();
				continue;
			}
		}

		for (ImportQueryRequest request:queryRequests) {
			
			String name = request.getName();
			name = Text.escapeIllegalJcrChars(name);
			try {
				name = WorkspaceUtil.getUniqueName(request.getName(), folder);
				Query query = folder.createQueryItem(name, "", request.getQuery(), request.getQueryType());
				folder.getWorkspace().fireItemImportedEvent(query);
				upgradeStatus();
			} catch (Exception e) {
				upgradeFailures();
				continue;
			}
		}
		
	}

	private boolean contentIsLocal(URI uri) {
	
		Session session = null;
		JCRFile file = null;
		try {
			session = JCRRepository.getSession();
			file = folder.getWorkspace().
			getGCUBEDocumentContent(session, uri.toString(), ContentType.GENERAL);
		} catch (RepositoryException e) {
			logger.error("GCUBE element root node not exist", e);
		} catch (InternalErrorException e) {
			logger.error("Session refused");
		} finally {
			if(session != null)
				session.logout();
		}
		logger.debug("-------------- content " + uri.toString() +" is local " + (file != null));
		return (file != null);
	}
	
	private void importGCubeElement(URI uri, GCubeElement element, InputStream inputStream) {
		
		try {
			if (element instanceof GCubeDocument) {	
				logger.debug("The GCubeElement " + element + "is a GCubeDocument");
				HashMap<String,String> mapMetadata = new HashMap<String,String>(); 
				for (GCubeMetadata metadata:((GCubeDocument)element).metadata()){
					String schemaName = metadata.schemaName();
					String language = metadata.language();
					byte[] bytes = metadata.bytestream();

					if (schemaName != null && language != null && bytes != null) {
						String data = new String(bytes);
						mapMetadata.put(schemaName + "_" + language, data);
					}
				}
				JCRDocument document = (JCRDocument)importGCubeDocument(uri.toString(), element, mapMetadata, inputStream);
				upgradeStatus();
				
				List<DocumentAlternativeLink> alternatives = getDocumentAlternativeLinks(((GCubeDocument)element).uri().toString(),
						((GCubeDocument)element).alternatives());
				List<DocumentPartLink> parts = getDocumentPartLinks(((GCubeDocument)element).id(),
						((GCubeDocument)element).parts());
				
				Session session = null;
				try { 
					session = JCRRepository.getSession();
					document.setAlternatives(session, alternatives);
					document.setParts(session, parts);
				} finally { 
					session.logout();
				}
			}
			else if (element instanceof GCubeAlternative || element instanceof GCubePart) {
				importGCubeDocument(uri.toString(), element, new HashMap<String,String>(), inputStream);
				upgradeStatus();
			} else if (element instanceof GCubeMetadata) {
				importGCubeMetadata((GCubeMetadata)element, inputStream);
				upgradeStatus();
			} else if (element instanceof GCubeAnnotation) 
				upgradeStatus();
				return;
		} catch (Exception e) {
			upgradeFailures();
			logger.error(e);
		};
		
	}
	
	private void importGCubeMetadata(GCubeMetadata metadata, InputStream inputStream) throws Exception {
			
			String oid = metadata.uri().toString();
			String name = metadata.name();
			String schema = metadata.schemaName();
			String language = metadata.language();

			String collectionName = getMetadataViewName(schema, metadata.schemaURI(),
					language, metadata.uri());
			if (collectionName==null)
				collectionName = "unknown";

			StringWriter sw = new StringWriter();
			IOUtils.copy(inputStream, sw);
			String data = sw.toString();

			name = Text.escapeIllegalJcrChars(name);
			name = WorkspaceUtil.getUniqueName(name, folder);

			Metadata importedMetaData =  folder.createMetadataItem(name, "", oid, schema, language, data, collectionName);
			folder.getWorkspace().fireItemImportedEvent(importedMetaData);
		
	}
		
	private Document importGCubeDocument(String oid, GCubeElement document, HashMap<String,String> metadata, InputStream documentData) throws Exception {
			
			logger.debug("Import GCubeDocument " + document.uri());
			//FIXME check content management bug.
//			String oid = document.uri().toString();
			String name = document.name();
			String mimeType = document.mimeType();
			String collectionName = (getCollectionName(document.uri()) == null)?"Unknown":getCollectionName(document.uri());

			name = Text.escapeIllegalJcrChars(name);
			name = WorkspaceUtil.getUniqueName(name, folder);

			Document importedDocument = null;
			
			if (mimeType != null){
				if (mimeType.startsWith("image")){
					logger.debug("Import GCubeImageDocument");
					importedDocument = folder.createImageDocumentItem(name, "", oid, mimeType, documentData,
							metadata, new LinkedHashMap<String, String>(), collectionName);
				}else if (mimeType.equals("application/pdf")){
					logger.debug("Import GCubePdfDocument");
					importedDocument = folder.createPDFDocumentItem(name, "", oid, mimeType, documentData, 
							metadata, new LinkedHashMap<String,String>(), collectionName);
				}else if (mimeType.equals("text/uri-list")){
					logger.debug("Import GCubeUrlDocument");
					importedDocument = folder.createUrlDocumentItem(name, "", oid, mimeType, documentData,
							metadata, new LinkedHashMap<String,String>(), collectionName);
				}else{
					logger.debug("Import GCubeDocument");
					importedDocument = folder.createDocumentItem(name, "", oid, mimeType, documentData, 
							metadata, new LinkedHashMap<String, String>(), collectionName); 	
				}
			}else{
				importedDocument = folder.createDocumentItem(collectionName, "", oid, "n/a", documentData,
						metadata, new LinkedHashMap<String,String>(), collectionName);
			}

			folder.getWorkspace().fireItemImportedEvent(importedDocument);
			return importedDocument;
	}
	
	
	private List<DocumentAlternativeLink> getDocumentAlternativeLinks(String parentId,
			InnerElements<GCubeAlternative> alternatives) throws IllegalStateException, URISyntaxException {
		List<DocumentAlternativeLink> alternativeLinks = new LinkedList<DocumentAlternativeLink>();
		for (GCubeAlternative alternative:alternatives) {
			DocumentAlternativeLink alternativeLink = new JCRDocumentAlternativeLink(parentId, alternative.uri().toString(),
					alternative.name(), alternative.mimeType());
			alternativeLinks.add(alternativeLink);		
		}
		return alternativeLinks;
	}

	private List<DocumentPartLink> getDocumentPartLinks(String parentId,
			InnerElements<GCubePart> parts) throws IllegalStateException, URISyntaxException {
		List<DocumentPartLink> partLinks = new LinkedList<DocumentPartLink>();
		for (GCubePart part:parts) {
			DocumentPartLink partLink = new JCRDocumentPartLink(parentId, part.uri().toString(), part.name(), part.mimeType());
			partLinks.add(partLink);
		}
		return partLinks;
	}
	
	private String getCollectionName(URI uri) throws Exception {

		String collectionId = URIs.collectionID(uri);

		Collection collection = collectionsCache.get(collectionId);
		if (collection != null) {
			logger.debug("Collection name = " + collection.getName());
			return collection.getName();
		}
		
		try {
			logger.debug("Retrieve collection Name");
			List<Collection> foundCollections = Collections.findById(scope, collectionId);
			if (foundCollections.size() == 1) {
				logger.debug("Collection found");
				collection = foundCollections.get(0);
				collectionsCache.put(collection.getId(), collection);
				logger.debug("Collection name = " + collection.getName());
				return collection.getName();
			} 
		} catch(Exception e) {
			return null;
		}
		return null;
	}
	
	private String getMetadataViewName(String schemaName, URI schemaUri, String language, 
			URI uri) throws URISyntaxException {
		
		String collectionId = URIs.collectionID(uri);
		MetadataView metadataView = findView(collectionId, schemaName, schemaUri, language);
		if (metadataView != null)
			return metadataView.name();
		return null;
	}

	private MetadataView findView(String collectionId, String schemaName, URI schemaUri,
			String language) {
		
		for (MetadataView view:metadataViewsCache){
			if (view.collectionId().equals(collectionId) 
					&& view.schemaName().equals(schemaName)
					&& view.schemaURI().equals(schemaUri)
					&& view.language().equals(language)) return view;
		}

		//try to find on IS
		MetadataView metadataView = new MetadataView(scope);
		metadataView.setCollectionId(collectionId);
		metadataView.setSchemaName(schemaName);
		metadataView.setSchemaURI(schemaUri);
		metadataView.setLanguage(new Locale(language));

		try{
			List<MetadataView> foundViews = metadataView.findSimilar();
			if (foundViews.size() == 1) {
				MetadataView view = foundViews.get(0);
				metadataViewsCache.add(view);
				return view;
			}
		} catch(Exception e) {
			return null;
		}

		return null;
	}
	
	private DocumentReader getReader(String collectionId) throws Exception{
		
		DocumentReader reader = readersCache.get(collectionId);
		if (reader == null) {
			logger.debug("Return a new reader for scope " + scope);
			reader = new DocumentReader(collectionId, scope);
			readersCache.put(collectionId, reader);
		}
		return reader;
	}

}
