package org.gcube.portlets.user.homelibrary.jcr.workspace;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import javax.jcr.Node;
import javax.jcr.PathNotFoundException;
import javax.jcr.Property;
import javax.jcr.PropertyIterator;
import javax.jcr.RepositoryException;
import javax.jcr.Session;

import org.apache.jackrabbit.util.Text;
import org.gcube.portlets.user.homelibrary.home.exceptions.InternalErrorException;
import org.gcube.portlets.user.homelibrary.home.workspace.Properties;
import org.gcube.portlets.user.homelibrary.home.workspace.folder.items.gcube.DocumentMetadata;
import org.gcube.portlets.user.homelibrary.jcr.content.JCRRepository;
import org.gcube.portlets.user.homelibrary.jcr.workspace.folder.items.gcube.JCRDocumentMetadata;

public class JCRProperties implements Properties {
	
	private static final String PROPERTIES = "hl:metadata";
	private String identifier;
	
	public JCRProperties(Node itemNode) throws RepositoryException {
		this.identifier = itemNode.getIdentifier();
	}
	
	@Override
	public String getId() throws InternalErrorException {
		Session session = JCRRepository.getSession();
		try {
			Node node = session.getNodeByIdentifier(identifier);
			return node.getNode(PROPERTIES).getIdentifier();
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}	
	}

	@Override
	public String getPropertyValue(String propertyName) throws InternalErrorException {
		Session session = JCRRepository.getSession();
		try {
			Node itemNode = session.getNodeByIdentifier(identifier);
			return itemNode.getNode(PROPERTIES).getProperty(propertyName).getString();
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}

	@Override
	public Map<String, String> getProperties() throws InternalErrorException {
		
		Session session = JCRRepository.getSession();
		try {
			Node itemNode = session.getNodeByIdentifier(identifier);
			Map<String,String> map = new HashMap<String,String>();
			
			for (PropertyIterator iterator = itemNode.getNode(PROPERTIES).getProperties();
					iterator.hasNext();) {
				Property property = iterator.nextProperty();
				if(!property.getName().startsWith("jcr:")) {
					String unescapeName = Text.unescape(property.getName());
					map.put(unescapeName,
							property.getValue().toString());
				}
			}
			return Collections.unmodifiableMap(map);
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}

	@Override
	public void addProperty(String name, String value) throws InternalErrorException {
		
		Session session = JCRRepository.getSession();
		try {
			Node itemNode = session.getNodeByIdentifier(identifier);
			String escapeName = Text.escapeIllegalJcrChars(name);
			itemNode.getNode(PROPERTIES).setProperty(escapeName, value);
			session.save();
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
		
	}

}
