/**
 * 
 */
package org.gcube.portlets.user.gcubegisviewer.server;

import java.awt.image.BufferedImage;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLConnection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Timer;
import java.util.TimerTask;

import javax.imageio.ImageIO;
import javax.servlet.ServletException;
import javax.servlet.http.HttpSession;

import org.apache.log4j.Logger;
import org.gcube.application.framework.core.session.ASLSession;
import org.gcube.application.framework.core.session.SessionManager;
import org.gcube.common.core.contexts.GHNContext;
import org.gcube.common.core.informationsystem.client.AtomicCondition;
import org.gcube.common.core.informationsystem.client.ISClient;
import org.gcube.common.core.informationsystem.client.queries.GCUBEGenericResourceQuery;
import org.gcube.common.core.informationsystem.client.queries.GCUBERuntimeResourceQuery;
import org.gcube.common.core.resources.GCUBEGenericResource;
import org.gcube.common.core.resources.GCUBERuntimeResource;
import org.gcube.common.core.resources.runtime.AccessPoint;
import org.gcube.common.core.scope.GCUBEScope;
import org.gcube.common.core.scope.GCUBEScope.MalformedScopeExpressionException;
import org.gcube.common.core.utils.logging.GCUBEClientLog;
import org.gcube.common.geoserverinterface.GeoCaller;
import org.gcube.portlets.user.gcubegisviewer.client.GCubeGisViewerService;
import org.gcube.portlets.user.gcubegisviewer.client.GCubeGisViewerServiceException;
import org.gcube.portlets.user.gisviewer.client.Constants;
import org.gcube.portlets.user.gisviewer.server.GisViewerServiceImpl;
import org.gcube.portlets.user.gisviewer.server.GisViewerServiceParameters;
import org.gcube.portlets.user.gisviewer.server.MapGeneratorUtils;
import org.gcube.portlets.user.homelibrary.home.HomeLibrary;
import org.gcube.portlets.user.homelibrary.home.workspace.Workspace;
import org.gcube.portlets.user.homelibrary.home.workspace.WorkspaceFolder;
import org.gcube.portlets.user.homelibrary.home.workspace.exceptions.InsufficientPrivilegesException;
import org.gcube.portlets.user.homelibrary.home.workspace.exceptions.ItemAlreadyExistException;
import org.gcube.portlets.user.homelibrary.util.WorkspaceUtil;

/**
 * @author "Federico De Faveri defaveri@isti.cnr.it"
 *
 */
public class GCubeGisViewerServletImpl extends GisViewerServiceImpl implements GCubeGisViewerService {

	private static final long serialVersionUID = 804152795418658243L;

	public static final String USERNAME_ATTRIBUTE = "username";
	public static final String GEOSERVER_RESOURCE_NAME = "GeoServer";
	public static final String GEONETWORK_RESOURCE_NAME = "GeoNetwork";
	public static final String TRANSECT_RESOURCE_NAME = "Transect";
	public static final String GEOCALLER_ATTRIBUTE_NAME = "GEOCALLER";
	public static final long CACHE_REFRESH_TIME = 10*60*1000;

	public static GCUBEClientLog log = new GCUBEClientLog(GCubeGisViewerServletImpl.class);

	protected Map<GCUBEScope, GisViewerServiceParameters> parametersCache = new HashMap<GCUBEScope, GisViewerServiceParameters>();
	protected Logger logger = Logger.getLogger(GCubeGisViewerServletImpl.class);
	protected Timer timer;

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void init() throws ServletException {
		super.init();

		timer = new Timer(true);
		timer.schedule(new TimerTask() {
			@Override
			public void run() {
				refreshParametersCache();
			}
		}, 0, CACHE_REFRESH_TIME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void destroy() {
		super.destroy();
		timer.cancel();
	}

	protected ASLSession getASLSession(HttpSession httpSession)
	{
		String sessionID = httpSession.getId();
		String user = (String) httpSession.getAttribute(USERNAME_ATTRIBUTE);

		if (user == null) {

			log.info("STARTING IN TEST MODE - NO USER FOUND");

			//for test only
			user = "test.user";
			httpSession.setAttribute(USERNAME_ATTRIBUTE, user);
			ASLSession session = SessionManager.getInstance().getASLSession(sessionID, user);
			session.setScope(Constants.defaultScope);
			//session.setScope("/gcube/devsec/devVRE");

			return session;
		} else logger.trace("user found in session "+user);
		return SessionManager.getInstance().getASLSession(sessionID, user);
	}


	protected ServerParameters getParameters(GCUBEScope scope, String resourceName) throws Exception
	{
		ISClient client = GHNContext.getImplementation(ISClient.class);
		GCUBERuntimeResourceQuery rtrQuery = client.getQuery(GCUBERuntimeResourceQuery.class);
		rtrQuery.addAtomicConditions(new AtomicCondition("/Profile/Name",resourceName));

		List<GCUBERuntimeResource> rtrs = client.execute(rtrQuery, scope);

		if (rtrs.size()==0) throw new Exception("Resource "+resourceName+" not found");

		GCUBERuntimeResource rtr = rtrs.get(0);

		if (rtr.getAccessPoints().size()==0) throw new Exception("Accesspoint in resource "+resourceName+" not found");

		ServerParameters parameters = new ServerParameters();

		AccessPoint ap = rtr.getAccessPoints().get(0);
		parameters.setUrl(ap.getEndpoint());
		parameters.setUser(ap.getUsername()); //username
		parameters.setPassword(ap.getPassword()); //password

		return parameters;
	}

	protected String getTransectUrl(GCUBEScope scope, String resourceName) throws Exception
	{
		ISClient client = GHNContext.getImplementation(ISClient.class);
		GCUBEGenericResourceQuery rtrQuery = client.getQuery(GCUBEGenericResourceQuery.class);
		rtrQuery.addAtomicConditions(new AtomicCondition("/Profile/Name",resourceName));

		List<GCUBEGenericResource> rtrs = client.execute(rtrQuery, scope);

		if (rtrs.size()==0) throw new Exception("Generic resource "+resourceName+" not found");

		GCUBEGenericResource rtr = rtrs.get(0);

		return rtr.getBody(); //host
	}

	protected GisViewerServiceParameters retrieveGisParameters(GCUBEScope scope) throws Exception
	{
		GisViewerServiceParameters gisViewerServiceParameters = new GisViewerServiceParameters();

		try {
			ServerParameters geoServerParameters = getParameters(scope, GEOSERVER_RESOURCE_NAME);
			gisViewerServiceParameters.setGeoServerUrl(geoServerParameters.getUrl());
			gisViewerServiceParameters.setGeoServerUser(geoServerParameters.getUser());
			gisViewerServiceParameters.setGeoServerPwd(geoServerParameters.getPassword());
		} catch (Exception e)
		{
			logger.error("Error retrieving the GeoServer parameters", e);
			throw new Exception("Error retrieving the GeoServer parameters", e);
		}

		try {
			ServerParameters geoNetworkParameters = getParameters(scope, GEONETWORK_RESOURCE_NAME);
			gisViewerServiceParameters.setGeoNetworkUrl(geoNetworkParameters.getUrl());
			gisViewerServiceParameters.setGeoNetworkUser(geoNetworkParameters.getUser());
			gisViewerServiceParameters.setGeoNetworkPwd(geoNetworkParameters.getPassword());
		} catch (Exception e)
		{
			logger.error("Error retrieving the GeoNetwork parameters", e);
			throw new Exception("Error retrieving the GeoNetwork parameters", e);
		}

		try {
			String transectUrl = getTransectUrl(scope, TRANSECT_RESOURCE_NAME);
			gisViewerServiceParameters.setTransectUrl(transectUrl);
		} catch (Exception e)
		{
			logger.warn("Error retrieving the Transect url", e);
			//throw new Exception("Error retrieving the Transect url", e);
		}

		logger.trace("retrieved parameters: "+gisViewerServiceParameters);

		return gisViewerServiceParameters;		
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected GisViewerServiceParameters getParameters() throws Exception {
		HttpSession httpSession = this.getThreadLocalRequest().getSession();
		ASLSession session = getASLSession(httpSession);
		GCUBEScope scope = session.getScope();

		GisViewerServiceParameters parameters = parametersCache.get(scope);

		if (parameters == null) {
			parameters = retrieveGisParameters(scope);
			parametersCache.put(scope, parameters);
		}

		return parameters;
	}

	protected void refreshParametersCache()
	{
		for (GCUBEScope scope:parametersCache.keySet()) {
			try{
				GisViewerServiceParameters parameters = retrieveGisParameters(scope);
				parametersCache.put(scope, parameters);
			} catch (Exception e) {
				logger.warn("An error occured retrieving gis parameters for scope "+scope, e);
			}
		}
	}

	@Override
	protected GeoCaller getGeoCaller() throws Exception
	{
		HttpSession httpSession = this.getThreadLocalRequest().getSession();
		ASLSession session = getASLSession(httpSession);
		GeoCaller geoCaller = (GeoCaller) session.getAttribute(GEOCALLER_ATTRIBUTE_NAME);
		if (geoCaller==null) {
			try {

				GisViewerServiceParameters parameters = getParameters();
				String geoserverUrl = parameters.getGeoServerUrl();
				String geonetworkUrl = parameters.getGeoNetworkUrl();
				String gnUser = parameters.getGeoNetworkUser();
				String gnPwd = parameters.getGeoNetworkPwd();
				String gsUser = parameters.getGeoServerUser();
				String gsPwd = parameters.getGeoServerPwd();

				geoCaller = new GeoCaller(geonetworkUrl, gnUser, gnPwd, geoserverUrl, gsUser, gsPwd, researchMethod);
				session.setAttribute(GEOCALLER_ATTRIBUTE_NAME, geoCaller);

			} catch (Exception e) {
				throw new Exception("Error initializing the GeoCaller", e);
			}
		}

		return geoCaller;
	}

	public static void main(String[] args) throws MalformedScopeExpressionException, Exception
	{
		GCubeGisViewerServletImpl impl = new GCubeGisViewerServletImpl();
		GisViewerServiceParameters parameters = impl.retrieveGisParameters(GCUBEScope.getScope("/gcube/devsec/deVRE"));
		System.out.println(parameters);

	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void saveLayerItem(String name, String mimeType, String url, String destinationFolderId) throws Exception {
		logger.trace("saveLayerItem name: "+name+" mimeType: "+mimeType+" url: "+url+" destinationFolderId: "+destinationFolderId);

		try{

			HttpSession httpSession = this.getThreadLocalRequest().getSession();
			ASLSession session = getASLSession(httpSession);
			Workspace wa= HomeLibrary.getUserWorkspace(session);
			WorkspaceFolder destinationFolder = (WorkspaceFolder) wa.getItem(destinationFolderId);
			
			URL urlObject = new URL(url);
			URLConnection connection = urlObject.openConnection();
			connection.connect();
			name = WorkspaceUtil.getUniqueName(name, destinationFolder);
			InputStream is = connection.getInputStream();
			
			//workaround for a bug in the HomeLibrary
			if ("image/svg+xml".equals(mimeType)) destinationFolder.createExternalFileItem(name, "", mimeType, is);
			else WorkspaceUtil.createExternalFile(destinationFolder, name, "", mimeType, is);
		} catch (MalformedURLException e) {
			logger.error("", e);
			throw new GCubeGisViewerServiceException("Sorry, unable to retrieve layer data");
		} catch (IOException e) {
			logger.error("", e);
			throw new GCubeGisViewerServiceException("Sorry, unable to load layer data");
		} catch (InsufficientPrivilegesException e) {
			logger.error("", e);
			throw new GCubeGisViewerServiceException("Sorry, unable to save layer data. Please check your privileges");
		} catch (ItemAlreadyExistException e) {
			logger.error("", e);
			throw new GCubeGisViewerServiceException("Sorry, unable to save layer data : an Item with the chosen name already exists.");
		} catch (Exception e){
			logger.error("", e);
			throw new GCubeGisViewerServiceException("Sorry, unable to save layer. Unexpected Exception occurred. Try again or notify to administrator.");
		}
	}

	/* (non-Javadoc)
	 * @see org.gcube.portlets.user.gcubegisviewer.client.GCubeGisViewerService#saveMapImageItem(java.lang.String, java.lang.String, java.lang.String, java.lang.String, java.lang.String, java.lang.String[], java.lang.String[], java.lang.String[], java.lang.String[], java.lang.String[], java.lang.String[], java.lang.String)
	 */
	@Override
	public void saveMapImageItem(String name, String mimeType, String bbox,
			String width, String height, String[] geoservers, String[] layers,
			String[] styles, String[] opacities, String[] cqlfilters,
			String[] gsrefs, String destinationFolderId) throws Exception {
		try{

			HttpSession httpSession = this.getThreadLocalRequest().getSession();
			ASLSession session = getASLSession(httpSession);
			Workspace wa= HomeLibrary.getUserWorkspace(session);
			WorkspaceFolder destinationFolder = (WorkspaceFolder) wa.getItem(destinationFolderId);

			BufferedImage imgRis = MapGeneratorUtils.createMapImage(mimeType, bbox, width, height, geoservers, layers, styles, opacities, cqlfilters, gsrefs);

			name = WorkspaceUtil.getUniqueName(name, destinationFolder);
			
			File tmp = File.createTempFile("GCube", ".img");
			ImageIO.write(imgRis, MapGeneratorUtils.getOutputExtension(mimeType), tmp);

			InputStream is = new FileInputStream(tmp);
			
			//workaround for a bug in the HomeLibrary
			if ("image/svg+xml".equals(mimeType)) destinationFolder.createExternalFileItem(name, "", mimeType, is);
			else WorkspaceUtil.createExternalFile(destinationFolder, name, "", mimeType, is);
			
//			tmp.delete();
			System.out.println(tmp.getAbsolutePath());
			
		} catch (MalformedURLException e) {
			logger.error("", e);
			throw new GCubeGisViewerServiceException("Sorry, unable to retrieve layer data");
		} catch (IOException e) {
			logger.error("", e);
			throw new GCubeGisViewerServiceException("Sorry, unable to load layer data");
		} catch (InsufficientPrivilegesException e) {
			logger.error("", e);
			throw new GCubeGisViewerServiceException("Sorry, unable to save layer data. Please check your privileges");
		} catch (ItemAlreadyExistException e) {
			logger.error("", e);
			throw new GCubeGisViewerServiceException("Sorry, unable to save layer data : an Item with the chosen name already exists.");
		} catch (Exception e){
			logger.error("", e);
			throw new GCubeGisViewerServiceException("Sorry, unable to save layer. Unexpected Exception occurred. Try again or notify to administrator.");
		}
	}

	
}
