/****************************************************************************
 *  This software is part of the gCube Project.
 *  Site: http://www.gcube-system.org/
 ****************************************************************************
 * The gCube/gCore software is licensed as Free Open Source software
 * conveying to the EUPL (http://ec.europa.eu/idabc/eupl).
 * The software and documentation is provided by its authors/distributors
 * "as is" and no expressed or
 * implied warranty is given for its use, quality or fitness for a
 * particular case.
 ****************************************************************************
 * Filename: ScopeManager.java
 ****************************************************************************
 * @author <a href="mailto:daniele.strollo@isti.cnr.it">Daniele Strollo</a>
 ***************************************************************************/

package org.gcube.resourcemanagement.support.server.managers.scope;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import org.gcube.common.core.contexts.GHNContext;
import org.gcube.common.core.informationsystem.ISException;
import org.gcube.common.core.informationsystem.client.AtomicCondition;
import org.gcube.common.core.informationsystem.client.ISClient;
import org.gcube.common.core.informationsystem.client.queries.GCUBEGenericResourceQuery;
import org.gcube.common.core.resources.GCUBEGenericResource;
import org.gcube.common.core.scope.GCUBEScope;
import org.gcube.common.core.scope.ServiceMap;
import org.gcube.common.core.scope.VO;
import org.gcube.common.core.scope.VRE;
import org.gcube.resourcemanagement.support.server.utils.ServerConsole;
import org.w3c.dom.Document;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

/**
 * Scopes and their maps are persisted in this structure.
 * 
 * @author Massimiliano Assante (ISTI-CNR)
 * @author Daniele Strollo 
 *
 */
public class ScopeManager {
	private static final Map<String, GCUBEScope> SCOPES = new LinkedHashMap<String, GCUBEScope>();
	private static String confFile = null;
	private static final String LOG_PREFIX = "[SCOPE-MGR]";

	public static GCUBEScope getScope(final String scope) throws Exception {
		synchronized (SCOPES) {
			if (getAvailableScopes().containsKey(scope)) {
				return getAvailableScopes().get(scope);
			}
		}
		ServerConsole.warn(LOG_PREFIX, "Using DEFAULT scope manager");
		return GCUBEScope.getScope(scope);
	}

	public static void setScopeConfigFile(final String file) {
		confFile = file;
	}

	/**
	 * Refreshes the list of scopes and associated maps.
	 */

	public static Map<String, GCUBEScope> getAvailableScopes()
			throws Exception {
		if (SCOPES.size() == 0) {
			update();
		}
		return SCOPES;
	}

	public static void clear() {
		SCOPES.clear();
	}

	public static void update()
			throws Exception {
		if (confFile == null) {
			throw new NullPointerException("the scope file has not been defined");
		}
		String scopeXML = fileToString(confFile);
		ServerConsole.info(LOG_PREFIX, "Starting retrieving scopes..");
		Document scopeDocument = getDocumentGivenXML(scopeXML);
		NodeList voElements = scopeDocument.getElementsByTagName("vo");
		for (int i = 0; i < voElements.getLength(); i++) {
			NodeList voDetails = voElements.item(i).getChildNodes();
			String voString = voDetails.item(5).getFirstChild().getNodeValue();
			String fileName = voDetails.item(3).getFirstChild().getNodeValue();
			// String voName = voDetails.item(1).getFirstChild().getNodeValue();
			GCUBEScope vo = GCUBEScope.getScope(voString);

			try {
				vo.setServiceMap(loadServiceMap((VO) vo, fileName));
				SCOPES.put(vo.toString(), vo);
				
				ServerConsole.info(LOG_PREFIX, " Scopes in VO " + vo.toString());
				
				try {
					for (VRE vre : getVREFromVO((VO) vo)) {
						// This operation overrides the vo map
						vre.getEnclosingScope().setServiceMap(vo.getServiceMap());
						SCOPES.put(vre.toString(), vre);
					}
				} catch (ISException e) {
					ServerConsole.error(LOG_PREFIX, "Exception raised while loading VREs for VO : " + vo, e);
				}
			}
			catch (FileNotFoundException e) {
				ServerConsole.warn(LOG_PREFIX, "skipping... map not found for VO : " + vo, e);
			}


		}
		ServerConsole.info(LOG_PREFIX, "*** found scopes : " + SCOPES.keySet());
	}

	protected static List<VRE> getVREFromVO(final VO vo)
			throws Exception {
		ServerConsole.info(LOG_PREFIX, "*******************\n\n********************\nStarting Retrieving VREs for VO : " + vo);
		List<VRE> toReturn = new ArrayList<VRE>();

		ISClient client = GHNContext.getImplementation(ISClient.class);

		// FIXME query to get VREs
		try {
			GCUBEGenericResourceQuery query = client.getQuery(GCUBEGenericResourceQuery.class);
			query.addAtomicConditions(new AtomicCondition("/Profile/SecondaryType", "VRE"));
			//query.addGenericCondition("not($result/Scopes/Scope/string() eq '" + vo.toString() + "')");
			System.out.println( "**************  \n\n\n *****************\nready to query : " + query.getExpression());

			for (GCUBEGenericResource resource : client.execute(query, vo)) {
				ServerConsole.info(LOG_PREFIX, "Found: " + resource.getName());
				for (String vreName : resource.getScopes().keySet()) {
					GCUBEScope vre = resource.getScopes().get(vreName);
					if (vre.getType().equals(GCUBEScope.Type.VRE)) {
						toReturn.add((VRE) vre);
					}
				}

			}
		} catch (Exception e) {
			ServerConsole.error(LOG_PREFIX, e);
		}

		return toReturn;

	}

	/**
	 * Given a scope, if it is a VO or an infrastructure, the corresponding
	 * map file is retrieved from the globus location and the contained xml
	 * representation is returned.
	 * @param searchvo
	 * @return
	 * @throws Exception
	 */
	public static String getMapXML(final GCUBEScope searchvo) throws Exception {
		if (confFile == null) {
			throw new NullPointerException("the scope file has not been defined");
		}
		String scopeXML = fileToString(confFile);
		ServerConsole.info(LOG_PREFIX, "Starting retrieving scopes..");
		Document scopeDocument = getDocumentGivenXML(scopeXML);
		NodeList voElements = scopeDocument.getElementsByTagName("vo");
		for (int i = 0; i < voElements.getLength(); i++) {
			NodeList voDetails = voElements.item(i).getChildNodes();
			String voString = voDetails.item(5).getFirstChild().getNodeValue();
			String fileName = voDetails.item(3).getFirstChild().getNodeValue();
			// String voName = voDetails.item(1).getFirstChild().getNodeValue();
			GCUBEScope vo = GCUBEScope.getScope(voString);

			if (vo.equals(searchvo)) {
				return fileToString(System.getenv("GLOBUS_LOCATION") + File.separator + "config" + File.separator + fileName);
			}
		}
		ServerConsole.error(LOG_PREFIX, "*** maps for " + searchvo + " not found");
		return null;
	}

	private static ServiceMap loadServiceMap(final VO vo, final String fileName) throws Exception {
		ServiceMap map = new ServiceMap();
		String filePath = System.getenv("GLOBUS_LOCATION") + File.separator + "config" + File.separator + fileName;
		ServerConsole.info(LOG_PREFIX, "--- Loading " + vo.getName() + " from: " + filePath);
		map.load(new FileReader(filePath));
		return map;
	}

	public static String fileToString(final String path) throws IOException {
		BufferedReader filebuf = null;
		String nextStr = null;
		StringBuilder ret = new StringBuilder();

		filebuf = new BufferedReader(new FileReader(path));
		nextStr = filebuf.readLine(); // legge una riga dal file
		while (nextStr != null) {
			ret.append(nextStr);
			nextStr = filebuf.readLine(); // legge la prossima riga
		}
		filebuf.close(); // chiude il file

		return ret.toString();
	}

	public static Document getDocumentGivenXML(final String result) {
		DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
		dbf.setValidating(false);
		DocumentBuilder db;
		Document document = null;
		try {
			db = dbf.newDocumentBuilder();
			document = db.parse(new ByteArrayInputStream(result.getBytes()));
		} catch (ParserConfigurationException e1) {
			e1.printStackTrace();
		} catch (SAXException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		}

		return document;
	}

}
