/****************************************************************************
 *  This software is part of the gCube Project.
 *  Site: http://www.gcube-system.org/
 ****************************************************************************
 * The gCube/gCore software is licensed as Free Open Source software
 * conveying to the EUPL (http://ec.europa.eu/idabc/eupl).
 * The software and documentation is provided by its authors/distributors
 * "as is" and no expressed or
 * implied warranty is given for its use, quality or fitness for a
 * particular case.
 ****************************************************************************
 * Filename: SWRepositoryManager.java
 ****************************************************************************
 * @author <a href="mailto:daniele.strollo@isti.cnr.it">Daniele Strollo</a>
 ***************************************************************************/

package org.gcube.resourcemanagement.support.managers.services;

import java.util.ArrayList;
import java.util.List;
import java.util.Random;
import java.util.Vector;
import org.apache.axis.message.addressing.EndpointReferenceType;
import org.gcube.common.core.contexts.GCUBERemotePortTypeContext;
import org.gcube.common.core.contexts.GHNContext;
import org.gcube.common.core.informationsystem.client.AtomicCondition;
import org.gcube.common.core.informationsystem.client.ISClient;
import org.gcube.common.core.informationsystem.client.queries.GCUBERIQuery;
import org.gcube.common.core.resources.GCUBERunningInstance;
import org.gcube.common.core.scope.GCUBEScope;
import org.gcube.common.core.security.GCUBESecurityManagerImpl;
import org.gcube.resourcemanagement.support.exceptions.ResourceAccessException;
import org.gcube.resourcemanagement.support.exceptions.ResourceParameterException;
import org.gcube.resourcemanagement.support.utils.Assertion;
import org.gcube.resourcemanagement.support.utils.ServerConsole;
import org.gcube.vremanagement.softwarerepository.stubs.SoftwareRepositoryPortType;
import org.gcube.vremanagement.softwarerepository.stubs.service.SoftwareRepositoryServiceAddressingLocator;

/**
 * A support library to handle and retrieve remote software
 * repository manager.
 * <br/>
 * Useful to handle the proxy for repository manager.
 * <br/>Usage:
 * <pre>
 *  <b>SWRepositoryManager</b> repMgr = new SWRepositoryManager();
 *  // if needed sets the time out for socket connection
 *  // -- repMgr.setCallTimeOut(240000);
 *  SoftwareRepositoryPortType swrep = repMgr.getSoftwareRepository(scope);
 *  // Applies a deploy
 *  swrep.<i>store(...)</i>;
 * </pre>
 * @author Daniele Strollo (ISTI-CNR)
 */
public class SWRepositoryManager {
	private GCUBESecurityManagerImpl managerSec = null;
	private ISClient client = null;
	private int callTimeOut = 240000;
	private static final String LOG_PREFIX = "[SW-UPDT-MGR]";

	public SWRepositoryManager() throws ResourceAccessException {
		/**
		 * Initially the security management is disabled.
		 */
		this.managerSec = new GCUBESecurityManagerImpl() {
			public boolean isSecurityEnabled() {
				return false;
			}
		};

		try {
			client = GHNContext.getImplementation(ISClient.class);
		} catch (Exception e) {
			throw new ResourceAccessException("Cannot instantiate the ISClient");
		}
	}

	/**
	 * Internally used by {@link AbstractResourceManager#getResourceManager(GCUBEScope)}.
	 * @param scope
	 * @return a raw list of resource manager descriptors.
	 * @throws Exception
	 */
	private List<GCUBERunningInstance> getSoftwareRepositoryFromScope(final GCUBEScope scope)
	throws Exception {

		GCUBERIQuery query = this.client.getQuery(GCUBERIQuery.class);
		query.addAtomicConditions(new AtomicCondition("//Profile/ServiceClass", "VREManagement"));
		query.addAtomicConditions(new AtomicCondition("//Profile/ServiceName", "SoftwareRepository"));

		List<GCUBERunningInstance> result = client.execute(query, scope);
		List<GCUBERunningInstance> toReturn = new ArrayList<GCUBERunningInstance>();
		for (GCUBERunningInstance ri : result) {
			if (ri.getScopes().containsValue(scope)) {
				toReturn.add(ri);
			}
		}
		return toReturn;
	}

	/**
	 * The the list of resource managers able to handle the resource in a given scope.
	 * @param scope the scope in which to operate
	 * @return all the available managers
	 * @throws ResourceAccessException if no manager can be instantiated
	 * @throws ResourceParameterException if the parameters are invalid
	 */
	private List<SoftwareRepositoryPortType> getSoftwareRepositories(final GCUBEScope scope)
	throws ResourceAccessException, ResourceParameterException {
		Assertion<ResourceParameterException> checker = new Assertion<ResourceParameterException>();
		checker.validate(scope != null, new ResourceParameterException("Invalid scope"));

		List<GCUBERunningInstance> resourceManagerList = null;
		try {
			resourceManagerList = this.getSoftwareRepositoryFromScope(scope);
		} catch (Exception e) {
			throw new ResourceAccessException("Cannot retrieve the software repository in scope: " + scope.toString());
		}

		List<SoftwareRepositoryPortType> retval = new Vector<SoftwareRepositoryPortType>();

		if (resourceManagerList.isEmpty()) {
			throw new ResourceAccessException("Unable to find the software repository in scope: " + scope.toString());
		}

		EndpointReferenceType endpoint = null;
		SoftwareRepositoryPortType pt = null;

		for (GCUBERunningInstance resourceManager : resourceManagerList) {
			try {
				endpoint = resourceManager.getAccessPoint().getEndpoint("gcube/vremanagement/softwarerepository/SoftwareRepository");

				pt = GCUBERemotePortTypeContext.getProxy(
						new SoftwareRepositoryServiceAddressingLocator()
						.getSoftwareRepositoryPortTypePort(endpoint),
						scope,
						this.callTimeOut,
						this.managerSec);
				if (pt != null) {
					retval.add(pt);
				}
			} catch (Throwable e) {
				// trying on next entry
				ServerConsole.error(LOG_PREFIX, e);
			}
		}
		if (retval != null && retval.size() > 0) {
			// Return a random manager from the available ones
			return retval;
		}
		// no managers found
		throw new ResourceAccessException("Unable to find the software repository in scope: " + scope.toString());
	}

	/**
	 * Retrieves form the current scope the list of registered software managers
	 * and returns one of them (random choice).
	 * @param scope the scope in which search the sw repository
	 * @return one randomly chosen repository manager (if many available).
	 * @throws ResourceAccessException if no sw manager found
	 * @throws ResourceParameterException if wrong parameters passed
	 */
	public final SoftwareRepositoryPortType getSoftwareRepository(final GCUBEScope scope)
	throws ResourceAccessException, ResourceParameterException {
		List<SoftwareRepositoryPortType> retval = this.getSoftwareRepositories(scope);
		if (retval != null && retval.size() > 0) {
			Random generator = new Random();
			// Return a random software repository manager from the available ones
			return retval.get(generator.nextInt(retval.size()));
		}
		// no managers found
		throw new ResourceAccessException("Unable to find SoftwareRepository in scope: " + scope.toString());
	}

	/**
	 * Used to set the time out for socket connection with the software repository
	 * to update.
	 * @param callTimeOut
	 */
	public final void setCallTimeOut(final int callTimeOut) {
		this.callTimeOut = callTimeOut;
	}

	public final int getCallTimeOut() {
		return callTimeOut;
	}
}
