package org.gcube.portlets.admin.service;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.liferay.portal.kernel.json.JSONFactoryUtil;
import com.liferay.portal.kernel.json.JSONObject;
import com.liferay.portal.kernel.log.Log;
import com.liferay.portal.kernel.log.LogFactoryUtil;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;

import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.NameValuePair;
import org.apache.http.client.HttpClient;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.util.EntityUtils;
import org.osgi.service.component.annotations.Component;

/**
 * Service for handling OIDC token exchange operations
 * 
 * @author netfarm-m2
 */
@Component(
    immediate = true,
    service = OIDCTokenService.class
)
public class OIDCTokenService {

    private static final Log _log = LogFactoryUtil.getLog(OIDCTokenService.class);
    private final ObjectMapper objectMapper = new ObjectMapper();

    /**
     * Performs token exchange using OAuth2 Token Exchange flow
     * 
     * @param tokenUrl the OIDC token endpoint URL
     * @param accessToken the current access token
     * @param context the target context/audience
     * @param clientId the exchange client ID
     * @param clientSecret the exchange client secret
     * @return JSONObject containing the exchanged token information
     * @throws Exception if token exchange fails
     */
    public JSONObject exchangeToken(String tokenUrl, String accessToken, String context, 
                                  String clientId, String clientSecret) throws Exception {
        
        if (_log.isDebugEnabled()) {
            _log.debug("Exchanging token for context: " + context);
        }

        HttpClient httpClient = HttpClients.createDefault();
        HttpPost httpPost = new HttpPost(tokenUrl);

        // Prepare form parameters for token exchange
        List<NameValuePair> params = new ArrayList<>();
        params.add(new BasicNameValuePair("subject_token", accessToken));
        params.add(new BasicNameValuePair("client_id", clientId));
        params.add(new BasicNameValuePair("client_secret", clientSecret));
        params.add(new BasicNameValuePair("grant_type", "urn:ietf:params:oauth:grant-type:token-exchange"));
        params.add(new BasicNameValuePair("subject_token_type", "urn:ietf:params:oauth:token-type:access_token"));
        params.add(new BasicNameValuePair("requested_token_type", "urn:ietf:params:oauth:token-type:access_token"));

        // URL encode context if needed
        String encodedContext = context;
        if (context.startsWith("/")) {
            try {
                encodedContext = URLEncoder.encode(context, StandardCharsets.UTF_8.toString());
            } catch (UnsupportedEncodingException e) {
                _log.error("Cannot URL encode context", e);
            }
        }

        // Set headers
        httpPost.setHeader("Content-Type", "application/x-www-form-urlencoded");
        httpPost.setHeader("X-D4Science-Context", encodedContext);

        // Set entity
        httpPost.setEntity(new UrlEncodedFormEntity(params, StandardCharsets.UTF_8));

        try {
            HttpResponse response = httpClient.execute(httpPost);
            HttpEntity entity = response.getEntity();
            String responseBody = EntityUtils.toString(entity);

            int statusCode = response.getStatusLine().getStatusCode();
            
            if (statusCode == 200) {
                // Parse successful response
                JsonNode jsonNode = objectMapper.readTree(responseBody);
                
                JSONObject result = JSONFactoryUtil.createJSONObject();
                result.put("success", true);
                result.put("access_token", jsonNode.get("access_token").asText());
                
                if (jsonNode.has("refresh_token")) {
                    result.put("refresh_token", jsonNode.get("refresh_token").asText());
                }
                
                if (jsonNode.has("expires_in")) {
                    result.put("expires_in", jsonNode.get("expires_in").asInt());
                }
                
                result.put("token_url", tokenUrl);
                result.put("client_id", clientId);
                result.put("raw_token", responseBody);
                
                return result;
                
            } else {
                // Handle error response
                _log.error("Token exchange failed with status: " + statusCode + ", response: " + responseBody);
                
                JSONObject errorResult = JSONFactoryUtil.createJSONObject();
                errorResult.put("success", false);
                errorResult.put("error", "Token exchange failed");
                errorResult.put("status_code", statusCode);
                errorResult.put("message", responseBody);
                
                return errorResult;
            }
            
        } catch (IOException e) {
            _log.error("Error during token exchange", e);
            
            JSONObject errorResult = JSONFactoryUtil.createJSONObject();
            errorResult.put("success", false);
            errorResult.put("error", "Network error during token exchange");
            errorResult.put("message", e.getMessage());
            
            return errorResult;
        }
    }

    /**
     * Validates if a token is still valid (basic validation)
     * 
     * @param token the token to validate
     * @return true if token appears valid, false otherwise
     */
    public boolean isTokenValid(String token) {
        return token != null && !token.trim().isEmpty() && token.contains(".");
    }

    /**
     * Extracts basic information from a JWT token without validation
     * 
     * @param token the JWT token
     * @return JSONObject with basic token information
     */
    public JSONObject getTokenInfo(String token) {
        JSONObject info = JSONFactoryUtil.createJSONObject();
        
        if (!isTokenValid(token)) {
            info.put("valid", false);
            return info;
        }
        
        try {
            // Basic JWT parsing (header.payload.signature)
            String[] parts = token.split("\\.");
            if (parts.length >= 2) {
                // Decode payload (base64url)
                String payload = new String(
                    java.util.Base64.getUrlDecoder().decode(parts[1]), 
                    StandardCharsets.UTF_8
                );
                
                JsonNode payloadNode = objectMapper.readTree(payload);
                
                info.put("valid", true);
                if (payloadNode.has("exp")) {
                    info.put("expires_at", payloadNode.get("exp").asLong());
                }
                if (payloadNode.has("aud")) {
                    info.put("audience", payloadNode.get("aud").asText());
                }
                if (payloadNode.has("sub")) {
                    info.put("subject", payloadNode.get("sub").asText());
                }
            }
        } catch (Exception e) {
            _log.warn("Could not parse token info", e);
            info.put("valid", false);
        }
        
        return info;
    }
}