package org.gcube.opensearch.opensearchdatasource;

import java.net.InetAddress;
import java.net.URI;
import java.rmi.RemoteException;

import org.gcube.common.core.utils.logging.GCUBELog;
import org.gcube.opensearch.opensearchdatasource.cacherefresh.CacheRefresher;
import org.gcube.opensearch.opensearchdatasource.stubs.AddProvidersParams;
import org.gcube.opensearch.opensearchdatasource.stubs.AddProvidersResponse;
import org.gcube.opensearch.opensearchdatasource.stubs.Provider;
import org.gcube.opensearch.opensearchdatasource.stubs.RefreshCache;
import org.gcube.opensearch.opensearchdatasource.stubs.RefreshCacheResponse;
import org.gcube.opensearch.opensearchoperator.OpenSearchOp;
import org.gcube.opensearch.opensearchoperator.resource.OpenSearchResource;
import org.globus.wsrf.ResourceContext;

/**
 * OpenSearchDataSource implements the functionality needed to query the underlying OpenSearch providers using the {@link OpenSearchOp} OpenSearch operator
 * Each {@link OpenSearchDataSourceResource} of this stateful service handles a number of OpenSearch providers. Each provider is associated with
 * a collection and an {@link OpenSearchResource} generic resource containing information for the provider.
 * A {@link OpenSearchDataSourceResource} published a set field information, which specifies which search fields are searchable and presentable on a collection.
 * While querying, all generic resources and description documents retrieved are stored in a cache contained in each {@link OpenSearchDataSourceResource}
 * The cache can be refreshed either on demand or periodically, or through both ways to allow the service to update its state in the event of generic resource update
 * or modifications in the description documents of providers.
 * The cache purging time interval as well as whether the service should clear the cache on startup are configurable parameters 
 * 
 * @author Vassilis Verroios, NKUA
 * @author gerasimos.farantatos, NKUA
 * 
 * @version 1.2
 */
public class OpenSearchDataSource {

		public static enum SupportedRelations {
			eq {
				public String toString() {
					return "=";
				}
			},
			exact {
				public String toString() {
					return "==";
				}
			}, any, all
		};
		
		public String[] supportedRelations;
		
        String hostName = "";

        static GCUBELog logger = new GCUBELog(OpenSearchDataSource.class);


        /**
         * Constructor - sets up some variables and creates a logging instance
         */
        public OpenSearchDataSource() throws RemoteException {

                try{
                        hostName = InetAddress.getLocalHost().getHostName();
                }catch(Exception e) {
                    logger.error("Cannot set hostname");
                    logger.error(e.toString());
                    throw new RemoteException("Exception when getting hostname",e);
                }

                logger.debug("OpenSearchDataSource constructed");
        }


        /**
         * Performs a query on the OpenSearch provider associated with the WS resource used and described by the corresponding generic resource
         * @param cqlQuery <code>String</code>  - the gCQL query to be translated a set of queries with the syntax of the OpenSearch operator -
         *  space separated <(URLEncodedParameterNamespace|config):parameterName> strings)
         * @return <code>String</code>  - a gRS2 locator which holds the results of the query.
         * @throws RemoteException In case of error
         */
        public String query(String cqlQuery) throws RemoteException {
            OpenSearchDataSourceResource opensearchResource = getResource();
            logger.info(opensearchResource.getAdaptorID() + " received query: " + cqlQuery);
            
            URI loc = opensearchResource.query(cqlQuery);
			return loc.toString();
        }
         
        /**
         * Forced cache refresh operation
         * @see {@link CacheRefresher#forceRefresh()}
         * 
         * @param req An empty {@link RefreshCache} request
         * @return An empty {@link RefreshCacheResponse}
         * @throws RemoteException In case of error
         */
        public RefreshCacheResponse refreshCache(RefreshCache req) throws RemoteException {
        	OpenSearchDataSourceResourceHome wsHome = (OpenSearchDataSourceResourceHome)StatefulContext.getPortTypeContext().getWSHome();
        	wsHome.refreshCache();
        	return new RefreshCacheResponse();
        }
        
        public AddProvidersResponse addProviders(AddProvidersParams req) throws RemoteException {
        	OpenSearchDataSourceResource opensearchResource = getResource();
        	String[] fields = req.getFields().getArray() != null ? req.getFields().getArray() : new String[0];
        	int providerCount = req.getProviders().length;
        	String[] collectionIDs = new String[providerCount];
    		String[] openSearchResourceIDs = new String[providerCount];
    		String[][] fixedParameters = new String[providerCount][];
    		Provider[] providers = req.getProviders();
    		for(int i = 0; i < providerCount; i++) {
    			collectionIDs[i] = providers[i].getCollectionID();
    			openSearchResourceIDs[i] = providers[i].getOpenSearchResourceID();
    			fixedParameters[i] = (providers[i].getFixedParameters().getArray() != null ? providers[i].getFixedParameters().getArray() : new String[0]);
    		}
  
        	try {
        		opensearchResource.addProviders(fields, collectionIDs, openSearchResourceIDs, fixedParameters);
        	}catch(Exception e) {
        		throw new RemoteException("Error while adding provider", e);
        	}
        	opensearchResource.store();
        	return new AddProvidersResponse();
        }
        
        /*
         * Private method that gets a reference to the resource specified in the
         * endpoint reference.
         */
        private OpenSearchDataSourceResource getResource() throws RemoteException {
                Object resource = null;
                try {
                        resource = ResourceContext.getResourceContext().getResource();
                } catch (Exception e) {
                    logger.error("Unable to get the OpenSearchDataSource resource");
                    throw new RemoteException("Unable to access resource.", e);
                }

                OpenSearchDataSourceResource openSearchResource = (OpenSearchDataSourceResource) resource;
                return openSearchResource;
        }
        
        public static void ready(){

        }
}
