package org.gcube.messaging.common.producer;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import javax.jms.JMSException;
import javax.jms.QueueConnection;
import javax.jms.QueueConnectionFactory;
import javax.jms.TopicConnection;
import javax.jms.TopicConnectionFactory;

import org.apache.activemq.ActiveMQConnectionFactory;
import org.apache.axis.message.addressing.Address;
import org.apache.axis.message.addressing.EndpointReferenceType;
import org.apache.axis.types.URI.MalformedURIException;
import org.gcube.common.core.contexts.GHNContext;
import org.gcube.common.core.monitoring.GCUBETestProbe;
import org.gcube.common.core.monitoring.LocalMonitor;
import org.gcube.common.core.scope.GCUBEScope;
import org.gcube.common.core.scope.GCUBEScopeNotSupportedException;
import org.gcube.common.core.utils.logging.GCUBEClientLog;


/**
 * Implementation of a Local Monitor
 * 
 * @author Andrea Manzi(CERN)
 *
 */
public class GCUBELocalProducer extends LocalMonitor{


	/**ActiveMQ Connection factory**/
	private static HashMap<GCUBEScope,ArrayList<ActiveMQConnectionFactory>> connectionFactoryMap = null;
	private static HashMap<GCUBEScope,ArrayList<TopicConnection>> topicConnectionMap = null;
	private static HashMap<GCUBEScope,ArrayList<QueueConnection>> queueConnectionMap = null;

	public static GCUBEClientLog logger = null;
	static Properties resources = new Properties();
	public GCUBELocalProducer(){}

	/**
	 * {@inheritDoc} 
	 */
	@Override
	public void loadProbes(Map<String,Class<? extends GCUBETestProbe>> map) {

		try {
			Properties query= new Properties();
			query.load(GCUBELocalProducer.class.getResourceAsStream("/probes.properties"));

			for (Object prop : query.keySet()) {
				try {
					map.put((String) prop, (Class<? extends GCUBETestProbe>) Class.forName((String)query.get(prop)));
				} catch (Exception e){
					logger.error("Error loading probe class",e);
				}
			}
		}
		catch (Exception e) {
			logger.error("Error loading probes configuration",e);
		}
	}

	/**
	 * Get the TopicConnection list for the given scope
	 * @return the topic Connection list for the given scope
	 */
	public static ArrayList<TopicConnection> getTopicConnection(GCUBEScope scope){
		ArrayList<TopicConnection> connection = topicConnectionMap.get(scope);
		if( connection == null ) {
			logger.debug("CONNECTION MAP NULL");
		}
		return connection;
	}

	/**
	 * Get the QueueConnection list for the given scope
	 * @return the queue Connection list for the given scope
	 */
	public static ArrayList<QueueConnection> getQueueConnection(GCUBEScope scope){
		ArrayList<QueueConnection> connection = queueConnectionMap.get(scope);
		if( connection == null ) {
			logger.debug("CONNECTION MAP NULL");
		}
		return connection;
	}

	/**
	 * Add the Scope to the monitoredMap
	 * @return the scope to add
	 * @throws GCUBEScopeNotSupportedException 
	 * @throws BrokerNotConfiguredInScopeException
	 */
	public synchronized static void addScope (GCUBEScope scope) throws BrokerNotConfiguredInScopeException, GCUBEScopeNotSupportedException{
		if (!(brokerMap.containsKey(scope)))
		{
			if (scope.getServiceMap().getEndpoints(GHNContext.MSGBROKER)!= null) 
			{
				ArrayList<EndpointReferenceType> listScopes = new ArrayList<EndpointReferenceType>();
				for (EndpointReferenceType msgBrokerEpr:scope.getServiceMap().getEndpoints(GHNContext.MSGBROKER)) {
					logger.info("MSG-Broker found: "+msgBrokerEpr.getAddress().toString()+" for scope: "+scope.toString());
					listScopes.add(msgBrokerEpr);
				}
				brokerMap.put(scope, listScopes);
			}
			else throw new BrokerNotConfiguredInScopeException();

		}
	}

	/**
	 * retrun the Monitored scope
	 * @return The Set of Scope to monitor
	 */
	public synchronized static Set<GCUBEScope> getMonitoredScope () 
	{
		return brokerMap.keySet();
	}

	/**
	 * Remove Scope from BrokerMap
	 * @param scope the scope to remove
	 */
	public synchronized static void removeScope (GCUBEScope scope) 
	{
		brokerMap.remove(scope);
	}


	/**
	 * Reload the scope connection
	 * @param scope the scope to reload
	 */
	public synchronized static void reloadConnection(GCUBEScope scope) {

		//close current connections:
		stopConnections(scope);
		ArrayList<ActiveMQConnectionFactory> factoryList = new ArrayList<ActiveMQConnectionFactory>();
		ArrayList<TopicConnection> connectionList = new ArrayList<TopicConnection>();
		ArrayList<QueueConnection> queueConnectionList = new ArrayList<QueueConnection>();

		logger.debug("Reload JMS connections");
		for (EndpointReferenceType epr : brokerMap.get(scope))
		{
			try {
				logger.info("MSG-Broker found: "+epr.getAddress().toString()+" for scope: "+scope.toString());
				ActiveMQConnectionFactory factory = new ActiveMQConnectionFactory(epr.getAddress().toString());
				factoryList.add(factory);
				TopicConnection connection = ((TopicConnectionFactory)factory).createTopicConnection();
				connectionList.add(connection);
				connection.start();	
				QueueConnection queueCon = ((QueueConnectionFactory)factory).createQueueConnection();
				queueConnectionList.add(queueCon);
				queueCon.start();

			} catch (JMSException e1) {
				logger.error("Error creating Topic Connection",e1);
			} catch (Exception e) {
				logger.error("Error creating Topic Connection",e);
			}
		}
		topicConnectionMap.put(scope, connectionList);
		queueConnectionMap.put(scope,queueConnectionList );
		connectionFactoryMap.put(scope, factoryList);

	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void run() {

		connectionFactoryMap = new HashMap<GCUBEScope,ArrayList<ActiveMQConnectionFactory>>();
		topicConnectionMap =  new HashMap<GCUBEScope,ArrayList<TopicConnection>>();
		queueConnectionMap = new HashMap<GCUBEScope, ArrayList<QueueConnection>>();

		try {
			resources.load(GCUBELocalProducer.class.getResourceAsStream("/GCUBELocalProducerLogger.properties"));
		} catch (IOException e) {	
			e.printStackTrace();
		}
		logger = new GCUBEClientLog(GCUBELocalProducer.class,resources);

		for (GCUBEScope scope : brokerMap.keySet())	{
			logger.info("Broker Map, scope to monitor: "+scope.toString());
			reloadConnection(scope);
		}

		if (implementationMap==null) {
			implementationMap = Collections.synchronizedMap(new HashMap<String,Class<? extends GCUBETestProbe>>());
			loadProbes(implementationMap);
		}	

		for (Class<? extends GCUBETestProbe> classProbe :implementationMap.values())
		{
			try {
				GCUBETestProbe probe = classProbe.newInstance();
				probe.setInterval(interval);
				logger.info("Initializing " + classProbe + " Probe");
				probe.execute();
			} catch (Exception e) {
				logger.error("Error Initializing probe "+classProbe,e);	
			}
		}
	}	

	/**
	 * Check if the given scope belongs to start scopes
	 * @param scope the scope to check
	 * @return true/false
	 */
	public static boolean checkStartScope(GCUBEScope scope) {
		boolean isStartScope = false;
		for (GCUBEScope scopeStart:GHNContext.getContext().getStartScopes())
		{
			if (scope.equals(scopeStart) || scope.isInfrastructure())
				isStartScope = true;
		}
		return isStartScope;
	}


	private static void stopConnections(GCUBEScope scope){

		if (topicConnectionMap != null && topicConnectionMap.get(scope) != null) {
			for (TopicConnection con :topicConnectionMap.get(scope)){
				try {
					con.stop();
					con.close();
				}
				catch (JMSException e ){
					logger.error("Error stopping topicConnections",e);
				}
			}
		}
		if (queueConnectionMap != null  && queueConnectionMap.get(scope) != null) {
			//close current connections:
			for (QueueConnection con :queueConnectionMap.get(scope)){
				try {
					con.stop();
					con.close();
				}
				catch (JMSException e ){
					logger.error("Error stopping queueConnections",e);
				}
			}
		}
	}
	/**
	 * Main
	 * @param args
	 */
	public static void main(String [] args){
		long interval = 1200;
		String brokerURL= "tcp://ui.grid.research-infrastructures.eu:6166";
		GCUBEScope scope = GCUBEScope.getScope("/gcube/devsec");
		if (args.length == 3) {
			brokerURL = args[0];
			interval = Long.parseLong(args[1]);
			scope = GCUBEScope.getScope(args[2]);

		}
		HashMap<GCUBEScope,ArrayList<EndpointReferenceType>> monitoredScopes = new HashMap<GCUBEScope,ArrayList<EndpointReferenceType>>();
		ArrayList<EndpointReferenceType> eprs = new ArrayList<EndpointReferenceType>();
		try {
			eprs.add(new EndpointReferenceType(new Address(brokerURL)));
		} catch (MalformedURIException e) {
			e.printStackTrace();
		}

		monitoredScopes.put(scope,eprs);
		GCUBELocalProducer local = new GCUBELocalProducer();
		local.setBrokerMap(monitoredScopes);
		local.setInterval(interval);
		local.run();
	}


}

