package org.gcube.informationsystem.resourceregistry.rest;

import java.util.List;

import org.eclipse.microprofile.openapi.annotations.Operation;
import org.eclipse.microprofile.openapi.annotations.enums.ParameterIn;
import org.eclipse.microprofile.openapi.annotations.enums.SchemaType;
import org.eclipse.microprofile.openapi.annotations.media.Content;
import org.eclipse.microprofile.openapi.annotations.media.Schema;
import org.eclipse.microprofile.openapi.annotations.parameters.Parameter;
import org.eclipse.microprofile.openapi.annotations.responses.APIResponse;
import org.eclipse.microprofile.openapi.annotations.tags.Tag;
import org.gcube.informationsystem.resourceregistry.ResourceInitializer;
import org.gcube.informationsystem.resourceregistry.api.exceptions.ResourceRegistryException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.types.SchemaException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.types.SchemaNotFoundException;
import org.gcube.informationsystem.resourceregistry.api.rest.TypePath;
import org.gcube.informationsystem.resourceregistry.rest.requests.ServerRequestInfo;
import org.gcube.informationsystem.resourceregistry.types.TypeManagement;
import org.gcube.informationsystem.types.TypeMapper;
import org.gcube.informationsystem.types.reference.Type;

import jakarta.ws.rs.Consumes;
import jakarta.ws.rs.DefaultValue;
import jakarta.ws.rs.GET;
import jakarta.ws.rs.InternalServerErrorException;
import jakarta.ws.rs.PUT;
import jakarta.ws.rs.Path;
import jakarta.ws.rs.PathParam;
import jakarta.ws.rs.Produces;
import jakarta.ws.rs.QueryParam;
import jakarta.ws.rs.WebApplicationException;
import jakarta.ws.rs.core.MediaType;
import jakarta.ws.rs.core.Response;
import jakarta.ws.rs.core.Response.Status;


/**
 * @author Luca Frosini (ISTI - CNR)
 */
@Path(TypePath.TYPES_PATH_PART)
@Tag(name = "Types", description = "Operations for managing type definitions in the Resource Registry.")
public class TypeManager extends BaseRest {

	/**
	 * Constant representing the type name path parameter.
	 */
	public static final String TYPE_PATH_PARAMETER = "type-name";
	
	public TypeManager() {
		super();
	}
	
	/**
	 * Clean the types cache
	 * @return
	 */
	// @DELETE
	public Response cleanCache() {
		try {
			// TODO
			return Response.status(Status.NO_CONTENT).build();
		}catch (WebApplicationException e) {
			throw e;
		}catch (Exception e) {
			throw new InternalServerErrorException(e);
		}
	}
	
	
	/**
	 * Creates a new type definition in the Information System.
	 * Only the highest level administrative users can perform this operation due to its system-wide impact.
	 * 
	 * <strong>REST Endpoint:</strong> {@code PUT /types/{type-name}}
	 * 
	 * <strong>Request Examples:</strong>
	 * <ul>
	 * <li>PUT /types/ContactFacet (creates a new ContactFacet type definition)</li>
	 * <li>PUT /types/MyCustomResource (creates a new custom resource type).</li>
	 * </ul>
	 * 
	 * <strong>Path Parameters:</strong>
	 * <ul>
	 * <li><strong>type-name</strong>: The name of the type to create (e.g., "ContactFacet", "MyCustomResource").</li>
	 * </ul>
	 * 
	 * <strong>Request Body:</strong>
	 * <ul>
	 * <li>JSON representation of the type definition</li>
	 * <li>Must conform to the Information System type schema.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * 
	 * <strong>IS-Manager:</strong>
	 * <ul>
	 * <li>Can create any type definition without restrictions</li>
	 * <li>Has full administrative privileges across the entire Information System.</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Can create any type definition without restrictions</li>
	 * <li>Has full administrative privileges across the entire Information System.</li>
	 * </ul>
	 * 
	 * <strong>All Other Users (including Context-Manager):</strong>
	 * <ul>
	 * <li>Cannot create type definitions</li>
	 * <li>Will receive authorization errors if attempting these operations</li>
	 * <li>Type creation is restricted to prevent system-wide compatibility issues.</li>
	 * </ul>
	 * 
	 * <strong>Operation Behavior:</strong>
	 * 
	 * <strong>Type Update Policy:</strong>
	 * 
	 * <strong>No REST Update Method Exposed:</strong>
	 * <ul>
	 * <li>Although the underlying code supports type updates, the update functionality is intentionally NOT exposed via REST API</li>
	 * <li>Type modifications can break compatibility across the entire Information System</li>
	 * <li>Examples of breaking changes:</li>
	 * </ul>
	 *   • Making a previously optional facet property mandatory;
	 *   • Changing property types or constraints;
	 *   • Modifying validation rules that could invalidate existing instances.
	 * 
	 * <strong>Offline Update Process:</strong>
	 * <ul>
	 * <li>Type updates must be performed offline after careful impact assessment</li>
	 * <li>Changes require evaluation of all existing instances and dependent systems</li>
	 * <li>Updates are applied through controlled maintenance procedures</li>
	 * <li>This ensures system stability and prevents breaking existing client applications.</li>
	 * </ul>
	 * 
	 * <strong>Response Codes:</strong>
	 * <ul>
	 * <li>201 Created: Type successfully created</li>
	 * <li>409 Conflict: Type already exists with the given name</li>
	 * <li>400 Bad Request: Invalid JSON schema or malformed request</li>
	 * <li>403 Forbidden: User lacks authorization to create types.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Content-Type: application/json</li>
	 * <li>Body: Complete type definition including metadata and schema information.</li>
	 * </ul>
	 * 
	 * @param typeName the name of the type to create (must be unique in the system)
	 * @param json the JSON schema definition of the type to create
	 * @return HTTP 201 Created response with the complete type definition
	 * @throws SchemaException if the provided schema is invalid or malformed
	 * @throws ResourceRegistryException for creation errors or authorization failures
	 */
	@PUT
	@Path("{" + TypeManager.TYPE_PATH_PARAMETER + "}")
	@Consumes({MediaType.TEXT_PLAIN, ResourceInitializer.APPLICATION_JSON_CHARSET_UTF_8})
	@Produces(ResourceInitializer.APPLICATION_JSON_CHARSET_UTF_8)
	@Operation(
		summary = "Create Type Definition",
		description = """
		Creates a new type definition in the Information System.

		Only the highest level administrative users can perform this operation due to its system-wide impact.


		**Request Examples:**
		- PUT /types/ContactFacet (creates a new ContactFacet type definition);
		- PUT /types/MyCustomResource (creates a new custom resource type).


		**Authorization Requirements:**
		- **IS-Manager:**
			- Can create any type definition without restrictions;
			- Has full administrative privileges across the entire Information System.

		- **Infrastructure-Manager:**
			- Can create any type definition without restrictions;
			- Has full administrative privileges across the entire Information System.

		- **All Other Users (including Context-Manager):**
			- Cannot create type definitions;
			- Will receive authorization errors if attempting these operations;
			- Type creation is restricted to prevent system-wide compatibility issues.


		**Operation Behavior:**

		**Type Update Policy:**

		**No REST Update Method Exposed:**
		- Although the underlying code supports type updates, the update functionality is intentionally NOT exposed via REST API;
		- Type modifications can break compatibility across the entire Information System;
		- Examples of breaking changes:
			- Making a previously optional facet property mandatory;
			- Changing property types or constraints;
			- Modifying validation rules that could invalidate existing instances.

		**Offline Update Process:**
		- Type updates must be performed offline after careful impact assessment;
		- Changes require evaluation of all existing instances and dependent systems;
		- Updates are applied through controlled maintenance procedures;
		- This ensures system stability and prevents breaking existing client applications.
		"""
	)
	@APIResponse(
		responseCode = "201",
		description = "Type successfully created",
		content = @Content(mediaType = "application/json")
	)
	@APIResponse(
		responseCode = "409",
		description = "Type already exists with the given name"
	)
	@APIResponse(
		responseCode = "400",
		description = "Invalid JSON schema or malformed request"
	)
	@APIResponse(
		responseCode = "403",
		description = "User lacks authorization to create types"
	)
	public Response create(
			@PathParam(TypeManager.TYPE_PATH_PARAMETER)
			@Parameter(
				name = TypeManager.TYPE_PATH_PARAMETER,
				in = ParameterIn.PATH,
				description = """
				The name of the type to create.
				- Must be unique in the system (e.g., "ContactFacet", "MyCustomResource").
				""",
				required = true,
				schema = @Schema(type = SchemaType.STRING, example = "ContactFacet")
			)
			String typeName, String json)
			throws SchemaException, ResourceRegistryException {
		logger.info("Requested {} creation with schema {}", typeName, json);
		setAccountingMethod(Method.CREATE, Type.NAME);
		
		ServerRequestInfo serverRequestInfo = initRequestInfo();
		serverRequestInfo.setIncludeMeta(true);
		serverRequestInfo.setAllMeta(true);
		
		TypeManagement typeManagement = new TypeManagement();
		typeManagement.setTypeName(typeName);
		typeManagement.setJson(json);
		String ret = typeManagement.create();
		return Response.status(Status.CREATED).entity(ret).type(ResourceInitializer.APPLICATION_JSON_CHARSET_UTF_8)
				.build();
	}
	
	/**
	 * Retrieves the schema definition(s) for the specified type.
	 * The response can include subtypes and metadata based on query parameters and user authorization.
	 * 
	 * <strong>REST Endpoint:</strong> {@code GET /types/{type-name}[?polymorphic={true|false}&includeMeta={true|false}]}
	 * 
	 * <strong>Request Examples:</strong>
	 * <ul>
	 * <li>GET /types/ContactFacet (returns basic ContactFacet type definition only)</li>
	 * <li>GET /types/ContactFacet?polymorphic=true (returns ContactFacet and all its subtypes)</li>
	 * <li>GET /types/ContactFacet?includeMeta=true (returns ContactFacet with metadata for authorized users)</li>
	 * <li>GET /types/ContactFacet?polymorphic=true&amp;includeMeta=true (returns ContactFacet, subtypes, and metadata).</li>
	 * </ul>
	 * 
	 * <strong>Path Parameters:</strong>
	 * <ul>
	 * <li><strong>type-name</strong>: The name of the type to retrieve (case-sensitive).</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>polymorphic</strong> (optional):
	 * <ul>
	 * <li>Whether to include subtypes in the response</li>
	 * <li>Default value: false (returns only the specified type)</li>
	 * <li>When true: returns the specified type AND all its existing subtypes</li>
	 * <li>When false: returns only the specified type definition</li>
	 * <li>Note: No pagination is applied - ALL subtypes are returned when polymorphic=true</li>
	 * <li>Example: ?polymorphic=true (includes all subtypes of the requested type).</li>
	 * </ul>
	 * 
	 * <strong>includeMeta</strong> (optional):
	 * <ul>
	 * <li>Whether to include metadata in the response</li>
	 * <li>Default value: false (basic type information only)</li>
	 * <li>Values: true|false</li>
	 * <li>Example: ?includeMeta=true (includes metadata based on user authorization).</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * 
	 * <strong>IS-Manager:</strong>
	 * <ul>
	 * <li>Receive basic type information by default</li>
	 * <li>Can explicitly request metadata via includeMeta=true query parameter</li>
	 * <li>When metadata is requested, receive complete type definitions with full metadata</li>
	 * <li>All metadata fields are included when includeMeta=true</li>
	 * <li>No obfuscation or filtering of sensitive information.</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Receive basic type information by default</li>
	 * <li>Can explicitly request metadata via includeMeta=true query parameter</li>
	 * <li>When metadata is requested, receive complete type definitions with full metadata</li>
	 * <li>All metadata fields are included when includeMeta=true</li>
	 * <li>No obfuscation or filtering of sensitive information.</li>
	 * </ul>
	 * 
	 * <strong>All Other Users:</strong>
	 * <ul>
	 * <li>Receive basic type information by default</li>
	 * <li>Can explicitly request metadata via includeMeta=true query parameter</li>
	 * <li>When metadata is requested, receive metadata with sensitive information filtered:</li>
	 * </ul>
	 *   • Date fields (creation, modification) are visible;
	 *   • User identifiers (createdBy, lastUpdateBy) are obfuscated or hidden;
	 *   • Other administrative details may be filtered.
	 * 
	 * <strong>Response Codes:</strong>
	 * <ul>
	 * <li>200 OK: Type successfully retrieved</li>
	 * <li>404 Not Found: Type with the specified name does not exist</li>
	 * <li>403 Forbidden: User lacks authorization to access type information.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Content-Type: application/json</li>
	 * <li>Body: Type schema definition(s) with authorization-appropriate detail level</li>
	 * <li>When polymorphic=false: single type definition object</li>
	 * <li>When polymorphic=true: array of type definitions (parent + all subtypes).</li>
	 * </ul>
	 * 
	 * @param type the name of the type to retrieve (case-sensitive)
	 * @param polymorphic whether to include subtypes in the response (default: false)
	 * @return JSON containing type schema definition(s) with authorization-appropriate detail level
	 * @throws SchemaNotFoundException if the specified type is not found in the system
	 * @throws ResourceRegistryException for retrieval errors or other system failures
	 */
	@GET
	@Path("{" + TypeManager.TYPE_PATH_PARAMETER + "}")
	@Produces(ResourceInitializer.APPLICATION_JSON_CHARSET_UTF_8)
	@Operation(
		summary = "Read Type Schema(s)",
		description = """
		Retrieves the schema definition(s) for the specified type.

		The response can include subtypes and metadata based on query parameters and user authorization.


		**Request Examples:**
		- GET /types/ContactFacet (returns basic ContactFacet type definition only);
		- GET /types/ContactFacet?polymorphic=true (returns ContactFacet and all its subtypes);
		- GET /types/ContactFacet?includeMeta=true (returns ContactFacet with metadata for authorized users);
		- GET /types/ContactFacet?polymorphic=true&includeMeta=true (returns ContactFacet, subtypes, and metadata).


		**Authorization Requirements:**
		- **IS-Manager:**
			- Receive basic type information by default;
			- Can explicitly request metadata via includeMeta=true query parameter;
			- When metadata is requested, receive complete type definitions with full metadata;
			- All metadata fields are included when includeMeta=true;
			- No obfuscation or filtering of sensitive information.

		- **Infrastructure-Manager:**
			- Receive basic type information by default;
			- Can explicitly request metadata via includeMeta=true query parameter;
			- When metadata is requested, receive complete type definitions with full metadata;
			- All metadata fields are included when includeMeta=true;
			- No obfuscation or filtering of sensitive information.

		- **All Other Users:**
			- Receive basic type information by default;
			- Can explicitly request metadata via includeMeta=true query parameter;
			- When metadata is requested, receive metadata with sensitive information filtered:
				- Date fields (creation, modification) are visible;
				- User identifiers (createdBy, lastUpdateBy) are obfuscated or hidden;
				- Other administrative details may be filtered.
		"""
	)
	@Parameter(
		name = TypePath.POLYMORPHIC_QUERY_PARAMETER,
		in = ParameterIn.QUERY,
		description = """
		Whether to include subtypes in the response.
		- Default: false (returns only the specified type);
		- When true: returns the specified type AND all its existing subtypes;
		- When false: returns only the specified type definition;
		- Note: No pagination is applied - ALL subtypes are returned when polymorphic=true;
		- Example: ?polymorphic=true (includes all subtypes of the requested type).
		""",
		required = false,
		schema = @Schema(type = SchemaType.BOOLEAN, defaultValue = "false", example = "false")
	)
	@Parameter(
		name = TypePath.INCLUDE_META_QUERY_PARAMETER,
		in = ParameterIn.QUERY,
		description = """
		Whether to include metadata in the response.
		- Default: false (basic type information only);
		- Values: true|false;
		- Example: ?includeMeta=true (includes metadata based on user authorization).
		""",
		required = false,
		schema = @Schema(type = SchemaType.BOOLEAN, defaultValue = "false", example = "false")
	)
	@APIResponse(
		responseCode = "200",
		description = "Type successfully retrieved",
		content = @Content(mediaType = "application/json")
	)
	@APIResponse(
		responseCode = "404",
		description = "Type with the specified name does not exist"
	)
	@APIResponse(
		responseCode = "403",
		description = "User lacks authorization to access type information"
	)
	public String read(
			@PathParam(TypeManager.TYPE_PATH_PARAMETER)
			@Parameter(
				name = TypeManager.TYPE_PATH_PARAMETER,
				in = ParameterIn.PATH,
				description = "The name of the type to retrieve (case-sensitive)",
				required = true,
				schema = @Schema(type = SchemaType.STRING, example = "ContactFacet")
			)
			String type,
			@QueryParam(TypePath.POLYMORPHIC_QUERY_PARAMETER) @DefaultValue("false") Boolean polymorphic)
			throws SchemaNotFoundException, ResourceRegistryException {
		logger.info("Requested Schema for type {}", type);
		setAccountingMethod(Method.READ, Type.NAME);
		
		ServerRequestInfo serverRequestInfo = initRequestInfo();
		serverRequestInfo.setAllMeta(true);
		serverRequestInfo.checkBooleanQueryParameter(TypePath.INCLUDE_META_QUERY_PARAMETER);
		
		TypeManagement typeManagement = new TypeManagement();
		typeManagement.setTypeName(type);
		List<Type> types = typeManagement.read(polymorphic);
		try {
			return TypeMapper.serializeTypeDefinitions(types);
		}catch (Exception e) {
			throw new ResourceRegistryException(e);
		}
	}
	
}
