package org.gcube.informationsystem.resourceregistry.instances.base;

import java.util.Optional;
import java.util.UUID;

import org.gcube.informationsystem.base.reference.AccessType;
import org.gcube.informationsystem.model.reference.entities.Entity;
import org.gcube.informationsystem.model.reference.entities.Facet;
import org.gcube.informationsystem.model.reference.entities.Resource;
import org.gcube.informationsystem.model.reference.relations.ConsistsOf;
import org.gcube.informationsystem.model.reference.relations.IsRelatedTo;
import org.gcube.informationsystem.model.reference.relations.Relation;
import org.gcube.informationsystem.resourceregistry.api.exceptions.NotFoundException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.ResourceRegistryException;
import org.gcube.informationsystem.resourceregistry.contexts.security.SecurityContext;
import org.gcube.informationsystem.resourceregistry.instances.model.entities.EntityManagement;
import org.gcube.informationsystem.resourceregistry.instances.model.entities.FacetManagement;
import org.gcube.informationsystem.resourceregistry.instances.model.entities.ResourceManagement;
import org.gcube.informationsystem.resourceregistry.instances.model.relations.ConsistsOfManagement;
import org.gcube.informationsystem.resourceregistry.instances.model.relations.IsRelatedToManagement;
import org.gcube.informationsystem.resourceregistry.instances.model.relations.RelationManagement;
import org.gcube.informationsystem.resourceregistry.types.TypesCache;
import org.gcube.informationsystem.resourceregistry.utils.OrientDBUtility;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.orientechnologies.orient.core.db.document.ODatabaseDocument;
import com.orientechnologies.orient.core.metadata.schema.OClass;
import com.orientechnologies.orient.core.record.OEdge;
import com.orientechnologies.orient.core.record.OElement;
import com.orientechnologies.orient.core.record.OVertex;

/**
 * @author Luca Frosini (ISTI - CNR)
 */
public class ElementManagementUtility {
	
	private static Logger logger = LoggerFactory.getLogger(ElementManagementUtility.class);
	
	public static ElementManagement<?,?> getERManagement(String type) throws ResourceRegistryException {
		
		AccessType accessType = TypesCache.getInstance().getCachedType(type).getAccessType();
		
		ElementManagement<?,?> erManagement = null;
		
		switch (accessType) {
			case RESOURCE:
				erManagement = new ResourceManagement();
				break;
	
			case FACET:
				erManagement = new FacetManagement();
				break;
			
			case IS_RELATED_TO:
				erManagement = new IsRelatedToManagement();
				break;
				
			case CONSISTS_OF:
				erManagement = new ConsistsOfManagement();
				break;
			
			default:
				throw new ResourceRegistryException(String.format("%s is not querable", type.toString()));
		}
		erManagement.setElementType(type);
		
		return erManagement;
	}
	
	public static ElementManagement<?,?> getERManagement(SecurityContext workingContext, ODatabaseDocument orientGraph,
			OElement element) throws ResourceRegistryException {
		if(element instanceof OVertex) {
			return getEntityManagement(workingContext, orientGraph, (OVertex) element);
		} else if(element instanceof OEdge) {
			return getRelationManagement(workingContext, orientGraph, (OEdge) element);
		}
		throw new ResourceRegistryException(String.format("%s is not a %s nor a %s", element.getClass().getSimpleName(),
				Entity.NAME, Relation.NAME));
	}
	
	public static OElement getAnyElementByUUID(UUID uuid) throws NotFoundException, ResourceRegistryException {
		try {
			return OrientDBUtility.getElementByUUIDAsAdmin(null, uuid, OVertex.class);
		} catch(NotFoundException e) {
			return OrientDBUtility.getElementByUUIDAsAdmin(null, uuid, OEdge.class);
		} catch(ResourceRegistryException e) {
			throw e;
		} catch(Exception e) {
			throw new ResourceRegistryException(e);
		}
	}
	
	public static OElement getAnyElementByUUID(ODatabaseDocument oDatabaseDocument, UUID uuid)
			throws NotFoundException, ResourceRegistryException {
		try {
			return OrientDBUtility.getElementByUUID(oDatabaseDocument, null, uuid, OVertex.class);
		} catch(NotFoundException e) {
			return OrientDBUtility.getElementByUUID(oDatabaseDocument, null, uuid, OEdge.class);
		} catch(ResourceRegistryException e) {
			throw e;
		} catch(Exception e) {
			throw new ResourceRegistryException(e);
		}
	}
	
	public static ElementManagement<?, ?> getERManagementFromUUID(SecurityContext workingContext, ODatabaseDocument orientGraph,
			UUID uuid) throws ResourceRegistryException {
		OElement element;
		try {
			element = getAnyElementByUUID(orientGraph, uuid);
			return getERManagement(workingContext, orientGraph, element);
		} catch(Exception e) {
			throw new ResourceRegistryException(String.format("%s does not belong to an %s nor to a %s",
					uuid.toString(), Entity.NAME, Relation.NAME));
		}
	}
	
	public static EntityManagement<?, ?> getEntityManagement(SecurityContext workingContext, ODatabaseDocument oDatabaseDocument,
			OVertex vertex) throws ResourceRegistryException {
		
		if(oDatabaseDocument == null) {
			throw new ResourceRegistryException(
					ODatabaseDocument.class.getSimpleName() + "instance is null. " + OrientDBUtility.SHOULD_NOT_OCCUR_ERROR_MESSAGE);
		}
		
		if(vertex == null) {
			throw new ResourceRegistryException(
					OVertex.class.getSimpleName() + "instance is null. " + OrientDBUtility.SHOULD_NOT_OCCUR_ERROR_MESSAGE);
		}
		
		OClass oClass = null;
		try {
			oClass = ElementManagementUtility.getOClass(vertex);
		} catch(Exception e) {
			String error = String.format("Unable to detect type of %s. %s", vertex.toString(),
					OrientDBUtility.SHOULD_NOT_OCCUR_ERROR_MESSAGE);
			logger.error(error, e);
			throw new ResourceRegistryException(error);
		}
		
		EntityManagement<?, ?> entityManagement = null;
		if(oClass.isSubClassOf(Resource.NAME)) {
			entityManagement = new ResourceManagement();
		} else if(oClass.isSubClassOf(Facet.NAME)) {
			entityManagement = new FacetManagement();
		} else {
			String error = String.format("{%s is not a %s nor a %s. %s", vertex, Resource.NAME, Facet.NAME,
					OrientDBUtility.SHOULD_NOT_OCCUR_ERROR_MESSAGE);
			throw new ResourceRegistryException(error);
		}
		entityManagement.setODatabaseDocument(oDatabaseDocument);
		entityManagement.setWorkingContext(workingContext);
		entityManagement.setElement(vertex);
		return entityManagement;
	}
	
	public static RelationManagement<?,?> getRelationManagement(SecurityContext workingContext, ODatabaseDocument oDatabaseDocument,
			OEdge edge) throws ResourceRegistryException {
		
		if(oDatabaseDocument == null) {
			throw new ResourceRegistryException(
					ODatabaseDocument.class.getSimpleName() + "instance is null. " + OrientDBUtility.SHOULD_NOT_OCCUR_ERROR_MESSAGE);
		}
		
		if(edge == null) {
			throw new ResourceRegistryException(
					OEdge.class.getSimpleName() + "instance is null. " + OrientDBUtility.SHOULD_NOT_OCCUR_ERROR_MESSAGE);
		}
		
		OClass oClass = ElementManagementUtility.getOClass(edge);
		
		RelationManagement<?,?> relationManagement = null;
		if(oClass.isSubClassOf(ConsistsOf.NAME)) {
			relationManagement = new ConsistsOfManagement();
		} else if(oClass.isSubClassOf(IsRelatedTo.NAME)) {
			relationManagement = new IsRelatedToManagement();
		} else {
			String error = String.format("{%s is not a %s nor a %s. %s", edge, ConsistsOf.NAME, IsRelatedTo.NAME,
					OrientDBUtility.SHOULD_NOT_OCCUR_ERROR_MESSAGE);
			throw new ResourceRegistryException(error);
		}
		
		
		relationManagement.setODatabaseDocument(oDatabaseDocument);
		relationManagement.setWorkingContext(workingContext);
		
		relationManagement.setElement(edge);
		return relationManagement;
	}
	
	public static <E extends OElement> E getElementFromOptional(Optional<E> optional) throws ResourceRegistryException {
		if(optional.isPresent()) {
			return optional.get();
		}else {
			throw new ResourceRegistryException("An element not belonging to any defined type should not exists. Please contact the administrator.");
		}
	}
	
	public static OClass getOClass(OElement oElement) throws ResourceRegistryException {
		Optional<OClass> optional = oElement.getSchemaType();
		if(optional.isPresent()) {
			return optional.get();
		}else {
			throw new ResourceRegistryException("An element not belonging to any defined type should not exists. Please contact the administrator.");
		}
	}
	
}
