package org.gcube.informationsystem.resourceregistry.instances.base;

import java.util.UUID;

import org.gcube.informationsystem.base.reference.AccessType;
import org.gcube.informationsystem.model.reference.entities.Entity;
import org.gcube.informationsystem.model.reference.entities.Facet;
import org.gcube.informationsystem.model.reference.entities.Resource;
import org.gcube.informationsystem.model.reference.properties.Property;
import org.gcube.informationsystem.model.reference.relations.ConsistsOf;
import org.gcube.informationsystem.model.reference.relations.IsRelatedTo;
import org.gcube.informationsystem.model.reference.relations.Relation;
import org.gcube.informationsystem.resourceregistry.api.exceptions.NotFoundException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.ResourceRegistryException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.schema.SchemaException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.schema.SchemaNotFoundException;
import org.gcube.informationsystem.resourceregistry.contexts.ContextUtility;
import org.gcube.informationsystem.resourceregistry.contexts.security.AdminSecurityContext;
import org.gcube.informationsystem.resourceregistry.contexts.security.SecurityContext;
import org.gcube.informationsystem.resourceregistry.contexts.security.SecurityContext.PermissionMode;
import org.gcube.informationsystem.resourceregistry.instances.model.entities.EntityManagement;
import org.gcube.informationsystem.resourceregistry.instances.model.entities.FacetManagement;
import org.gcube.informationsystem.resourceregistry.instances.model.entities.ResourceManagement;
import org.gcube.informationsystem.resourceregistry.instances.model.relations.ConsistsOfManagement;
import org.gcube.informationsystem.resourceregistry.instances.model.relations.IsRelatedToManagement;
import org.gcube.informationsystem.resourceregistry.instances.model.relations.RelationManagement;
import org.gcube.informationsystem.resourceregistry.utils.Utility;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.orientechnologies.orient.core.db.document.ODatabaseDocument;
import com.orientechnologies.orient.core.metadata.OMetadata;
import com.orientechnologies.orient.core.metadata.schema.OClass;
import com.orientechnologies.orient.core.metadata.schema.OSchema;
import com.orientechnologies.orient.core.record.OEdge;
import com.orientechnologies.orient.core.record.OElement;
import com.orientechnologies.orient.core.record.OVertex;

/**
 * @author Luca Frosini (ISTI - CNR)
 */
public class ElementManagementUtility {
	
	private static Logger logger = LoggerFactory.getLogger(ElementManagementUtility.class);
	
	public static AccessType getBaseAccessType(String type) throws ResourceRegistryException {
		
		OClass oClass = ElementManagementUtility.getTypeSchema(type, null);
		
		if(oClass.isSubClassOf(Resource.NAME)) {
			return AccessType.RESOURCE;
		} else if(oClass.isSubClassOf(Facet.NAME)) {
			return AccessType.FACET;
		} else if(oClass.isSubClassOf(ConsistsOf.NAME)) {
			return AccessType.CONSISTS_OF;
		} else if(oClass.isSubClassOf(IsRelatedTo.NAME)) {
			return AccessType.IS_RELATED_TO;
		} else if(oClass.isSubClassOf(Property.NAME)) {
			return AccessType.PROPERTY;
		}
		
		throw new ResourceRegistryException(type + "is not a base type");
		
	}
	
	
	@SuppressWarnings("rawtypes")
	public static ElementManagement getERManagement(String type) throws ResourceRegistryException {
		
		OClass oClass = ElementManagementUtility.getTypeSchema(type, null);
		ElementManagement erManagement = null;
		
		if(oClass.isSubClassOf(Resource.NAME)) {
			erManagement = new ResourceManagement();
		} else if(oClass.isSubClassOf(Facet.NAME)) {
			erManagement = new FacetManagement();
		} else if(oClass.isSubClassOf(ConsistsOf.NAME)) {
			erManagement = new ConsistsOfManagement();
		} else if(oClass.isSubClassOf(IsRelatedTo.NAME)) {
			erManagement = new IsRelatedToManagement();
		}
		
		if(erManagement == null) {
			throw new ResourceRegistryException(String.format("%s is not querable", type.toString()));
		}
		
		erManagement.setElementType(type);
		return erManagement;
	}
	
	public static ElementManagement<?> getERManagement(SecurityContext workingContext, ODatabaseDocument orientGraph,
			OElement element) throws ResourceRegistryException {
		if(element instanceof OVertex) {
			return getEntityManagement(workingContext, orientGraph, (OVertex) element);
		} else if(element instanceof OEdge) {
			return getRelationManagement(workingContext, orientGraph, (OEdge) element);
		}
		throw new ResourceRegistryException(String.format("%s is not a %s nor a %s", element.getClass().getSimpleName(),
				Entity.NAME, Relation.NAME));
	}
	
	public static OElement getAnyElementByUUID(UUID uuid) throws NotFoundException, ResourceRegistryException {
		try {
			return Utility.getElementByUUIDAsAdmin(null, uuid, OVertex.class);
		} catch(NotFoundException e) {
			return Utility.getElementByUUIDAsAdmin(null, uuid, OEdge.class);
		} catch(ResourceRegistryException e) {
			throw e;
		} catch(Exception e) {
			throw new ResourceRegistryException(e);
		}
	}
	
	private static OElement getAnyElementByUUID(ODatabaseDocument orientGraph, UUID uuid)
			throws NotFoundException, ResourceRegistryException {
		try {
			return Utility.getElementByUUID(orientGraph, null, uuid, OVertex.class);
		} catch(NotFoundException e) {
			return Utility.getElementByUUID(orientGraph, null, uuid, OEdge.class);
		} catch(ResourceRegistryException e) {
			throw e;
		} catch(Exception e) {
			throw new ResourceRegistryException(e);
		}
	}
	
	@SuppressWarnings("rawtypes")
	public static ElementManagement getERManagementFromUUID(SecurityContext workingContext, ODatabaseDocument orientGraph,
			UUID uuid) throws ResourceRegistryException {
		OElement element;
		try {
			element = getAnyElementByUUID(orientGraph, uuid);
			return getERManagement(workingContext, orientGraph, element);
		} catch(Exception e) {
			throw new ResourceRegistryException(String.format("%s does not belong to an %s nor to a %s",
					uuid.toString(), Entity.NAME, Relation.NAME));
		}
	}
	
	@SuppressWarnings({"rawtypes", "unchecked"})
	public static EntityManagement getEntityManagement(SecurityContext workingContext, ODatabaseDocument oDatabaseDocument,
			OVertex vertex) throws ResourceRegistryException {
		
		if(oDatabaseDocument == null) {
			throw new ResourceRegistryException(
					ODatabaseDocument.class.getSimpleName() + "instance is null. " + Utility.SHOULD_NOT_OCCUR_ERROR_MESSAGE);
		}
		
		if(vertex == null) {
			throw new ResourceRegistryException(
					OVertex.class.getSimpleName() + "instance is null. " + Utility.SHOULD_NOT_OCCUR_ERROR_MESSAGE);
		}
		
		OClass oClass = null;
		try {
			oClass = ElementManagement.getOClass(vertex);
		} catch(Exception e) {
			String error = String.format("Unable to detect type of %s. %s", vertex.toString(),
					Utility.SHOULD_NOT_OCCUR_ERROR_MESSAGE);
			logger.error(error, e);
			throw new ResourceRegistryException(error);
		}
		
		EntityManagement entityManagement = null;
		if(oClass.isSubClassOf(Resource.NAME)) {
			entityManagement = new ResourceManagement(workingContext, oDatabaseDocument);
		} else if(oClass.isSubClassOf(Facet.NAME)) {
			entityManagement = new FacetManagement(workingContext, oDatabaseDocument);
		} else {
			String error = String.format("{%s is not a %s nor a %s. %s", vertex, Resource.NAME, Facet.NAME,
					Utility.SHOULD_NOT_OCCUR_ERROR_MESSAGE);
			throw new ResourceRegistryException(error);
		}
		entityManagement.setElement(vertex);
		return entityManagement;
	}
	
	@SuppressWarnings({"unchecked", "rawtypes"})
	public static RelationManagement getRelationManagement(SecurityContext workingContext, ODatabaseDocument oDatabaseDocument,
			OEdge edge) throws ResourceRegistryException {
		
		if(oDatabaseDocument == null) {
			throw new ResourceRegistryException(
					ODatabaseDocument.class.getSimpleName() + "instance is null. " + Utility.SHOULD_NOT_OCCUR_ERROR_MESSAGE);
		}
		
		if(edge == null) {
			throw new ResourceRegistryException(
					OEdge.class.getSimpleName() + "instance is null. " + Utility.SHOULD_NOT_OCCUR_ERROR_MESSAGE);
		}
		
		OClass oClass = ElementManagement.getOClass(edge);
		
		RelationManagement relationManagement = null;
		if(oClass.isSubClassOf(ConsistsOf.NAME)) {
			relationManagement = new ConsistsOfManagement(workingContext, oDatabaseDocument);
		} else if(oClass.isSubClassOf(IsRelatedTo.NAME)) {
			relationManagement = new IsRelatedToManagement(workingContext, oDatabaseDocument);
		} else {
			String error = String.format("{%s is not a %s nor a %s. %s", edge, ConsistsOf.NAME, IsRelatedTo.NAME,
					Utility.SHOULD_NOT_OCCUR_ERROR_MESSAGE);
			throw new ResourceRegistryException(error);
		}
		relationManagement.setElement(edge);
		return relationManagement;
	}
	
	public static OClass getTypeSchema(ODatabaseDocument oDatabaseDocument, String type, AccessType accessType)
			throws SchemaException, SchemaNotFoundException {
		OMetadata oMetadata = oDatabaseDocument.getMetadata();
		OSchema oSchema = oMetadata.getSchema();
		return getTypeSchema(oSchema, type, accessType);
	}
	
	public static OClass getTypeSchema(OSchema oSchema, String type, AccessType accessType)
			throws SchemaException, SchemaNotFoundException {
		try {
			OClass oClass = oSchema.getClass(type);
			if(oClass == null) {
				throw new SchemaNotFoundException(type + " was not registered");
			}
			if(accessType != null && type.compareTo(accessType.getName()) != 0) {
				if(!oClass.isSubClassOf(accessType.getName())) {
					throw new SchemaException(type + " is not a " + accessType.getName());
				}
			}
			return oClass;
		} catch(SchemaNotFoundException snfe) {
			throw snfe;
		} catch(Exception e) {
			throw new SchemaException(e.getMessage());
		}
	}
	
	public static OClass getTypeSchema(String type, AccessType accessType)
			throws SchemaException, ResourceRegistryException {
		ODatabaseDocument current = ContextUtility.getCurrentODatabaseDocumentFromThreadLocal();
		ODatabaseDocument oDatabaseDocument = null;
		try {
			logger.debug("Getting {} Type {} schema", accessType != null ? accessType.getName() : "", type);
			AdminSecurityContext adminSecurityContext = ContextUtility.getAdminSecurityContext();
			oDatabaseDocument = adminSecurityContext.getDatabaseDocument(PermissionMode.READER);
			return getTypeSchema(oDatabaseDocument, type, accessType);
		} catch(ResourceRegistryException e) {
			throw e;
		} catch(Exception e) {
			throw new ResourceRegistryException(e);
		} finally {
			if(oDatabaseDocument != null) {
				oDatabaseDocument.close();
			}
			
			if(current!=null) {
				current.activateOnCurrentThread();
			}
		}
	}
	
}
