package org.gcube.informationsystem.resourceregistry.publisher;

import java.util.List;
import java.util.Set;
import java.util.UUID;

import org.gcube.informationsystem.base.reference.IdentifiableElement;
import org.gcube.informationsystem.context.reference.entities.Context;
import org.gcube.informationsystem.model.reference.entities.Facet;
import org.gcube.informationsystem.model.reference.entities.Resource;
import org.gcube.informationsystem.model.reference.relations.ConsistsOf;
import org.gcube.informationsystem.model.reference.relations.IsRelatedTo;
import org.gcube.informationsystem.resourceregistry.api.exceptions.AlreadyPresentException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.AvailableInAnotherContextException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.NotFoundException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.ResourceRegistryException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.context.ContextNotFoundException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.entity.facet.FacetAlreadyPresentException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.entity.facet.FacetNotFoundException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.entity.resource.ResourceAlreadyPresentException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.entity.resource.ResourceNotFoundException;

/**
 * @author Luca Frosini (ISTI - CNR)
 */
public interface ResourceRegistryPublisher {

	public <IE extends IdentifiableElement> IE create(IE er)
			throws AlreadyPresentException, ResourceRegistryException;
	
	public String create(String json) throws AlreadyPresentException, ResourceRegistryException;
	
	
	public <IE extends IdentifiableElement> boolean exists(Class<IE> clazz, UUID uuid)
			throws NotFoundException, AvailableInAnotherContextException, ResourceRegistryException;
	
	public <IE extends IdentifiableElement> boolean exists(IE identifiableElement)
			throws NotFoundException, AvailableInAnotherContextException, ResourceRegistryException;
	
	public boolean exists(String type, UUID uuid)
			throws NotFoundException, AvailableInAnotherContextException, ResourceRegistryException;
	
	
	public <IE extends IdentifiableElement> IE read(IE identifiableElement) throws NotFoundException, ResourceRegistryException;

	public String read(String type, UUID uuid) throws NotFoundException, ResourceRegistryException;
	
	
	public <IE extends IdentifiableElement> IE update(IE identifiableElement)
			throws NotFoundException, ResourceRegistryException;
	
	public String update(String type, String json) throws NotFoundException, ResourceRegistryException;
	
	public String update(String json) throws NotFoundException, ResourceRegistryException;

	
	public <IE extends IdentifiableElement> boolean delete(IE identifiableElement) throws NotFoundException, ResourceRegistryException;

	public boolean delete(String type, UUID uuid) throws NotFoundException, ResourceRegistryException;
	
	
	/* ----- */
	
	
	public <F extends Facet> F createFacet(F facet)
			throws FacetAlreadyPresentException, ResourceRegistryException;
	
	public String createFacet(String facet) throws FacetAlreadyPresentException, ResourceRegistryException;
	
	
	public <F extends Facet> F readFacet(F facet) throws FacetNotFoundException, ResourceRegistryException;

	public String readFacet(String facetType, UUID uuid) throws FacetNotFoundException, ResourceRegistryException;
	
	
	public <F extends Facet> F updateFacet(F facet)
			throws FacetNotFoundException, ResourceRegistryException;

	public String updateFacet(String facet) throws FacetNotFoundException, ResourceRegistryException;

	
	public <F extends Facet> boolean deleteFacet(F facet) throws FacetNotFoundException, ResourceRegistryException;

	public boolean deleteFacet(String facetType, UUID uuid) throws FacetNotFoundException, ResourceRegistryException;

	
	/* ----- */
	
	
	public <R extends Resource> R createResource(R resource)
			throws ResourceAlreadyPresentException, ResourceRegistryException;

	public String createResource(String resource) throws ResourceAlreadyPresentException, ResourceRegistryException;
	
	
	public <R extends Resource> R readResource(R resource)
			throws ResourceNotFoundException, ResourceRegistryException;

	public String readResource(String resourceType, UUID uuid) throws ResourceNotFoundException, ResourceRegistryException;
	

	public <R extends Resource> R updateResource(R resource)
			throws ResourceNotFoundException, ResourceRegistryException;

	public String updateResource(String resource)
			throws ResourceNotFoundException, ResourceRegistryException;
	
	
	public <R extends Resource> boolean deleteResource(R resource)
			throws ResourceNotFoundException, ResourceRegistryException;

	public boolean deleteResource(String resourceType, UUID uuid) throws ResourceNotFoundException, ResourceRegistryException;

	
	/* ----- */
	
	
	public <C extends ConsistsOf<? extends Resource, ? extends Facet>> C createConsistsOf(
			C consistsOf) throws NotFoundException, ResourceRegistryException;

	public String createConsistsOf(String consistsOf)
			throws NotFoundException, ResourceRegistryException;
	
	
	public <C extends ConsistsOf<? extends Resource, ? extends Facet>> C readConsistsOf(
			C consistsOf) throws NotFoundException, ResourceRegistryException;

	public String readConsistsOf(String consistsOfType, UUID uuid) throws NotFoundException, ResourceRegistryException;
	

	public <C extends ConsistsOf<? extends Resource, ? extends Facet>> C updateConsistsOf(C consistsOf)
			throws NotFoundException, ResourceRegistryException;

	public String updateConsistsOf(String consistsOf)
			throws NotFoundException, ResourceRegistryException;
	
	
	public <C extends ConsistsOf<? extends Resource, ? extends Facet>> boolean deleteConsistsOf(C consistsOf)
			throws ResourceRegistryException;

	public boolean deleteConsistsOf(String consistsOfType, UUID uuid) throws ResourceRegistryException;

	
	/* ----- */
	
	
	public <I extends IsRelatedTo<? extends Resource, ? extends Resource>> I createIsRelatedTo(
			I isRelatedTo) throws ResourceNotFoundException, ResourceRegistryException;

	public String createIsRelatedTo(String isRelatedTo)
			throws ResourceNotFoundException, ResourceRegistryException;
	
	
	public <I extends IsRelatedTo<? extends Resource, ? extends Resource>> I readIsRelatedTo(
			I isRelatedTo) throws NotFoundException, ResourceRegistryException;

	public String readIsRelatedTo(String isRelatedToType, UUID uuid) throws NotFoundException, ResourceRegistryException;
	

	public <I extends IsRelatedTo<? extends Resource, ? extends Resource>> I updateIsRelatedTo(I isRelatedTo)
			throws NotFoundException, ResourceRegistryException;

	public String updateIsRelatedTo(String isRelatedTo)
			throws NotFoundException, ResourceRegistryException;
	
	
	public <I extends IsRelatedTo<? extends Resource, ? extends Resource>> boolean deleteIsRelatedTo(I isRelatedTo)
			throws ResourceRegistryException;

	public boolean deleteIsRelatedTo(String isRelatedToType, UUID uuid) throws ResourceRegistryException;

	
	/* ----- */
	
	
	public boolean addToContext(String type, UUID instanceUUID, UUID contextUUID)
			throws NotFoundException, ResourceRegistryException;

	public <IE extends IdentifiableElement> boolean addToContext(IE identifiableElement, UUID contextUUID)
			throws NotFoundException, ResourceRegistryException;
	
	public boolean addToCurrentContext(String type, UUID instanceUUID)
			throws NotFoundException, ResourceRegistryException;

	public <IE extends IdentifiableElement> boolean addToCurrentContext(IE identifiableElement)
			throws NotFoundException, ResourceRegistryException;

	public boolean removeFromContext(String type, UUID instanceUUID, UUID contextUUID)
			throws NotFoundException, ResourceRegistryException;

	public <IE extends IdentifiableElement> boolean removeFromContext(IE identifiableElement, UUID contextUUID)
			throws ResourceNotFoundException, ContextNotFoundException, ResourceRegistryException;
	
	public boolean removeFromCurrentContext(String type, UUID instanceUUID)
			throws ResourceNotFoundException, ContextNotFoundException, ResourceRegistryException;

	public <IE extends IdentifiableElement> boolean removeFromCurrentContext(IE identifiableElement)
			throws ResourceNotFoundException, ContextNotFoundException, ResourceRegistryException;
	
	public Set<UUID> getElementContexts(String type, UUID instanceUUID)
			throws ResourceNotFoundException, ContextNotFoundException, ResourceRegistryException;

	public <IE extends IdentifiableElement> Set<UUID> getElementContexts(IE identifiableElement)
			throws ResourceNotFoundException, ContextNotFoundException, ResourceRegistryException;
	
	
	/* ----- */
	
	
	public boolean addResourceToContext(String resourceType, UUID resourceUUID, UUID contextUUID)
			throws ResourceNotFoundException, ContextNotFoundException, ResourceRegistryException;

	public <R extends Resource> boolean addResourceToContext(R resource, UUID contextUUID)
			throws ResourceNotFoundException, ContextNotFoundException, ResourceRegistryException;
	
	public boolean addResourceToCurrentContext(String resourceType, UUID resourceUUID)
			throws ResourceNotFoundException, ContextNotFoundException, ResourceRegistryException;

	public <R extends Resource> boolean addResourceToCurrentContext(R resource)
			throws ResourceNotFoundException, ContextNotFoundException, ResourceRegistryException;

	public boolean removeResourceFromContext(String resourceType, UUID resourceUUID, UUID contextUUID)
			throws ResourceNotFoundException, ContextNotFoundException, ResourceRegistryException;

	public <R extends Resource> boolean removeResourceFromContext(R resource, UUID contextUUID)
			throws ResourceNotFoundException, ContextNotFoundException, ResourceRegistryException;
	
	public boolean removeResourceFromCurrentContext(String resourceType, UUID resourceUUID)
			throws ResourceNotFoundException, ContextNotFoundException, ResourceRegistryException;

	public <R extends Resource> boolean removeResourceFromCurrentContext(R resource)
			throws ResourceNotFoundException, ContextNotFoundException, ResourceRegistryException;
	
	public Set<UUID> getResourceContexts(String resourceType, UUID resourceUUID)
			throws ResourceNotFoundException, ContextNotFoundException, ResourceRegistryException;

	public <R extends Resource> Set<UUID> getResourceContexts(R resource)
			throws ResourceNotFoundException, ContextNotFoundException, ResourceRegistryException;
	
	
	/* ----- */
	
	
	public boolean addFacetToContext(String facetType, UUID facetUUID, UUID contextUUID)
			throws FacetNotFoundException, ContextNotFoundException, ResourceRegistryException;

	public <F extends Facet> boolean addFacetToContext(F facet, UUID contextUUID)
			throws FacetNotFoundException, ContextNotFoundException, ResourceRegistryException;
	
	public boolean addFacetToCurrentContext(String facetType, UUID facetUUID)
			throws FacetNotFoundException, ContextNotFoundException, ResourceRegistryException;

	public <F extends Facet> boolean addFacetToCurrentContext(F facet)
			throws FacetNotFoundException, ContextNotFoundException, ResourceRegistryException;
	
	public boolean removeFacetFromContext(String facetType, UUID facetUUID, UUID contextUUID)
			throws FacetNotFoundException, ContextNotFoundException, ResourceRegistryException;

	public <F extends Facet> boolean removeFacetFromContext(F facet, UUID contextUUID)
			throws FacetNotFoundException, ContextNotFoundException, ResourceRegistryException;
	
	public boolean removeFacetFromCurrentContext(String facetType, UUID facetUUID)
			throws FacetNotFoundException, ContextNotFoundException, ResourceRegistryException;

	public <F extends Facet> boolean removeFacetFromCurrentContext(F facet)
			throws FacetNotFoundException, ContextNotFoundException, ResourceRegistryException;
	
	public Set<UUID> getFacetContexts(String facetType, UUID facetUUID)
			throws FacetNotFoundException, ContextNotFoundException, ResourceRegistryException;

	public <F extends Facet> Set<UUID> getFacetContexts(F facet)
			throws FacetNotFoundException, ContextNotFoundException, ResourceRegistryException;
	
	
	/* ----- */
	
	
	public Context getContext(UUID uuid) throws ContextNotFoundException, ResourceRegistryException;
	
	public Context getCurrentContext() throws ContextNotFoundException, ResourceRegistryException;
	
	public List<Context> getAllContext() throws ResourceRegistryException;
	
	
}
