package org.gcube.informationsystem.resourceregistry.client;

import java.util.List;
import java.util.Map;
import java.util.UUID;

import org.gcube.com.fasterxml.jackson.databind.JsonNode;
import org.gcube.informationsystem.base.reference.Direction;
import org.gcube.informationsystem.contexts.reference.entities.Context;
import org.gcube.informationsystem.model.knowledge.ModelKnowledge;
import org.gcube.informationsystem.model.reference.ERElement;
import org.gcube.informationsystem.model.reference.ModelElement;
import org.gcube.informationsystem.model.reference.entities.Entity;
import org.gcube.informationsystem.model.reference.entities.Facet;
import org.gcube.informationsystem.model.reference.entities.Resource;
import org.gcube.informationsystem.model.reference.relations.ConsistsOf;
import org.gcube.informationsystem.model.reference.relations.IsRelatedTo;
import org.gcube.informationsystem.queries.templates.reference.entities.QueryTemplate;
import org.gcube.informationsystem.resourceregistry.api.contexts.ContextCache;
import org.gcube.informationsystem.resourceregistry.api.exceptions.AvailableInAnotherContextException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.NotFoundException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.ResourceRegistryException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.contexts.ContextNotFoundException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.queries.InvalidQueryException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.queries.templates.QueryTemplateNotFoundException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.types.SchemaNotFoundException;
import org.gcube.informationsystem.resourceregistry.api.request.RequestInfo;
import org.gcube.informationsystem.tree.Node;
import org.gcube.informationsystem.types.knowledge.TypeInformation;
import org.gcube.informationsystem.types.reference.Type;

/**
 * Client interface for querying and reading Information System entities, relations, and metadata from the Resource Registry.
 * 
 * <strong>Purpose:</strong>
 * This interface provides comprehensive read-only access to Information System model instances
 * (Resources, Facets, ConsistsOf relations, IsRelatedTo relations), type schemas, query templates,
 * and context information within the Resource Registry service.
 * 
 * <strong>Instantiation:</strong>
 * Clients should be created using the factory pattern for proper configuration:
 * <pre>
 * // Recommended way to create a client
 * ResourceRegistryClient client = ResourceRegistryClientFactory.create();
 * </pre>
 * 
 * <strong>Supported Operations:</strong>
 * <ul>
 * <li><strong>Context Operations</strong>: Access context information and hierarchy</li>
 * <li><strong>Schema Operations</strong>: Access type definitions, hierarchies, and model knowledge</li>
 * <li><strong>Instance Operations</strong>: Retrieve instance by type, UUID, or complex queries with filtering and pagination</li>
 * <li><strong>Query Operations</strong>: Execute custom queries and manage query templates</li>
 * <li><strong>Instance Navigation</strong>: Traverse relations between resources, facets, and other entities.</li>
 * </ul>
 * 
 * <strong>Query Parameter Configuration:</strong>
 * Most methods support optional query parameters that can be configured via the client configuration:
 * <ul>
 * <li><strong>includeMeta</strong>: Include metadata in responses (role-based filtering applied)</li>
 * <li><strong>allMeta</strong>: Include metadata for all nested instances</li>
 * <li><strong>includeContexts</strong>: Show context availability information</li>
 * <li><strong>hierarchical</strong>: Include instances from child contexts (admin-only)</li>
 * <li><strong>limit</strong>: Maximum number of results per request</li>
 * <li><strong>offset</strong>: Number of results to skip for pagination.</li>
 * </ul>
 * 
 * <strong>Authorization:</strong>
 * All operations respect authorization context derived from the current authentication token.
 * Users can only access instances within their authorized contexts and see metadata appropriate for their role.
 * 
 * <strong>Context Support:</strong>
 * The client automatically operates within the context determined by the authorization token.
 * Additional methods support cross-context queries for authorized users with hierarchical access.
 * 
 * @author Luca Frosini (ISTI - CNR)
 */
public interface ResourceRegistryClient extends RequestInfo {

	/**
	 * Adds a custom HTTP header to be included in requests.
	 * 
	 * @param name Header name
	 * @param value Header value
	 */
	public void addHeader(String name, String value);
	
	/**
	 * Adds a custom HTTP header to be included in requests.
	 * 
	 * @param name Header name
	 * @param value boolean value
	 */
	public void addHeader(String name, boolean value);

	
	/* ---------------------- Contexts APIs --------------------------------- */
	
	/**
	 * Gets the context cache used for context-related operations.
	 * 
	 * <strong>Cache Behavior:</strong>
	 * <ul>
	 * <li>Provides access to the internal context cache used by all context methods;</li>
	 * <li>Cache improves performance by avoiding repeated server requests;</li>
	 * <li>Automatically manages cache updates and invalidation;</li>
	 * <li>Used internally by {@code getAllContexts()}, {@code existContext()}, {@code getContext()}, and {@code getCurrentContext()}.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>Available to all authenticated users;</li>
	 * <li>Cache content is filtered based on user authorization level.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryClient client = ResourceRegistryClientFactory.create();
	 * 
	 * // Access context cache for advanced operations
	 * ContextCache cache = client.getContextCache();
	 * 
	 * // Force cache refresh if needed
	 * cache.refresh();
	 * </pre>
	 * 
	 * @return The context cache instance
	 */
	public ContextCache getContextCache();

	/**
	 * Retrieves all available contexts using internal cache.
	 * 
	 * <strong>Cache Behavior:</strong>
	 * <ul>
	 * <li><strong>Uses internal context cache</strong> - does NOT bypass cache like server-direct methods</li>
	 * <li>First call may trigger server request to populate cache</li>
	 * <li>Subsequent calls return cached data for improved performance</li>
	 * <li>Cache is automatically maintained and refreshed as needed.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters NOT Applicable:</strong>
	 * <ul>
	 * <li>Standard query parameters like {@code includeMeta}, {@code limit}, {@code offset}, {@code hierarchical} are NOT applicable to cached methods</li>
	 * <li>Cache behavior is controlled internally for optimal performance</li>
	 * <li>For direct server access with full query control, use the implementation's FromServer methods.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>Available to all authenticated users</li>
	 * <li>Returns contexts accessible based on user authorization level</li>
	 * <li>Context list is filtered according to user permissions.</li>
	 * </ul>
	 * 
	 * <strong>Context Content:</strong>
	 * <ul>
	 * <li>Includes all contexts where the user has any level of access</li>
	 * <li>Context details include basic information (name, UUID, parent relationships)</li>
	 * <li>Metadata inclusion depends on cache configuration and user role.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryClient client = ResourceRegistryClientFactory.create();
	 * 
	 * // Get all contexts from cache
	 * List&lt;Context&gt; contexts = client.getAllContexts();
	 * 
	 * // Iterate through available contexts
	 * for (Context context : contexts) {
	 *     System.out.println("Context: " + context.getName() + " (" + context.getUuid() + ")");
	 * }
	 * </pre>
	 * 
	 * @return List of all contexts accessible to the current user (from cache)
	 * @throws ResourceRegistryException If fails to retrieve contexts from cache or server
	 */
	public List<Context> getAllContexts() throws ResourceRegistryException;
	
	/**
	 * Checks if a context with the specified UUID (as string) exists in the system using internal cache.
	 * 
	 * <strong>Cache Behavior:</strong>
	 * <ul>
	 * <li><strong>Uses internal context cache</strong> - does NOT bypass cache like server-direct methods</li>
	 * <li>First call may trigger cache population from server if needed</li>
	 * <li>Subsequent calls use cached data for improved performance</li>
	 * <li>Cache is automatically maintained and refreshed as needed.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters NOT Applicable:</strong>
	 * <ul>
	 * <li>Standard query parameters like {@code includeMeta}, {@code hierarchical} are NOT applicable to cached methods</li>
	 * <li>Cache behavior is controlled internally for optimal performance</li>
	 * <li>For direct server existence checks, use the implementation's FromServer methods.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>Available to all authenticated users</li>
	 * <li>Returns true only if context exists AND is accessible based on user authorization level</li>
	 * <li>Contexts outside user's access scope will return false.</li>
	 * </ul>
	 * 
	 * <strong>Input Validation:</strong>
	 * <ul>
	 * <li>UUID string format is validated before cache lookup</li>
	 * <li>Invalid UUID format throws IllegalArgumentException</li>
	 * <li>Null or empty strings are handled gracefully.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryClient client = ResourceRegistryClientFactory.create();
	 * 
	 * // Check context existence using cache
	 * String contextUUID = "c0f314e7-2807-4241-a792-2a6c79ed4fd0";
	 * if (client.existContext(contextUUID)) {
	 *     System.out.println("Context exists and is accessible");
	 * } else {
	 *     System.out.println("Context not found or not accessible");
	 * }
	 * </pre>
	 * 
	 * @param uuid the UUID of the context (as string) to check for existence
	 * @return true if the context exists and is accessible to the current user, false if not found
	 * @throws ContextNotFoundException If there are communication errors with the Resource Registry service
	 * @throws ResourceRegistryException If an error occurs during the existence check
	 * @throws IllegalArgumentException If the provided string is not a valid UUID format
	 */
	public boolean existContext(String uuid) throws ContextNotFoundException, ResourceRegistryException;

	/**
	 * Checks if a context with the specified UUID exists in the system using internal cache.
	 * 
	 * <strong>Cache Behavior:</strong>
	 * <ul>
	 * <li><strong>Uses internal context cache</strong> - does NOT bypass cache like server-direct methods</li>
	 * <li>First call may trigger cache population from server if needed</li>
	 * <li>Subsequent calls use cached data for improved performance</li>
	 * <li>Cache is automatically maintained and refreshed as needed.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters NOT Applicable:</strong>
	 * <ul>
	 * <li>Standard query parameters like {@code includeMeta}, {@code hierarchical} are NOT applicable to cached methods</li>
	 * <li>Cache behavior is controlled internally for optimal performance</li>
	 * <li>For direct server existence checks, use the implementation's FromServer methods.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>Available to all authenticated users</li>
	 * <li>Returns true only if context exists AND is accessible based on user authorization level</li>
	 * <li>Contexts outside user's access scope will return false.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryClient client = ResourceRegistryClientFactory.create();
	 * 
	 * // Check context existence using cache
	 * UUID contextUUID = UUID.fromString("c0f314e7-2807-4241-a792-2a6c79ed4fd0");
	 * if (client.existContext(contextUUID)) {
	 *     System.out.println("Context exists and is accessible");
	 * } else {
	 *     System.out.println("Context not found or not accessible");
	 * }
	 * </pre>
	 * 
	 * @param uuid the UUID of the context to check for existence
	 * @return true if the context exists and is accessible to the current user, false if not found
	 * @throws ContextNotFoundException If the context is not found or not accessible
	 * @throws ResourceRegistryException If an error occurs during the existence check
	 */
	public boolean existContext(UUID uuid) throws ContextNotFoundException, ResourceRegistryException;

	/**
	 * Retrieves a specific context by its UUID (as string) using internal cache.
	 * 
	 * <strong>Cache Behavior:</strong>
	 * <ul>
	 * <li><strong>Uses internal context cache</strong> - does NOT bypass cache like server-direct methods</li>
	 * <li>First call may trigger cache population from server if needed</li>
	 * <li>Subsequent calls use cached data for improved performance</li>
	 * <li>Cache is automatically maintained and refreshed as needed.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters NOT Applicable:</strong>
	 * <ul>
	 * <li>Standard query parameters like {@code includeMeta}, {@code hierarchical} are NOT applicable to cached methods</li>
	 * <li>Metadata inclusion depends on cache configuration and user role</li>
	 * <li>For direct server access with full query control, use the implementation's FromServer methods.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>Available to all authenticated users</li>
	 * <li>Returns context only if accessible based on user authorization level</li>
	 * <li>Context details are filtered according to user permissions and role.</li>
	 * </ul>
	 * 
	 * <strong>Context Content:</strong>
	 * <ul>
	 * <li>Includes basic context information (name, UUID, parent relationships)</li>
	 * <li>Metadata inclusion depends on user role: IS-Manager/Infrastructure-Manager see complete metadata, others see filtered metadata</li>
	 * <li>Sensitive information (createdBy, lastUpdatedBy) may be obfuscated for non-administrative users.</li>
	 * </ul>
	 * 
	 * <strong>Input Validation:</strong>
	 * <ul>
	 * <li>UUID string format is validated before cache lookup</li>
	 * <li>Invalid UUID format throws IllegalArgumentException</li>
	 * <li>Null or empty strings are handled gracefully.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryClient client = ResourceRegistryClientFactory.create();
	 * 
	 * // Get context from cache
	 * String contextUUID = "c0f314e7-2807-4241-a792-2a6c79ed4fd0";
	 * Context context = client.getContext(contextUUID);
	 * 
	 * System.out.println("Context name: " + context.getName());
	 * System.out.println("Context UUID: " + context.getUuid());
	 * System.out.println("Parent context: " + context.getParent());
	 * </pre>
	 * 
	 * @param uuid the UUID of the context (as string) to retrieve
	 * @return the Context object with role-appropriate detail level (from cache)
	 * @throws ContextNotFoundException If the context with the specified UUID is not found
	 * @throws ResourceRegistryException If an error occurs during context retrieval or authorization issues
	 * @throws IllegalArgumentException If the provided string is not a valid UUID format
	 */
	public Context getContext(String uuid) throws ContextNotFoundException, ResourceRegistryException;

	/**
	 * Retrieves a specific context by its UUID using internal cache.
	 * 
	 * <strong>Cache Behavior:</strong>
	 * <ul>
	 * <li><strong>Uses internal context cache</strong> - does NOT bypass cache like server-direct methods</li>
	 * <li>First call may trigger cache population from server if needed</li>
	 * <li>Subsequent calls use cached data for improved performance</li>
	 * <li>Cache is automatically maintained and refreshed as needed.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters NOT Applicable:</strong>
	 * <ul>
	 * <li>Standard query parameters like {@code includeMeta}, {@code hierarchical} are NOT applicable to cached methods</li>
	 * <li>Metadata inclusion depends on cache configuration and user role</li>
	 * <li>For direct server access with full query control, use the implementation's FromServer methods.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>Available to all authenticated users</li>
	 * <li>Returns context only if accessible based on user authorization level</li>
	 * <li>Context details are filtered according to user permissions and role.</li>
	 * </ul>
	 * 
	 * <strong>Context Content:</strong>
	 * <ul>
	 * <li>Includes basic context information (name, UUID, parent relationships)</li>
	 * <li>Metadata inclusion depends on user role: IS-Manager/Infrastructure-Manager see complete metadata, others see filtered metadata</li>
	 * <li>Sensitive information (createdBy, lastUpdatedBy) may be obfuscated for non-administrative users.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryClient client = ResourceRegistryClientFactory.create();
	 * 
	 * // Get context from cache
	 * UUID contextUUID = UUID.fromString("c0f314e7-2807-4241-a792-2a6c79ed4fd0");
	 * Context context = client.getContext(contextUUID);
	 * 
	 * System.out.println("Context name: " + context.getName());
	 * System.out.println("Context UUID: " + context.getUuid());
	 * System.out.println("Parent context: " + context.getParent());
	 * </pre>
	 * 
	 * @param uuid the UUID of the context to retrieve
	 * @return the Context object with role-appropriate detail level (from cache)
	 * @throws ContextNotFoundException If the context with the specified UUID is not found
	 * @throws ResourceRegistryException If an error occurs during context retrieval or authorization issues
	 */
	public Context getContext(UUID uuid) throws ContextNotFoundException, ResourceRegistryException;

	/**
	 * Retrieves the current context associated with the client's authorization token using internal cache.
	 * 
	 * <strong>Cache Behavior:</strong>
	 * <ul>
	 * <li><strong>Uses internal context cache</strong> - does NOT bypass cache like server-direct methods</li>
	 * <li>Current context is determined from the authorization token/secret</li>
	 * <li>First call may trigger cache population from server if needed</li>
	 * <li>Subsequent calls use cached data for improved performance.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters NOT Applicable:</strong>
	 * <ul>
	 * <li>Standard query parameters like {@code includeMeta}, {@code hierarchical} are NOT applicable to cached methods</li>
	 * <li>Metadata inclusion depends on cache configuration and user role</li>
	 * <li>For direct server access with full query control, use the implementation's FromServer methods.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>Available to all authenticated users</li>
	 * <li>Returns the context associated with the current authorization token</li>
	 * <li>Context details are filtered according to user permissions and role.</li>
	 * </ul>
	 * 
	 * <strong>Context Resolution:</strong>
	 * <ul>
	 * <li>Current context is extracted from the authorization token/secret</li>
	 * <li>Token must contain valid context information</li>
	 * <li>Context must be accessible to the user associated with the token.</li>
	 * </ul>
	 * 
	 * <strong>Context Content:</strong>
	 * <ul>
	 * <li>Includes basic context information (name, UUID, parent relationships)</li>
	 * <li>Metadata inclusion depends on user role: IS-Manager/Infrastructure-Manager see complete metadata, others see filtered metadata</li>
	 * <li>Sensitive information (createdBy, lastUpdatedBy) may be obfuscated for non-administrative users.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryClient client = ResourceRegistryClientFactory.create();
	 * 
	 * // Get current context from cache
	 * Context currentContext = client.getCurrentContext();
	 * 
	 * System.out.println("Current context name: " + currentContext.getName());
	 * System.out.println("Current context UUID: " + currentContext.getUuid());
	 * System.out.println("Current context parent: " + currentContext.getParent());
	 * </pre>
	 * 
	 * @return the Context object associated with the current authorization token (from cache)
	 * @throws ContextNotFoundException If the current context cannot be found or resolved
	 * @throws ResourceRegistryException If an error occurs during context retrieval or authorization issues
	 */
	public Context getCurrentContext() throws ContextNotFoundException, ResourceRegistryException;


	/* -------------------------- Type APIs --------------------------------- */

	/**
	 * Retrieves the complete model knowledge containing type definitions and their information.
	 * The model knowledge provides access to all registered types and their hierarchical relationships.
	 * 
	 * <strong>Corresponding REST API:</strong> Multiple endpoints used internally to build the complete model knowledge
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Returns a cached version of the complete type system model</li>
	 * <li>Includes all type definitions, their properties, and hierarchical relationships</li>
	 * <li>Provides access to type information used for validation and schema operations</li>
	 * <li>Cache is automatically maintained and updated as needed.</li>
	 * </ul>
	 * 
	 * <strong>Model Knowledge Content:</strong>
	 * <ul>
	 * <li><strong>Type Definitions</strong>: Complete schema definitions for all registered types</li>
	 * <li><strong>Type Hierarchy</strong>: Parent-child relationships between types</li>
	 * <li><strong>Type Properties</strong>: Property definitions and constraints for each type</li>
	 * <li><strong>Type Metadata</strong>: Additional information about type usage and validation rules.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>Available to all authenticated users</li>
	 * <li>No special permissions required for accessing type definitions</li>
	 * <li>Type information is considered public metadata within the system.</li>
	 * </ul>
	 * 
	 * <strong>Caching Behavior:</strong>
	 * <ul>
	 * <li>Model knowledge is cached for performance</li>
	 * <li>Cache is automatically refreshed when schema changes are detected</li>
	 * <li>Use {@link #renewModelKnowledge()} to force cache refresh if needed.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryClient client = ResourceRegistryClientFactory.create();
	 * 
	 * // Get complete model knowledge
	 * ModelKnowledge&lt;Type, TypeInformation&gt; modelKnowledge = client.getModelKnowledge();
	 * 
	 * // Access type information
	 * TypeInformation contactFacetInfo = modelKnowledge.getTypeInformation("ContactFacet");
	 * Type eServiceType = modelKnowledge.getType("EService");
	 * 
	 * // Check type hierarchy
	 * Set&lt;String&gt; subTypes = modelKnowledge.getSubTypes("Resource", true);
	 * </pre>
	 * 
	 * @return the model knowledge containing type definitions and information
	 */
	public ModelKnowledge<Type, TypeInformation> getModelKnowledge();
	
	/**
	 * Forces a renewal of the cached model knowledge by fetching the latest version from the Resource Registry.
	 * This method should be called when the schema has been modified and the local cache needs to be updated.
	 * 
	 * <strong>Corresponding REST API:</strong> Multiple endpoints used internally to rebuild the complete model knowledge
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Clears the current cached model knowledge</li>
	 * <li>Fetches the latest type definitions from the Resource Registry</li>
	 * <li>Rebuilds the complete model knowledge structure</li>
	 * <li>Updates internal caches with the new information.</li>
	 * </ul>
	 * 
	 * <strong>When to Use:</strong>
	 * <ul>
	 * <li>After schema modifications that add, modify, or remove type definitions</li>
	 * <li>When working with dynamic schema changes in development environments</li>
	 * <li>If type validation errors suggest schema inconsistencies</li>
	 * <li>When explicit cache refresh is needed for testing purposes.</li>
	 * </ul>
	 * 
	 * <strong>Performance Considerations:</strong>
	 * <ul>
	 * <li>This operation involves multiple REST API calls to rebuild the model</li>
	 * <li>Should be used sparingly in production environments</li>
	 * <li>Consider the impact on application performance when calling this method</li>
	 * <li>Automatic cache refresh is usually sufficient for most use cases.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>Available to all authenticated users</li>
	 * <li>No special permissions required for refreshing type definitions</li>
	 * <li>Same authorization level as {@link #getModelKnowledge()}.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryClient client = ResourceRegistryClientFactory.create();
	 * 
	 * // Force refresh of model knowledge after schema changes
	 * client.renewModelKnowledge();
	 * 
	 * // Now get updated model knowledge
	 * ModelKnowledge&lt;Type, TypeInformation&gt; freshModelKnowledge = client.getModelKnowledge();
	 * </pre>
	 */
	public void renewModelKnowledge();
	
	/**
	 * Checks if a type definition exists in the Information System using the type name.
	 * 
	 * @param typeName the name of the type to check for existence (case-sensitive)
	 * @return true if the type exists in the system, false if not found
	 * @throws ResourceRegistryException if an error occurs during the existence check
	 */
	public boolean existType(String typeName) throws ResourceRegistryException;
	
	/**
	 * Checks if a type definition exists in the Information System using a model element class.
	 * 
	 * @param <ME> the model element type parameter
	 * @param clazz the class of the model element type to check for existence
	 * @return true if the type corresponding to the class exists in the system, false if not found
	 * @throws ResourceRegistryException if an error occurs during the existence check
	 */
	public <ME extends ModelElement> boolean existType(Class<ME> clazz) throws ResourceRegistryException;

	/**
	 * Reads the schema definition for a specified type by name.
	 * 
	 * @param typeName the name of the type to retrieve (case-sensitive)
	 * @param polymorphic whether to include subtypes in the response (when true, includes all subtypes)
	 * @return JSON string containing type schema definition(s) with authorization-appropriate detail level
	 * @throws SchemaNotFoundException if the specified type is not found in the system
	 * @throws ResourceRegistryException for retrieval errors or other system failures
	 */
	public String getType(String typeName, Boolean polymorphic)
			throws SchemaNotFoundException, ResourceRegistryException;

	/**
	 * Reads the schema definition for a specified type by name up to a specific hierarchy level.
	 * 
	 * @param typeName the name of the type to retrieve (case-sensitive)
	 * @param level the maximum hierarchy level to include in the response
	 * @return JSON string containing type schema definition with hierarchy limited to the specified level
	 * @throws SchemaNotFoundException if the specified type is not found in the system
	 * @throws ResourceRegistryException for retrieval errors or other system failures
	 */
	public String getType(String typeName, int level)
			throws SchemaNotFoundException, ResourceRegistryException;
	
	/**
	 * Reads the type definitions for a specified model element class.
	 * 
	 * @param <ME> the model element type parameter
	 * @param clazz the class of the model element type to retrieve
	 * @param polymorphic whether to include subtypes in the response (when true, includes all subtypes)
	 * @return list of Type objects representing the type definition(s) with authorization-appropriate detail level
	 * @throws SchemaNotFoundException if the type corresponding to the specified class is not found
	 * @throws ResourceRegistryException for retrieval errors or other system failures
	 */
	public <ME extends ModelElement> List<Type> getType(Class<ME> clazz, Boolean polymorphic)
			throws SchemaNotFoundException, ResourceRegistryException;
	
	/**
	 * Reads the type definitions for a specified model element class up to a specific hierarchy level.
	 * 
	 * @param <ME> the model element type parameter
	 * @param clazz the class of the model element type to retrieve
	 * @param level the maximum hierarchy level to include in the response
	 * @return list of Type objects representing the type definition(s) with hierarchy limited to the specified level
	 * @throws SchemaNotFoundException if the type corresponding to the specified class is not found
	 * @throws ResourceRegistryException for retrieval errors or other system failures
	 */
	public <ME extends ModelElement> List<Type> getType(Class<ME> clazz, int level)
			throws SchemaNotFoundException, ResourceRegistryException;
	
	/**
	 * Retrieves the type tree node for a specific type identified by its model element class.
	 * The tree node contains the type definition and its hierarchical position within the type system.
	 * 
	 * @param <ME> the model element type parameter
	 * @param clazz the class of the model element type to retrieve
	 * @return the tree node containing the type and its hierarchical information
	 * @throws SchemaNotFoundException if the type corresponding to the specified class does not exist in the schema
	 * @throws ResourceRegistryException if an error occurs during the operation
	 */
	public <ME extends ModelElement> Node<Type> getTypeTreeNode(Class<ME> clazz)
			throws SchemaNotFoundException, ResourceRegistryException;
	
	/**
	 * Retrieves the type tree node for a specific type identified by its name.
	 * The tree node contains the type definition and its hierarchical position within the type system.
	 * 
	 * @param typeName the name of the type to retrieve
	 * @return the tree node containing the type and its hierarchical information
	 * @throws SchemaNotFoundException if the specified type name does not exist in the schema
	 * @throws ResourceRegistryException if an error occurs during the operation
	 */
	public Node<Type> getTypeTreeNode(String typeName)
			throws SchemaNotFoundException, ResourceRegistryException;
	
	
	/* --------------------- Instances APIs --------------------------------- */

	/**
	 * Retrieves all instances of the specified type from the Resource Registry.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code GET /access/instances/{type-name}[?polymorphic={true|false}&limit={number}&offset={number}&includeMeta={true|false}&allMeta={true|false}&includeContexts={true|false}&hierarchical={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Returns all instances of the specified type that are accessible in the current context</li>
	 * <li>Supports polymorphic queries to include instances of subtypes</li>
	 * <li>Results are filtered based on user authorization and context access</li>
	 * <li>Returns strongly-typed objects instead of JSON strings.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>limit</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Maximum number of instances to return in a single response</li>
	 * <li>Default value: server configuration dependent</li>
	 * <li>Usage: {@code client.setLimit(50)}</li>
	 * <li>Example: Set to 50 to get at most 50 instances per request</li>
	 * <li>Unlimited results: Set to {@code RequestInfo.UNBOUNDED_LIMIT} (-1) for unlimited results (may cause timeout if results are too many)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#LIMIT_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>offset</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Number of instances to skip from the beginning of the result set</li>
	 * <li>Default value: 0</li>
	 * <li>Usage: {@code client.setOffset(10)}</li>
	 * <li>Example: Set to 10 to skip the first 10 instances (useful for pagination)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#OFFSET_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response instance</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code client.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata for all nested instances (ConsistsOf relations, Facets, etc.) in the response</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main instance, more human-readable)</li>
	 * <li>Usage: {@code client.setAllMeta(true)}</li>
	 * <li>Values: true (complete metadata) | false (main instance only, more readable)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include the list of contexts where the instance and its nested elements are available in the response</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Usage: {@code client.setIncludeContexts(true)}</li>
	 * <li>Values: true (shows context availability) | false (no context information)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_CONTEXTS_QUERY_PARAMETER}</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present</li>
	 * <li><strong>Note:</strong> A Facet's context availability depends on the Resources that include it via ConsistsOf relations.</li>
	 * </ul>
	 * 
	 * <strong>hierarchical</strong> (configurable via client configuration, admin-only):
	 * <ul>
	 * <li>Whether to include instances from child contexts of the current context</li>
	 * <li>Default value: false (child contexts not included)</li>
	 * <li>Usage: {@code client.setHierarchicalMode(true)}</li>
	 * <li>Values: true (includes child contexts) | false (current context only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#HIERARCHICAL_MODE_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> Only available to IS-Manager, Infrastructure-Manager, and Context-Manager roles.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The type name is automatically derived from the provided class using {@link org.gcube.informationsystem.utils.TypeUtility#getTypeName(Class)}</li>
	 * <li>Supports all Information System model types and their subtypes: Entities (Resources and all Resource subtypes, Facets and all Facet subtypes) and Relations (ConsistsOf and all ConsistsOf subtypes, IsRelatedTo and all IsRelatedTo subtypes).</li>
	 * </ul>
	 *
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>The instances must be accessible in the current context or child contexts (depending on hierarchical mode)</li>
	 * <li>IS-Manager and Infrastructure-Manager are global roles (available in all contexts) and can use hierarchical mode in any contexts</li>
	 * <li>Context-Manager is a context-specific role and can use hierarchical mode only in contexts where they have this role.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>All users must have a valid token and read permissions for the specific context where they want to list instances.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryClient client = ResourceRegistryClientFactory.create();
	 * 
	 * // Get all EService instances including subtypes
	 * List&lt;EService&gt; services = client.getInstances(EService.class, true);
	 * 
	 * // Get only exact ContactFacet instances (no subtypes)
	 * List&lt;ContactFacet&gt; contacts = client.getInstances(ContactFacet.class, false);
	 * </pre>
	 * 
	 * @param <ERElem> The specific type of Information System element to retrieve
	 * @param clazz The class representing the type of instances to retrieve
	 * @param polymorphic Whether to include instances of subtypes (true) or only exact type instances (false). 
	 *                    When true, includes all subtypes of the specified type; when false, returns only instances 
	 *                    of the exact specified type. This parameter is automatically set in the {@code polymorphic}
	 *                    query parameter of the REST API call.
	 * @return List of instances of the specified type accessible in the current context
	 * @throws ResourceRegistryException If an error occurs during instance retrieval or if the type is invalid
	 */
	public <ERElem extends ERElement> List<ERElem> getInstances(Class<ERElem> clazz, Boolean polymorphic)
			throws ResourceRegistryException;

	/**
	 * Retrieves all instances of the specified type from the Resource Registry as JSON string.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code GET /access/instances/{type-name}[?polymorphic={true|false}&limit={number}&offset={number}&includeMeta={true|false}&allMeta={true|false}&includeContexts={true|false}&hierarchical={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Returns all instances of the specified type that are accessible in the current context</li>
	 * <li>Supports polymorphic queries to include instances of subtypes</li>
	 * <li>Results are filtered based on user authorization and context access</li>
	 * <li>Returns the response as a JSON array string.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>limit</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Maximum number of instances to return in a single response</li>
	 * <li>Default value: server configuration dependent</li>
	 * <li>Usage: {@code client.setLimit(50)}</li>
	 * <li>Example: Set to 50 to get at most 50 instances per request</li>
	 * <li>Unlimited results: Set to {@code RequestInfo.UNBOUNDED_LIMIT} (-1) for unlimited results (may cause timeout if results are too many)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#LIMIT_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>offset</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Number of instances to skip from the beginning of the result set</li>
	 * <li>Default value: 0</li>
	 * <li>Usage: {@code client.setOffset(10)}</li>
	 * <li>Example: Set to 10 to skip the first 10 instances (useful for pagination)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#OFFSET_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response instance</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code client.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata for all nested instances (ConsistsOf relations, Facets, etc.) in the response</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main instance, more human-readable)</li>
	 * <li>Usage: {@code client.setAllMeta(true)}</li>
	 * <li>Values: true (complete metadata) | false (main instance only, more readable)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include the list of contexts where the instance and its nested elements are available in the response</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Usage: {@code client.setIncludeContexts(true)}</li>
	 * <li>Values: true (shows context availability) | false (no context information)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_CONTEXTS_QUERY_PARAMETER}</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present</li>
	 * <li><strong>Note:</strong> A Facet's context availability depends on the Resources that include it via ConsistsOf relations.</li>
	 * </ul>
	 * 
	 * <strong>hierarchical</strong> (configurable via client configuration, admin-only):
	 * <ul>
	 * <li>Whether to include instances from child contexts of the current context</li>
	 * <li>Default value: false (child contexts not included)</li>
	 * <li>Usage: {@code client.setHierarchicalMode(true)}</li>
	 * <li>Values: true (includes child contexts) | false (current context only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#HIERARCHICAL_MODE_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> Only available to IS-Manager, Infrastructure-Manager, and Context-Manager roles.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>Supports all Information System model types and their subtypes: Entities (Resources and all Resource subtypes, Facets and all Facet subtypes) and Relations (ConsistsOf and all ConsistsOf subtypes, IsRelatedTo and all IsRelatedTo subtypes)</li>
	 * <li>Type names are case-sensitive and must match exactly the class names defined in the Information System model.</li>
	 * </ul>
	 *
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>The instances must be accessible in the current context or child contexts (depending on hierarchical mode)</li>
	 * <li>IS-Manager and Infrastructure-Manager are global roles (available in all contexts) and can use hierarchical mode in any contexts</li>
	 * <li>Context-Manager is a context-specific role and can use hierarchical mode only in contexts where they have this role.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>All users must have a valid token and read permissions for the specific context where they want to list instances.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryClient client = ResourceRegistryClientFactory.create();
	 * 
	 * // Get all EService instances including subtypes as JSON
	 * String servicesJson = client.getInstances("EService", true);
	 * 
	 * // Get only exact ContactFacet instances (no subtypes) as JSON
	 * String contactsJson = client.getInstances("ContactFacet", false);
	 * </pre>
	 * 
	 * @param type The name of the Information System type to retrieve (case-sensitive)
	 * @param polymorphic Whether to include instances of subtypes (true) or only exact type instances (false).
	 *                    When true, includes all subtypes of the specified type; when false, returns only instances 
	 *                    of the exact specified type. This parameter is automatically set in the {@code polymorphic} 
	 *                    query parameter of the REST API call.
	 * @return JSON array string containing instances of the specified type accessible in the current context
	 * @throws ResourceRegistryException If an error occurs during instance retrieval or if the type is invalid
	 */
	public String getInstances(String type, Boolean polymorphic) throws ResourceRegistryException;

	/**
	 * Checks if an Information System element instance exists in the Resource Registry using class and UUID.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code HEAD /access/instances/{type-name}/{uuid}[?hierarchical={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Verifies the existence of the specified instance using its UUID and class type</li>
	 * <li>Type name is automatically derived from the class using reflection</li>
	 * <li>Does not return the instance data, only confirms existence</li>
	 * <li>Checks accessibility within the current context and user authorization</li>
	 * <li>Convenient method when working with strongly-typed classes.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Instance exists and is accessible in the current context</li>
	 * <li><strong>404 Not Found</strong>: Instance does not exist or is not accessible</li>
	 * <li><strong>403 Forbidden</strong>: Instance exists but user lacks authorization to access it</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>hierarchical</strong> (configurable via client configuration, admin-only):
	 * <ul>
	 * <li>Whether to include instances from child contexts of the current context</li>
	 * <li>Default value: false (child contexts not included)</li>
	 * <li>Usage: {@code client.setHierarchicalMode(true)}</li>
	 * <li>Values: true (includes child contexts) | false (current context only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#HIERARCHICAL_MODE_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> Only available to IS-Manager, Infrastructure-Manager, and Context-Manager roles.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The type name is automatically derived from the provided class using {@link org.gcube.informationsystem.utils.TypeUtility#getTypeName(Class)}</li>
	 * <li>Supports all Information System model types and their subtypes: Entities (Resources and all Resource subtypes, Facets and all Facet subtypes) and Relations (ConsistsOf and all ConsistsOf subtypes, IsRelatedTo and all IsRelatedTo subtypes).</li>
	 * </ul>
	 * 
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>The instance must be accessible in the current context or child contexts (depending on hierarchical mode)</li>
	 * <li>IS-Manager and Infrastructure-Manager are global roles (available in all contexts) and can use hierarchical mode in any contexts</li>
	 * <li>Context-Manager is a context-specific role and can use hierarchical mode only in contexts where they have this role.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>All users must have a valid token and read permissions for the specific context where they want to check existence.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryClient client = ResourceRegistryClientFactory.create();
	 * 
	 * // Check if an EService instance exists by class and UUID
	 * UUID serviceUUID = UUID.fromString("...");
	 * boolean exists = client.existInstance(EService.class, serviceUUID);
	 * 
	 * // Check if a ContactFacet instance exists by class and UUID
	 * UUID contactUUID = UUID.fromString("...");
	 * boolean contactExists = client.existInstance(ContactFacet.class, contactUUID);
	 * </pre>
	 * 
	 * @param <ERElem> The specific type of Information System element to check
	 * @param clazz The class representing the type of instance to check
	 * @param uuid The UUID of the instance to check for existence
	 * @return true if the instance exists and is accessible in the current context, false if not found
	 * @throws AvailableInAnotherContextException If the instance exists but is not accessible in the current context
	 * @throws ResourceRegistryException If an error occurs during the existence check
	 */
	public <ERElem extends ERElement> boolean existInstance(Class<ERElem> clazz, UUID uuid)
			throws AvailableInAnotherContextException, ResourceRegistryException;

	/**
	 * Checks if an Information System element instance exists in the Resource Registry using type name and UUID.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code HEAD /access/instances/{type-name}/{uuid}[?hierarchical={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Verifies the existence of the specified instance using its UUID and type name</li>
	 * <li>Does not return the instance data, only confirms existence</li>
	 * <li>Checks accessibility within the current context and user authorization</li>
	 * <li>Most direct method when type name is already known.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Instance exists and is accessible in the current context</li>
	 * <li><strong>404 Not Found</strong>: Instance does not exist or is not accessible</li>
	 * <li><strong>403 Forbidden</strong>: Instance exists but user lacks authorization to access it</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>hierarchical</strong> (configurable via client configuration, admin-only):
	 * <ul>
	 * <li>Whether to include instances from child contexts of the current context</li>
	 * <li>Default value: false (child contexts not included)</li>
	 * <li>Usage: {@code client.setHierarchicalMode(true)}</li>
	 * <li>Values: true (includes child contexts) | false (current context only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#HIERARCHICAL_MODE_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> Only available to IS-Manager, Infrastructure-Manager, and Context-Manager roles.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>Type name must be a valid Information System model type</li>
	 * <li>Supports all Information System model types and their subtypes: Entities (Resources and all Resource subtypes, Facets and all Facet subtypes) and Relations (ConsistsOf and all ConsistsOf subtypes, IsRelatedTo and all IsRelatedTo subtypes)</li>
	 * <li>Type name is case-sensitive and must match exactly.</li>
	 * </ul>
	 * 
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>The instance must be accessible in the current context or child contexts (depending on hierarchical mode)</li>
	 * <li>IS-Manager and Infrastructure-Manager are global roles (available in all contexts) and can use hierarchical mode in any contexts</li>
	 * <li>Context-Manager is a context-specific role and can use hierarchical mode only in contexts where they have this role.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>All users must have a valid token and read permissions for the specific context where they want to check existence.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryClient client = ResourceRegistryClientFactory.create();
	 * 
	 * // Check if an EService instance exists by type name and UUID
	 * UUID serviceUUID = UUID.fromString("...");
	 * boolean exists = client.existInstance("EService", serviceUUID);
	 * 
	 * // Check if a ContactFacet instance exists by type name and UUID
	 * UUID contactUUID = UUID.fromString("...");
	 * boolean contactExists = client.existInstance("ContactFacet", contactUUID);
	 * </pre>
	 * 
	 * @param type The name of the Information System type to check
	 * @param uuid The UUID of the instance to check for existence
	 * @return true if the instance exists and is accessible in the current context, false if not found
	 * @throws AvailableInAnotherContextException If the instance exists but is not accessible in the current context
	 * @throws ResourceRegistryException If an error occurs during the existence check
	 */
	public boolean existInstance(String type, UUID uuid)
			throws AvailableInAnotherContextException, ResourceRegistryException;

	/**
	 * Reads an existing instance from the Resource Registry by its type class and UUID.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code GET /access/instances/{type-name}/{uuid}[?includeMeta={true|false}&allMeta={true|false}&includeContexts={true|false}&hierarchical={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Retrieves an existing instance (Entity or Relation) by its type class and unique identifier</li>
	 * <li>Type name is automatically derived from the class using reflection</li>
	 * <li>The instance must be accessible in the current context or in child contexts (based on hierarchical mode)</li>
	 * <li>Returns the complete instance with all properties and metadata</li>
	 * <li>Returns strongly-typed objects instead of JSON strings.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Instance successfully retrieved</li>
	 * <li><strong>404 Not Found</strong>: No instance found with the specified UUID and type, or not accessible in current context</li>
	 * <li><strong>403 Forbidden</strong>: User lacks read permissions for the instance or its context</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response instance</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code client.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata for all nested instances (ConsistsOf relations, Facets, etc.) in the response</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main instance, more human-readable)</li>
	 * <li>Usage: {@code client.setAllMeta(true)}</li>
	 * <li>Values: true (complete metadata) | false (main instance only, more readable)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include the list of contexts where the instance and its nested elements are available in the response</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Usage: {@code client.setIncludeContexts(true)}</li>
	 * <li>Values: true (shows context availability) | false (no context information)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_CONTEXTS_QUERY_PARAMETER}</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present</li>
	 * <li><strong>Note:</strong> A Facet's context availability depends on the Resources that include it via ConsistsOf relations.</li>
	 * </ul>
	 * 
	 * <strong>hierarchical</strong> (configurable via client configuration, admin-only):
	 * <ul>
	 * <li>Whether to include instances from child contexts of the current context</li>
	 * <li>Default value: false (child contexts not included)</li>
	 * <li>Usage: {@code client.setHierarchicalMode(true)}</li>
	 * <li>Values: true (includes child contexts) | false (current context only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#HIERARCHICAL_MODE_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> Only available to IS-Manager, Infrastructure-Manager, and Context-Manager roles.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The type name is automatically derived from the provided class using {@link org.gcube.informationsystem.utils.TypeUtility#getTypeName(Class)}</li>
	 * <li>Supports all Information System model types and their subtypes: Entities (Resources and all Resource subtypes, Facets and all Facet subtypes) and Relations (ConsistsOf and all ConsistsOf subtypes, IsRelatedTo and all IsRelatedTo subtypes).</li>
	 * </ul>
	 * 
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>The instance must be accessible in the current context or child contexts (depending on hierarchical mode)</li>
	 * <li>IS-Manager and Infrastructure-Manager are global roles (available in all contexts) and can use hierarchical mode in any contexts</li>
	 * <li>Context-Manager is a context-specific role and can use hierarchical mode only in contexts where they have this role.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>All users must have a valid token and read permissions for the specific context where they want to read instances.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryClient client = ResourceRegistryClientFactory.create();
	 * 
	 * // Read an EService instance by class and UUID
	 * UUID serviceUuid = UUID.fromString("...");
	 * EService service = client.getInstance(EService.class, serviceUuid);
	 * 
	 * // Read a ContactFacet instance by class and UUID
	 * UUID contactUuid = UUID.fromString("...");
	 * ContactFacet contact = client.getInstance(ContactFacet.class, contactUuid);
	 * </pre>
	 * 
	 * @param <ERElem> The specific type of Entity or Relation to read
	 * @param clazz The class of the Entity or Relation type to read
	 * @param uuid The unique identifier of the instance to read
	 * @return The complete instance from the Resource Registry
	 * @throws NotFoundException If no instance is found with the specified UUID and type
	 * @throws AvailableInAnotherContextException If the instance exists but is not accessible in the current context
	 * @throws ResourceRegistryException If an error occurs during instance retrieval
	 */
	public <ERElem extends ERElement> ERElem getInstance(Class<ERElem> clazz, UUID uuid)
			throws NotFoundException, AvailableInAnotherContextException, ResourceRegistryException;

	/**
	 * Reads an existing instance from the Resource Registry by its type name and UUID.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code GET /access/instances/{type-name}/{uuid}[?includeMeta={true|false}&allMeta={true|false}&includeContexts={true|false}&hierarchical={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Retrieves an existing instance (Entity or Relation) by its type name and unique identifier</li>
	 * <li>The instance must be accessible in the current context or in child contexts (based on hierarchical mode)</li>
	 * <li>Returns the complete instance as JSON with all properties and metadata</li>
	 * <li>Type validation ensures the UUID corresponds to an instance of the specified type.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Instance successfully retrieved</li>
	 * <li><strong>404 Not Found</strong>: No instance found with the specified UUID and type, or not accessible in current context</li>
	 * <li><strong>403 Forbidden</strong>: User lacks read permissions for the instance or its context</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response instance</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code client.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata for all nested instances (ConsistsOf relations, Facets, etc.) in the response</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main instance, more human-readable)</li>
	 * <li>Usage: {@code client.setAllMeta(true)}</li>
	 * <li>Values: true (complete metadata) | false (main instance only, more readable)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include the list of contexts where the instance and its nested elements are available in the response</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Usage: {@code client.setIncludeContexts(true)}</li>
	 * <li>Values: true (shows context availability) | false (no context information)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#INCLUDE_CONTEXTS_QUERY_PARAMETER}</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present</li>
	 * <li><strong>Note:</strong> A Facet's context availability depends on the Resources that include it via ConsistsOf relations.</li>
	 * </ul>
	 * 
	 * <strong>hierarchical</strong> (configurable via client configuration, admin-only):
	 * <ul>
	 * <li>Whether to include instances from child contexts of the current context</li>
	 * <li>Default value: false (child contexts not included)</li>
	 * <li>Usage: {@code client.setHierarchicalMode(true)}</li>
	 * <li>Values: true (includes child contexts) | false (current context only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.InstancePath#HIERARCHICAL_MODE_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> Only available to IS-Manager, Infrastructure-Manager, and Context-Manager roles.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>Type name must be a valid Information System model type</li>
	 * <li>Supports all Information System model types and their subtypes: Entities (Resources and all Resource subtypes, Facets and all Facet subtypes) and Relations (ConsistsOf and all ConsistsOf subtypes, IsRelatedTo and all IsRelatedTo subtypes)</li>
	 * <li>Type names are case-sensitive and must match exactly the class names defined in the Information System model.</li>
	 * </ul>
	 * 
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>The instance must be accessible in the current context or child contexts (depending on hierarchical mode)</li>
	 * <li>IS-Manager and Infrastructure-Manager are global roles (available in all contexts) and can use hierarchical mode in any contexts</li>
	 * <li>Context-Manager is a context-specific role and can use hierarchical mode only in contexts where they have this role.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li>All users must have a valid token and read permissions for the specific context where they want to read instances.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryClient client = ResourceRegistryClientFactory.create();
	 * 
	 * // Read an EService instance by type name and UUID as JSON
	 * UUID serviceUuid = UUID.fromString("...");
	 * String serviceJson = client.getInstance("EService", serviceUuid);
	 * 
	 * // Read a ContactFacet instance by type name and UUID as JSON
	 * UUID contactUuid = UUID.fromString("...");
	 * String contactJson = client.getInstance("ContactFacet", contactUuid);
	 * </pre>
	 * 
	 * @param type The name of the Information System type to read (case-sensitive)
	 * @param uuid The unique identifier of the instance to read
	 * @return JSON string containing the complete instance from the Resource Registry
	 * @throws NotFoundException If no instance is found with the specified UUID and type
	 * @throws AvailableInAnotherContextException If the instance exists but is not accessible in the current context
	 * @throws ResourceRegistryException If an error occurs during instance retrieval
	 */
	public String getInstance(String type, UUID uuid)
			throws NotFoundException, AvailableInAnotherContextException, ResourceRegistryException;

	/* ---------------------------------------------------------------------- */

	/**
	 * Retrieves the list of contexts where a specific Information System element is present in the Resource Registry.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code GET /access/instances/{type-name}/{instance-uuid}/contexts}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Retrieves all contexts where the specified element (Resource, Facet, ConsistsOf, or IsRelatedTo) is present</li>
	 * <li>Type name is automatically derived from the class using reflection</li>
	 * <li>The element must exist and be accessible to the user</li>
	 * <li>Returns a map where keys are context UUIDs and values are the full context paths</li>
	 * <li>No filtering, pagination, or metadata options are available</li>
	 * <li>Context information is determined by the element's presence across different VREs and organizational contexts.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Context list successfully retrieved</li>
	 * <li><strong>404 Not Found</strong>: Element does not exist with the specified type and UUID</li>
	 * <li><strong>403 Forbidden</strong>: Element exists but is not accessible for context listing in the current authorization scope</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager:</strong> Can retrieve context information for any element across the entire infrastructure</li>
	 * <li><strong>Infrastructure-Manager:</strong> Can retrieve context information for any element across the entire infrastructure</li>
	 * <li><strong>Context-Manager:</strong> Can retrieve context information for elements accessible within their managed contexts</li>
	 * <li><strong>Other Users:</strong> Can retrieve context information for elements accessible through their authorization token.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Returns a Map where keys are context UUIDs (UUID) and values are full context paths (String)</li>
	 * <li>Empty map is returned if the element exists but is not present in any contexts</li>
	 * <li>Each context path represents the complete hierarchical path (e.g., "/gcube/devsec/devVRE").</li>
	 * </ul>
	 * 
	 * <strong>Element Context Rules:</strong>
	 * <ul>
	 * <li><strong>Resources:</strong> Present in all contexts that form the union of contexts of all their Facets</li>
	 * <li><strong>Facets:</strong> Present in the contexts where they are explicitly available</li>
	 * <li><strong>ConsistsOf Relations:</strong> Present in all contexts where their target Facets are present</li>
	 * <li><strong>IsRelatedTo Relations:</strong> Present based on their specific context assignments.</li>
	 * </ul>
	 * 
	 * <strong>Type Safety:</strong>
	 * <ul>
	 * <li>This method provides compile-time type safety by accepting any ERElement subtype</li>
	 * <li>The element's type is automatically determined from the object's class</li>
	 * <li>Supports all Information System element types including Resources, Facets, and Relations.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryClient client = ResourceRegistryClientFactory.create();
	 * 
	 * // Get contexts for an EService by class and UUID
	 * UUID serviceUuid = UUID.fromString("48af15ad-7e56-4157-b624-71c98cea4f8f");
	 * Map&lt;UUID, String&gt; contexts = client.getInstanceContexts(EService.class, serviceUuid);
	 * 
	 * // Process the results
	 * for (Map.Entry&lt;UUID, String&gt; entry : contexts.entrySet()) {
	 *     System.out.println("Context UUID: " + entry.getKey() + " -> Path: " + entry.getValue());
	 * }
	 * </pre>
	 * 
	 * @param <ERElem> The type of Information System element (Resource, Facet, ConsistsOf, or IsRelatedTo)
	 * @param clazz The class of the Information System element whose contexts to retrieve
	 * @param uuid The unique identifier of the element whose contexts to retrieve
	 * @return Map containing context UUID to context path mappings where the element is present
	 * @throws NotFoundException If no element is found with the specified class and UUID
	 * @throws AvailableInAnotherContextException If the element exists but is not accessible for context listing
	 * @throws ResourceRegistryException If an error occurs during context retrieval
	 */
	public <ERElem extends ERElement> Map<UUID, String> getInstanceContexts(Class<ERElem> clazz, UUID uuid)
			throws NotFoundException, AvailableInAnotherContextException, ResourceRegistryException;

	/**
	 * Retrieves the list of contexts where a specific Information System element is present in the Resource Registry.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code GET /access/instances/{type-name}/{instance-uuid}/contexts}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Retrieves all contexts where the specified element (Resource, Facet, ConsistsOf, or IsRelatedTo) is present</li>
	 * <li>The element must exist and be accessible to the user</li>
	 * <li>Returns a map where keys are context UUIDs and values are the full context paths</li>
	 * <li>No filtering, pagination, or metadata options are available</li>
	 * <li>Context information is determined by the element's presence across different VREs and organizational contexts.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Context list successfully retrieved</li>
	 * <li><strong>404 Not Found</strong>: Element does not exist with the specified type and UUID</li>
	 * <li><strong>403 Forbidden</strong>: Element exists but is not accessible for context listing in the current authorization scope</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager:</strong> Can retrieve context information for any element across the entire infrastructure</li>
	 * <li><strong>Infrastructure-Manager:</strong> Can retrieve context information for any element across the entire infrastructure</li>
	 * <li><strong>Context-Manager:</strong> Can retrieve context information for elements accessible within their managed contexts</li>
	 * <li><strong>Other Users:</strong> Can retrieve context information for elements accessible through their authorization token.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * <ul>
	 * <li>Returns a Map where keys are context UUIDs (UUID) and values are full context paths (String)</li>
	 * <li>Empty map is returned if the element exists but is not present in any contexts</li>
	 * <li>Each context path represents the complete hierarchical path (e.g., "/gcube/devsec/devVRE").</li>
	 * </ul>
	 * 
	 * <strong>Element Context Rules:</strong>
	 * <ul>
	 * <li><strong>Resources:</strong> Present in all contexts that form the union of contexts of all their Facets</li>
	 * <li><strong>Facets:</strong> Present in the contexts where they are explicitly available</li>
	 * <li><strong>ConsistsOf Relations:</strong> Present in all contexts where their target Facets are present</li>
	 * <li><strong>IsRelatedTo Relations:</strong> Present based on their specific context assignments.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryClient client = ResourceRegistryClientFactory.create();
	 * 
	 * // Get contexts for an EService by type and UUID
	 * UUID serviceUuid = UUID.fromString("48af15ad-7e56-4157-b624-71c98cea4f8f");
	 * Map&lt;UUID, String&gt; contexts = client.getInstanceContexts("EService", serviceUuid);
	 * 
	 * // Process the results
	 * for (Map.Entry&lt;UUID, String&gt; entry : contexts.entrySet()) {
	 *     System.out.println("Context UUID: " + entry.getKey() + " -> Path: " + entry.getValue());
	 * }
	 * </pre>
	 * 
	 * @param type The exact type name of the Information System element (e.g., "EService", "ContactFacet")
	 * @param uuid The unique identifier of the element whose contexts to retrieve
	 * @return Map containing context UUID to context path mappings where the element is present
	 * @throws NotFoundException If no element is found with the specified type and UUID
	 * @throws AvailableInAnotherContextException If the element exists but is not accessible for context listing
	 * @throws ResourceRegistryException If an error occurs during context retrieval
	 */
	public Map<UUID, String> getInstanceContexts(String type, UUID uuid)
			throws NotFoundException, AvailableInAnotherContextException, ResourceRegistryException;

	
	/* --------------------- Query Templates APIs --------------------------- */

	/**
	 * Retrieves all query templates from the Resource Registry.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code GET /access/query-templates[?limit={number}&offset={number}&includeMeta={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Returns all query templates accessible to the current user</li>
	 * <li>Results are filtered based on user authorization</li>
	 * <li>Provides pagination support via limit and offset parameters</li>
	 * <li>Can include metadata with role-based filtering when requested.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>limit</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Maximum number of query templates to return in a single response</li>
	 * <li>Default value: 10</li>
	 * <li>Usage: {@code client.setLimit(50)}</li>
	 * <li>Example: Set to 50 to get at most 50 query templates per request</li>
	 * <li>Unlimited results: Set to {@code RequestInfo.UNBOUNDED_LIMIT} (-1) for unlimited results (may cause timeout if results are too many)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.QueryTemplatePath#LIMIT_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>offset</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Number of query templates to skip from the beginning of the result set</li>
	 * <li>Default value: 0</li>
	 * <li>Usage: {@code client.setOffset(100)}</li>
	 * <li>Example: Set to 100 to skip the first 100 query templates (useful for pagination)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.QueryTemplatePath#OFFSET_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response query templates</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code client.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.QueryTemplatePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager and Infrastructure-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Query templates successfully retrieved</li>
	 * <li><strong>400 Bad Request</strong>: Invalid query parameters (e.g., invalid limit/offset values)</li>
	 * <li><strong>403 Forbidden</strong>: Insufficient permissions to access query templates</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>No input parameters required for this operation</li>
	 * <li>All available query templates are retrieved based on user authorization.</li>
	 * </ul>
	 *
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>Query templates are global resources accessible across contexts</li>
	 * <li>Access level depends on user role and authorization.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Full access to all query templates with complete, unfiltered metadata including all administrative fields</li>
	 * <li><strong>Infrastructure-Manager</strong>: Full access to all query templates with complete, unfiltered metadata including all administrative fields</li>
	 * <li><strong>All Other Users</strong>: Basic access with metadata filtering; receive metadata with sensitive information filtered when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryClient client = ResourceRegistryClientFactory.create();
	 * 
	 * // Get all query templates with default pagination
	 * List&lt;QueryTemplate&gt; templates = client.getAllQueryTemplates();
	 * 
	 * // Get query templates with custom pagination and metadata
	 * client.setLimit(20);
	 * client.setOffset(40);
	 * client.setIncludeMeta(true);
	 * List&lt;QueryTemplate&gt; templatesWithMeta = client.getAllQueryTemplates();
	 * </pre>
	 * 
	 * @return List of all query templates accessible to the current user
	 * @throws ResourceRegistryException If an error occurs during query template retrieval
	 */
	public List<QueryTemplate> getAllQueryTemplates() throws ResourceRegistryException;

	/**
	 * Checks if a query template exists in the Resource Registry.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code GET /access/query-templates/{query-template-name}}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Verifies the existence of the specified query template using its name</li>
	 * <li>The template name is automatically extracted from the provided {@link QueryTemplate} object</li>
	 * <li>Does not return the template data, only confirms existence</li>
	 * <li>Checks accessibility based on user authorization.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * <ul>
	 * <li>This operation does not use query parameters for existence checking</li>
	 * <li>Existence is determined by attempting to retrieve the template without metadata.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Query template exists and is accessible</li>
	 * <li><strong>404 Not Found</strong>: Query template does not exist</li>
	 * <li><strong>403 Forbidden</strong>: Query template exists but user lacks authorization to access it</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The query template name is automatically extracted from the provided {@link QueryTemplate} object</li>
	 * <li>Only the name field is used for the existence check.</li>
	 * </ul>
	 *
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>Query templates are global resources accessible across contexts</li>
	 * <li>Access level depends on user role and authorization.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Full access to check existence of any query template</li>
	 * <li><strong>Infrastructure-Manager</strong>: Full access to check existence of any query template</li>
	 * <li><strong>All Other Users</strong>: Basic access with metadata filtering; can check existence of accessible templates.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryClient client = ResourceRegistryClientFactory.create();
	 * 
	 * // Check if a query template exists
	 * QueryTemplate template = new QueryTemplate();
	 * template.setName("GetAllEServiceWithState");
	 * boolean exists = client.existQueryTemplate(template);
	 * </pre>
	 * 
	 * @param queryTemplate The query template to check for existence (must have a valid name)
	 * @return true if the query template exists and is accessible, false if not found
	 * @throws ResourceRegistryException If an error occurs during the existence check
	 */
	public boolean existQueryTemplate(QueryTemplate queryTemplate) throws ResourceRegistryException;

	/**
	 * Checks if a query template exists in the Resource Registry using template name.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code GET /access/query-templates/{query-template-name}}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Verifies the existence of the specified query template using its name</li>
	 * <li>Does not return the template data, only confirms existence</li>
	 * <li>Checks accessibility based on user authorization</li>
	 * <li>Most direct method when template name is already known.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * <ul>
	 * <li>This operation does not use query parameters for existence checking</li>
	 * <li>Existence is determined by attempting to retrieve the template without metadata.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Query template exists and is accessible</li>
	 * <li><strong>404 Not Found</strong>: Query template does not exist</li>
	 * <li><strong>403 Forbidden</strong>: Query template exists but user lacks authorization to access it</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>Template name must be a valid query template name</li>
	 * <li>Template names are case-sensitive and must match exactly.</li>
	 * </ul>
	 * 
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>Query templates are global resources accessible across contexts</li>
	 * <li>Access level depends on user role and authorization.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Full access to check existence of any query template</li>
	 * <li><strong>Infrastructure-Manager</strong>: Full access to check existence of any query template</li>
	 * <li><strong>All Other Users</strong>: Basic access with metadata filtering; can check existence of accessible templates.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryClient client = ResourceRegistryClientFactory.create();
	 * 
	 * // Check if a query template exists by name
	 * boolean exists = client.existQueryTemplate("GetAllEServiceWithState");
	 * </pre>
	 * 
	 * @param queryTemplateName The name of the query template to check for existence
	 * @return true if the query template exists and is accessible, false if not found
	 * @throws ResourceRegistryException If an error occurs during the existence check
	 */
	public boolean existQueryTemplate(String queryTemplateName) throws ResourceRegistryException;

	/**
	 * Reads an existing query template from the Resource Registry using a provided template object.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code GET /access/query-templates/{query-template-name}[?includeMeta={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Retrieves an existing query template by extracting name from the provided template object</li>
	 * <li>The template must be accessible based on user authorization</li>
	 * <li>Returns the complete template with all properties and metadata</li>
	 * <li>The name is automatically determined from the provided template object.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response query template</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code client.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.QueryTemplatePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager and Infrastructure-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Query template successfully retrieved</li>
	 * <li><strong>404 Not Found</strong>: No query template found with the specified name</li>
	 * <li><strong>403 Forbidden</strong>: User lacks read permissions for the query template</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The query template name is automatically extracted from the provided {@link QueryTemplate} object</li>
	 * <li>Only the name field is used for template retrieval.</li>
	 * </ul>
	 * 
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>Query templates are global resources accessible across contexts</li>
	 * <li>Access level depends on user role and authorization.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Full access to retrieve any query template with complete, unfiltered metadata including all administrative fields</li>
	 * <li><strong>Infrastructure-Manager</strong>: Full access to retrieve any query template with complete, unfiltered metadata including all administrative fields</li>
	 * <li><strong>All Other Users</strong>: Basic access with metadata filtering; receive metadata with sensitive information filtered when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryClient client = ResourceRegistryClientFactory.create();
	 * 
	 * // Read a query template using template object
	 * QueryTemplate template = new QueryTemplate();
	 * template.setName("GetAllEServiceWithState");
	 * QueryTemplate readTemplate = client.readQueryTemplate(template);
	 * </pre>
	 * 
	 * @param queryTemplate Template object containing at least the name
	 * @return The complete query template from the Resource Registry
	 * @throws QueryTemplateNotFoundException If no query template is found with the specified name
	 * @throws ResourceRegistryException If an error occurs during template retrieval
	 */
	public QueryTemplate readQueryTemplate(QueryTemplate queryTemplate)
			throws QueryTemplateNotFoundException, ResourceRegistryException;

	/**
	 * Reads an existing query template from the Resource Registry by its name.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code GET /access/query-templates/{query-template-name}[?includeMeta={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Retrieves an existing query template by its name</li>
	 * <li>The template must be accessible based on user authorization</li>
	 * <li>Returns the complete template with all properties and metadata</li>
	 * <li>Most direct method when template name is already known.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response query template</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code client.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.QueryTemplatePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager and Infrastructure-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Query template successfully retrieved</li>
	 * <li><strong>404 Not Found</strong>: No query template found with the specified name</li>
	 * <li><strong>403 Forbidden</strong>: User lacks read permissions for the query template</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The specified name must be a valid query template name</li>
	 * <li>Template names are case-sensitive and must match exactly</li>
	 * <li>Name must correspond to an existing template.</li>
	 * </ul>
	 * 
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>Query templates are global resources accessible across contexts</li>
	 * <li>Access level depends on user role and authorization.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Full access to retrieve any query template with complete, unfiltered metadata including all administrative fields</li>
	 * <li><strong>Infrastructure-Manager</strong>: Full access to retrieve any query template with complete, unfiltered metadata including all administrative fields</li>
	 * <li><strong>All Other Users</strong>: Basic access with metadata filtering; receive metadata with sensitive information filtered when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryClient client = ResourceRegistryClientFactory.create();
	 * 
	 * // Read a query template by name
	 * QueryTemplate template = client.readQueryTemplate("GetAllEServiceWithState");
	 * </pre>
	 * 
	 * @param queryTemplateName The name of the query template to read
	 * @return The complete query template from the Resource Registry
	 * @throws QueryTemplateNotFoundException If no query template is found with the specified name
	 * @throws ResourceRegistryException If an error occurs during template retrieval
	 */
	public QueryTemplate readQueryTemplate(String queryTemplateName)
			throws QueryTemplateNotFoundException, ResourceRegistryException;

	/**
	 * Reads an existing query template from the Resource Registry as JSON string.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code GET /access/query-templates/{query-template-name}[?includeMeta={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Retrieves an existing query template by its name</li>
	 * <li>The template must be accessible based on user authorization</li>
	 * <li>Returns the complete template as JSON with all properties and metadata</li>
	 * <li>Returns the response as a JSON object string.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response query template</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code client.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.QueryTemplatePath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager and Infrastructure-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Query template successfully retrieved</li>
	 * <li><strong>404 Not Found</strong>: No query template found with the specified name</li>
	 * <li><strong>403 Forbidden</strong>: User lacks read permissions for the query template</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The specified name must be a valid query template name</li>
	 * <li>Template names are case-sensitive and must match exactly</li>
	 * <li>Name must correspond to an existing template.</li>
	 * </ul>
	 * 
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>Query templates are global resources accessible across contexts</li>
	 * <li>Access level depends on user role and authorization.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Full access to retrieve any query template with complete, unfiltered metadata including all administrative fields</li>
	 * <li><strong>Infrastructure-Manager</strong>: Full access to retrieve any query template with complete, unfiltered metadata including all administrative fields</li>
	 * <li><strong>All Other Users</strong>: Basic access with metadata filtering; receive metadata with sensitive information filtered when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryClient client = ResourceRegistryClientFactory.create();
	 * 
	 * // Read a query template as JSON string
	 * String templateJson = client.readQueryTemplateAsString("GetAllEServiceWithState");
	 * </pre>
	 * 
	 * @param queryTemplateName The name of the query template to read
	 * @return JSON representation of the complete query template from the Resource Registry
	 * @throws QueryTemplateNotFoundException If no query template is found with the specified name
	 * @throws ResourceRegistryException If an error occurs during template retrieval
	 */
	public String readQueryTemplateAsString(String queryTemplateName)
			throws QueryTemplateNotFoundException, ResourceRegistryException;

	/**
	 * Executes a query template and returns results as JSON string (deprecated method).
	 * 
	 * <strong>Corresponding REST API:</strong> {@code POST /access/query-templates/{query-template-name}[?polymorphic={true|false}&includeMeta={true|false}&allMeta={true|false}&includeContexts={true|false}&hierarchical={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Executes the specified query template with no parameters</li>
	 * <li>Returns query results as a JSON array string</li>
	 * <li>Uses default parameter values if the template defines any parameters</li>
	 * <li>Results are filtered based on user authorization and context access.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Query template successfully executed</li>
	 * <li><strong>404 Not Found</strong>: Query template with the specified name does not exist</li>
	 * <li><strong>400 Bad Request</strong>: Invalid query execution or parameter values</li>
	 * <li><strong>403 Forbidden</strong>: Insufficient permissions to execute the query template</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>Template name must be a valid query template name</li>
	 * <li>Template names are case-sensitive and must match exactly</li>
	 * <li>Name must correspond to an existing template.</li>
	 * </ul>
	 * 
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>Query execution respects context boundaries based on user authorization</li>
	 * <li>Hierarchical mode can be enabled for administrative users to query across context hierarchies.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Full access to execute any query template with hierarchical querying across all contexts; receives complete, unfiltered metadata</li>
	 * <li><strong>Infrastructure-Manager</strong>: Full access to execute any query template with hierarchical querying across all contexts; receives complete, unfiltered metadata</li>
	 * <li><strong>All Other Users</strong>: Basic access with metadata filtering; receive metadata with sensitive information filtered when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryClient client = ResourceRegistryClientFactory.create();
	 * 
	 * // Execute a query template and get results as JSON
	 * String resultsJson = client.runQueryTemplate("GetAllEService");
	 * </pre>
	 * 
	 * @param name The name of the query template to execute
	 * @return JSON array string containing query results
	 * @throws QueryTemplateNotFoundException If the query template with the specified name does not exist
	 * @throws ResourceRegistryException If an error occurs during query template execution
	 */
	public String runQueryTemplate(String name)
			throws QueryTemplateNotFoundException, ResourceRegistryException;

	/**
	 * Executes a query template using template object and returns results as JSON string.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code POST /access/query-templates/{query-template-name}[?polymorphic={true|false}&includeMeta={true|false}&allMeta={true|false}&includeContexts={true|false}&hierarchical={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Executes the query template with no parameters</li>
	 * <li>The template name is extracted from the provided {@link QueryTemplate} object</li>
	 * <li>Returns query results as a JSON array string</li>
	 * <li>Uses default parameter values if the template defines any parameters</li>
	 * <li>Results are filtered based on user authorization and context access.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Query template successfully executed</li>
	 * <li><strong>404 Not Found</strong>: Query template with the specified name does not exist</li>
	 * <li><strong>400 Bad Request</strong>: Invalid query execution or parameter values</li>
	 * <li><strong>403 Forbidden</strong>: Insufficient permissions to execute the query template</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The query template name is automatically extracted from the provided {@link QueryTemplate} object</li>
	 * <li>Only the name field is used for template execution.</li>
	 * </ul>
	 * 
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>Query execution respects context boundaries based on user authorization</li>
	 * <li>Hierarchical mode can be enabled for administrative users to query across context hierarchies.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Full access to execute any query template with hierarchical querying across all contexts; receives complete, unfiltered metadata</li>
	 * <li><strong>Infrastructure-Manager</strong>: Full access to execute any query template with hierarchical querying across all contexts; receives complete, unfiltered metadata</li>
	 * <li><strong>All Other Users</strong>: Basic access with metadata filtering; receive metadata with sensitive information filtered when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryClient client = ResourceRegistryClientFactory.create();
	 * 
	 * // Execute a query template using template object
	 * QueryTemplate template = new QueryTemplate();
	 * template.setName("GetAllEService");
	 * String resultsJson = client.runQueryTemplate(template);
	 * </pre>
	 * 
	 * @param queryTemplate The query template to execute (name is extracted from this object)
	 * @return JSON array string containing query results
	 * @throws QueryTemplateNotFoundException If the query template with the specified name does not exist
	 * @throws ResourceRegistryException If an error occurs during query template execution
	 */
	public String runQueryTemplate(QueryTemplate queryTemplate)
			throws QueryTemplateNotFoundException, ResourceRegistryException;

	/**
	 * Executes a query template with string parameters and returns results as JSON string.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code POST /access/query-templates/{query-template-name}[?polymorphic={true|false}&includeMeta={true|false}&allMeta={true|false}&includeContexts={true|false}&hierarchical={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Executes the specified query template with provided parameter values</li>
	 * <li>Parameters are provided as a JSON string containing parameter substitutions</li>
	 * <li>Returns query results as a JSON array string</li>
	 * <li>Template variables are replaced with provided parameter values</li>
	 * <li>Results are filtered based on user authorization and context access.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Query template successfully executed</li>
	 * <li><strong>404 Not Found</strong>: Query template with the specified name does not exist</li>
	 * <li><strong>400 Bad Request</strong>: Invalid query execution, parameter values, or parameter format</li>
	 * <li><strong>403 Forbidden</strong>: Insufficient permissions to execute the query template</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>Template name must be a valid query template name</li>
	 * <li>Template names are case-sensitive and must match exactly</li>
	 * <li>Parameters must be provided as valid JSON string containing parameter values for substitution.</li>
	 * </ul>
	 * 
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>Query execution respects context boundaries based on user authorization</li>
	 * <li>Hierarchical mode can be enabled for administrative users to query across context hierarchies.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Full access to execute any query template with hierarchical querying across all contexts; receives complete, unfiltered metadata</li>
	 * <li><strong>Infrastructure-Manager</strong>: Full access to execute any query template with hierarchical querying across all contexts; receives complete, unfiltered metadata</li>
	 * <li><strong>All Other Users</strong>: Basic access with metadata filtering; receive metadata with sensitive information filtered when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryClient client = ResourceRegistryClientFactory.create();
	 * 
	 * // Execute with string parameters
	 * String params = "{\"state\":\"ready\",\"namePattern\":\"%Service%\"}";
	 * String resultsJson = client.runQueryTemplate("GetAllEServiceWithState", params);
	 * </pre>
	 * 
	 * @param name The name of the query template to execute
	 * @param params JSON string containing parameter values for template substitution
	 * @return JSON array string containing query results
	 * @throws QueryTemplateNotFoundException If the query template with the specified name does not exist
	 * @throws ResourceRegistryException If an error occurs during query template execution
	 */
	public String runQueryTemplate(String name, String params)
			throws QueryTemplateNotFoundException, ResourceRegistryException;

	/**
	 * Executes a query template with JsonNode parameters and returns results as JSON string.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code POST /access/query-templates/{query-template-name}[?polymorphic={true|false}&includeMeta={true|false}&allMeta={true|false}&includeContexts={true|false}&hierarchical={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Executes the specified query template with provided parameter values</li>
	 * <li>Parameters are provided as a JsonNode object containing parameter substitutions</li>
	 * <li>Returns query results as a JSON array string</li>
	 * <li>Template variables are replaced with provided parameter values</li>
	 * <li>Results are filtered based on user authorization and context access.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Query template successfully executed</li>
	 * <li><strong>404 Not Found</strong>: Query template with the specified name does not exist</li>
	 * <li><strong>400 Bad Request</strong>: Invalid query execution, parameter values, or parameter format</li>
	 * <li><strong>403 Forbidden</strong>: Insufficient permissions to execute the query template</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>Template name must be a valid query template name</li>
	 * <li>Template names are case-sensitive and must match exactly</li>
	 * <li>Parameters must be provided as valid JsonNode containing parameter values for substitution.</li>
	 * </ul>
	 * 
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>Query execution respects context boundaries based on user authorization</li>
	 * <li>Hierarchical mode can be enabled for administrative users to query across context hierarchies.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Full access to execute any query template with hierarchical querying across all contexts; receives complete, unfiltered metadata</li>
	 * <li><strong>Infrastructure-Manager</strong>: Full access to execute any query template with hierarchical querying across all contexts; receives complete, unfiltered metadata</li>
	 * <li><strong>All Other Users</strong>: Basic access with metadata filtering; receive metadata with sensitive information filtered when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryClient client = ResourceRegistryClientFactory.create();
	 * 
	 * // Execute with JsonNode parameters
	 * ObjectMapper mapper = new ObjectMapper();
	 * JsonNode params = mapper.createObjectNode()
	 *     .put("state", "ready")
	 *     .put("namePattern", "%Service%");
	 * String resultsJson = client.runQueryTemplate("GetAllEServiceWithState", params);
	 * </pre>
	 * 
	 * @param name The name of the query template to execute
	 * @param jsonNode JsonNode containing parameter values for template substitution
	 * @return JSON array string containing query results
	 * @throws QueryTemplateNotFoundException If the query template with the specified name does not exist
	 * @throws ResourceRegistryException If an error occurs during query template execution
	 */
	public String runQueryTemplate(String name, JsonNode jsonNode)
			throws QueryTemplateNotFoundException, ResourceRegistryException;

	/**
	 * Executes a query template using template object with JsonNode parameters and returns results as JSON string.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code POST /access/query-templates/{query-template-name}[?polymorphic={true|false}&includeMeta={true|false}&allMeta={true|false}&includeContexts={true|false}&hierarchical={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Executes the query template with provided parameter values</li>
	 * <li>The template name is extracted from the provided {@link QueryTemplate} object</li>
	 * <li>Parameters are provided as a JsonNode object containing parameter substitutions</li>
	 * <li>Returns query results as a JSON array string</li>
	 * <li>Template variables are replaced with provided parameter values</li>
	 * <li>Results are filtered based on user authorization and context access.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Query template successfully executed</li>
	 * <li><strong>404 Not Found</strong>: Query template with the specified name does not exist</li>
	 * <li><strong>400 Bad Request</strong>: Invalid query execution, parameter values, or parameter format</li>
	 * <li><strong>403 Forbidden</strong>: Insufficient permissions to execute the query template</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>The query template name is automatically extracted from the provided {@link QueryTemplate} object</li>
	 * <li>Parameter values from JsonNode are used for template variable substitution</li>
	 * <li>All template parameters must be provided in the JsonNode or have default values.</li>
	 * </ul>
	 * 
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>Query execution respects context boundaries based on user authorization</li>
	 * <li>Hierarchical mode can be enabled for administrative users to query across context hierarchies.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Full access to execute any query template with hierarchical querying across all contexts; receives complete, unfiltered metadata</li>
	 * <li><strong>Infrastructure-Manager</strong>: Full access to execute any query template with hierarchical querying across all contexts; receives complete, unfiltered metadata</li>
	 * <li><strong>All Other Users</strong>: Basic access with metadata filtering; receive metadata with sensitive information filtered when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryClient client = ResourceRegistryClientFactory.create();
	 * 
	 * // Execute using template object with JsonNode parameters
	 * QueryTemplate template = new QueryTemplate();
	 * template.setName("GetAllEServiceWithState");
	 * 
	 * ObjectMapper mapper = new ObjectMapper();
	 * JsonNode params = mapper.createObjectNode()
	 *     .put("state", "ready")
	 *     .put("namePattern", "%Service%");
	 * 
	 * String resultsJson = client.runQueryTemplate(template, params);
	 * </pre>
	 * 
	 * @param queryTemplate The query template to execute (name is extracted from this object)
	 * @param jsonNode JsonNode containing parameter values for template substitution
	 * @return JSON array string containing query results
	 * @throws QueryTemplateNotFoundException If the query template with the specified name does not exist
	 * @throws ResourceRegistryException If an error occurs during query template execution
	 */
	public String runQueryTemplate(QueryTemplate queryTemplate, JsonNode jsonNode)
			throws QueryTemplateNotFoundException, ResourceRegistryException;

	
	/* ----------------------- Prepared Queries APIs ------------------------ */

	/**
	 * Gets resources connected to a specific facet instance through a ConsistsOf relation (typed version).
	 * 
	 * <strong>Corresponding REST API:</strong> {@code GET /access/query/{resource-type-name}/{relation-type-name}/{reference-type-name}[?_reference={reference-uuid}&_polymorphic={true|false}&_direction={out|in}&limit={number}&offset={number}&includeMeta={true|false}&allMeta={true|false}&hierarchical={true|false}&includeContexts={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Retrieves all resources connected to the specified facet instance through the specified relation</li>
	 * <li>Returns strongly-typed resource objects filtered based on user authorization</li>
	 * <li>Supports polymorphic queries to include subtypes and sophisticated filtering options</li>
	 * <li>Results are filtered based on context access and user permissions.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>_direction</strong> (automatically set to "out"):
	 * <ul>
	 * <li>Direction of relation traversal from resource perspective</li>
	 * <li>Fixed value: "out" (outgoing relations from resource to facet)</li>
	 * <li>Represents: Resources having outgoing ConsistsOf relations to the specified facet</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#_DIRECTION_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>limit</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Maximum number of resources to return in a single response</li>
	 * <li>Default value: server configuration dependent</li>
	 * <li>Usage: {@code client.setLimit(50)}</li>
	 * <li>Example: Set to 50 to get at most 50 resources per request</li>
	 * <li>Unlimited results: Set to {@code RequestInfo.UNBOUNDED_LIMIT} (-1) for unlimited results (may cause timeout if results are too many)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#LIMIT_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>offset</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Number of resources to skip from the beginning of the result set</li>
	 * <li>Default value: 0</li>
	 * <li>Usage: {@code client.setOffset(10)}</li>
	 * <li>Example: Set to 10 to skip the first 10 resources (useful for pagination)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#OFFSET_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response resources</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code client.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata for all nested elements (ConsistsOf relations, Facets, etc.) in the response</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main resources, more human-readable)</li>
	 * <li>Usage: {@code client.setAllMeta(true)}</li>
	 * <li>Values: true (complete metadata) | false (main resources only, more readable)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include the list of contexts where the resources and their nested elements are available in the response</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Usage: {@code client.setIncludeContexts(true)}</li>
	 * <li>Values: true (shows context availability) | false (no context information)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#INCLUDE_CONTEXTS_QUERY_PARAMETER}</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present</li>
	 * <li><strong>Note:</strong> A Facet's context availability depends on the Resources that include it via ConsistsOf relations.</li>
	 * </ul>
	 * 
	 * <strong>hierarchical</strong> (configurable via client configuration, admin-only):
	 * <ul>
	 * <li>Whether to include resources from child contexts of the current context</li>
	 * <li>Default value: false (child contexts not included)</li>
	 * <li>Usage: {@code client.setHierarchicalMode(true)}</li>
	 * <li>Values: true (includes child contexts) | false (current context only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#HIERARCHICAL_MODE_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> Only available to IS-Manager, Infrastructure-Manager, and Context-Manager roles.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Resources successfully retrieved</li>
	 * <li><strong>400 Bad Request</strong>: Invalid resource type, relation type, facet type, or query parameters</li>
	 * <li><strong>403 Forbidden</strong>: Insufficient permissions to execute prepared queries or access hierarchical data</li>
	 * <li><strong>404 Not Found</strong>: Specified types do not exist in the model.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>Resource, relation, and facet classes are automatically converted to type names</li>
	 * <li>The reference facet UUID is extracted from the provided facet instance</li>
	 * <li>Polymorphic parameter controls whether subtypes are included in results.</li>
	 * </ul>
	 * 
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>Resources are returned from the current context based on authorization</li>
	 * <li>Hierarchical mode can be enabled for administrative users to query across context hierarchies</li>
	 * <li>Results respect context boundaries and user access permissions.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Full access to execute prepared queries with hierarchical querying across all contexts; receives complete, unfiltered metadata</li>
	 * <li><strong>Infrastructure-Manager</strong>: Full access to execute prepared queries with hierarchical querying across all contexts; receives complete, unfiltered metadata</li>
	 * <li><strong>All Other Users</strong>: Basic access with metadata filtering; receive metadata with sensitive information filtered when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryClient client = ResourceRegistryClientFactory.create();
	 * 
	 * // Find all EServices identified by a specific SoftwareFacet
	 * SoftwareFacet softwareFacet = // ... obtained from somewhere
	 * List&lt;EService&gt; services = client.getResourcesFromReferenceFacet(
	 *     EService.class, IsIdentifiedBy.class, softwareFacet, true);
	 * </pre>
	 * 
	 * @param <R> The specific Resource type to retrieve
	 * @param <C> The specific ConsistsOf relation type connecting resources to facets
	 * @param <F> The specific Facet type used as reference
	 * @param resourceClass The class of resources to retrieve
	 * @param consistsOfClass The class of the relation connecting resources to facets
	 * @param referenceFacet The specific facet instance to use as reference. The UUID of this facet is automatically 
	 *                       set in the {@code _reference} query parameter of the REST API call.
	 * @param polymorphic Whether to include subtypes in the results. When true, includes all subtypes 
	 *                    of the specified type; when false, returns only instances of the exact specified type. 
	 *                    This parameter is automatically set in the {@code _polymorphic} query parameter of the 
	 *                    REST API call.
	 * @return List of resources connected to the specified facet instance
	 * @throws ResourceRegistryException If an error occurs during query execution
	 */
	public <R extends Resource, C extends ConsistsOf<?, ?>, F extends Facet> List<R> getResourcesFromReferenceFacet(
			Class<R> resourceClass, Class<C> consistsOfClass, F referenceFacet, boolean polymorphic)
			throws ResourceRegistryException;

	/**
	 * Gets resources connected to a facet of specific type and UUID through a ConsistsOf relation (typed version).
	 * 
	 * <strong>Corresponding REST API:</strong> {@code GET /access/query/{resource-type-name}/{relation-type-name}/{reference-type-name}[?_reference={reference-uuid}&_polymorphic={true|false}&_direction={out|in}&limit={number}&offset={number}&includeMeta={true|false}&allMeta={true|false}&hierarchical={true|false}&includeContexts={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Retrieves all resources connected to a facet of the specified type and UUID through the specified relation</li>
	 * <li>Returns strongly-typed resource objects filtered based on user authorization</li>
	 * <li>Supports polymorphic queries to include subtypes and sophisticated filtering options</li>
	 * <li>Results are filtered based on context access and user permissions.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>_direction</strong> (automatically set to "out"):
	 * <ul>
	 * <li>Direction of relation traversal from resource perspective</li>
	 * <li>Fixed value: "out" (outgoing relations from resource to facet)</li>
	 * <li>Represents: Resources having outgoing ConsistsOf relations to the specified facet</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#_DIRECTION_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>limit</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Maximum number of resources to return in a single response</li>
	 * <li>Default value: server configuration dependent</li>
	 * <li>Usage: {@code client.setLimit(50)}</li>
	 * <li>Example: Set to 50 to get at most 50 resources per request</li>
	 * <li>Unlimited results: Set to {@code RequestInfo.UNBOUNDED_LIMIT} (-1) for unlimited results (may cause timeout if results are too many)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#LIMIT_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>offset</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Number of resources to skip from the beginning of the result set</li>
	 * <li>Default value: 0</li>
	 * <li>Usage: {@code client.setOffset(10)}</li>
	 * <li>Example: Set to 10 to skip the first 10 resources (useful for pagination)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#OFFSET_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response resources</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code client.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata for all nested elements (ConsistsOf relations, Facets, etc.) in the response</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main resources, more human-readable)</li>
	 * <li>Usage: {@code client.setAllMeta(true)}</li>
	 * <li>Values: true (complete metadata) | false (main resources only, more readable)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include the list of contexts where the resources and their nested elements are available in the response</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Usage: {@code client.setIncludeContexts(true)}</li>
	 * <li>Values: true (shows context availability) | false (no context information)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#INCLUDE_CONTEXTS_QUERY_PARAMETER}</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present</li>
	 * <li><strong>Note:</strong> A Facet's context availability depends on the Resources that include it via ConsistsOf relations.</li>
	 * </ul>
	 * 
	 * <strong>hierarchical</strong> (configurable via client configuration, admin-only):
	 * <ul>
	 * <li>Whether to include resources from child contexts of the current context</li>
	 * <li>Default value: false (child contexts not included)</li>
	 * <li>Usage: {@code client.setHierarchicalMode(true)}</li>
	 * <li>Values: true (includes child contexts) | false (current context only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#HIERARCHICAL_MODE_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> Only available to IS-Manager, Infrastructure-Manager, and Context-Manager roles.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Resources successfully retrieved</li>
	 * <li><strong>400 Bad Request</strong>: Invalid resource type, relation type, facet type, UUID format, or query parameters</li>
	 * <li><strong>403 Forbidden</strong>: Insufficient permissions to execute prepared queries or access hierarchical data</li>
	 * <li><strong>404 Not Found</strong>: Specified types do not exist in the model.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>Resource, relation, and facet classes are automatically converted to type names</li>
	 * <li>The reference facet UUID must be a valid UUID format</li>
	 * <li>Polymorphic parameter controls whether subtypes are included in results.</li>
	 * </ul>
	 * 
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>Resources are returned from the current context based on authorization</li>
	 * <li>Hierarchical mode can be enabled for administrative users to query across context hierarchies</li>
	 * <li>Results respect context boundaries and user access permissions.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Full access to execute prepared queries with hierarchical querying across all contexts; receives complete, unfiltered metadata</li>
	 * <li><strong>Infrastructure-Manager</strong>: Full access to execute prepared queries with hierarchical querying across all contexts; receives complete, unfiltered metadata</li>
	 * <li><strong>All Other Users</strong>: Basic access with metadata filtering; receive metadata with sensitive information filtered when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryClient client = ResourceRegistryClientFactory.create();
	 * 
	 * // Find all EServices identified by a SoftwareFacet with specific UUID
	 * UUID softwareFacetUUID = UUID.fromString("12345678-1234-1234-1234-123456789abc");
	 * List&lt;EService&gt; services = client.getResourcesFromReferenceFacet(
	 *     EService.class, IsIdentifiedBy.class, SoftwareFacet.class, softwareFacetUUID, true);
	 * </pre>
	 * 
	 * @param <R> The specific Resource type to retrieve
	 * @param <C> The specific ConsistsOf relation type connecting resources to facets
	 * @param <F> The specific Facet type used as reference
	 * @param resourceClass The class of resources to retrieve
	 * @param consistsOfClass The class of the relation connecting resources to facets
	 * @param facetClass The class of facets used as reference
	 * @param referenceFacetUUID The UUID of the specific facet instance to use as reference. This value is automatically 
	 *                           set in the {@code _reference} query parameter of the REST API call.
	 * @param polymorphic Whether to include subtypes in the results. When true, includes all subtypes 
	 *                    of the specified type; when false, returns only instances of the exact specified type. 
	 *                    This parameter is automatically set in the {@code _polymorphic} query parameter of the 
	 *                    REST API call.
	 * @return List of resources connected to the specified facet instance
	 * @throws ResourceRegistryException If an error occurs during query execution
	 */
	public <R extends Resource, C extends ConsistsOf<?, ?>, F extends Facet> List<R> getResourcesFromReferenceFacet(
			Class<R> resourceClass, Class<C> consistsOfClass, Class<F> facetClass, UUID referenceFacetUUID,
			boolean polymorphic) throws ResourceRegistryException;

	/**
	 * Gets resources connected to a facet of specific type and UUID through a ConsistsOf relation (string version).
	 * 
	 * <strong>Corresponding REST API:</strong> {@code GET /access/query/{resource-type-name}/{relation-type-name}/{reference-type-name}[?_reference={reference-uuid}&_polymorphic={true|false}&_direction={out|in}&limit={number}&offset={number}&includeMeta={true|false}&allMeta={true|false}&hierarchical={true|false}&includeContexts={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Retrieves all resources connected to a facet of the specified type and UUID through the specified relation</li>
	 * <li>Returns JSON string representation of resources filtered based on user authorization</li>
	 * <li>Supports polymorphic queries to include subtypes and sophisticated filtering options</li>
	 * <li>Results are filtered based on context access and user permissions.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>_direction</strong> (automatically set to "out"):
	 * <ul>
	 * <li>Direction of relation traversal from resource perspective</li>
	 * <li>Fixed value: "out" (outgoing relations from resource to facet)</li>
	 * <li>Represents: Resources having outgoing ConsistsOf relations to the specified facet</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#_DIRECTION_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>limit</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Maximum number of resources to return in a single response</li>
	 * <li>Default value: server configuration dependent</li>
	 * <li>Usage: {@code client.setLimit(50)}</li>
	 * <li>Example: Set to 50 to get at most 50 resources per request</li>
	 * <li>Unlimited results: Set to {@code RequestInfo.UNBOUNDED_LIMIT} (-1) for unlimited results (may cause timeout if results are too many)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#LIMIT_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>offset</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Number of resources to skip from the beginning of the result set</li>
	 * <li>Default value: 0</li>
	 * <li>Usage: {@code client.setOffset(10)}</li>
	 * <li>Example: Set to 10 to skip the first 10 resources (useful for pagination)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#OFFSET_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response resources</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code client.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata for all nested elements (ConsistsOf relations, Facets, etc.) in the response</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main resources, more human-readable)</li>
	 * <li>Usage: {@code client.setAllMeta(true)}</li>
	 * <li>Values: true (complete metadata) | false (main resources only, more readable)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include the list of contexts where the resources and their nested elements are available in the response</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Usage: {@code client.setIncludeContexts(true)}</li>
	 * <li>Values: true (shows context availability) | false (no context information)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#INCLUDE_CONTEXTS_QUERY_PARAMETER}</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present</li>
	 * <li><strong>Note:</strong> A Facet's context availability depends on the Resources that include it via ConsistsOf relations.</li>
	 * </ul>
	 * 
	 * <strong>hierarchical</strong> (configurable via client configuration, admin-only):
	 * <ul>
	 * <li>Whether to include resources from child contexts of the current context</li>
	 * <li>Default value: false (child contexts not included)</li>
	 * <li>Usage: {@code client.setHierarchicalMode(true)}</li>
	 * <li>Values: true (includes child contexts) | false (current context only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#HIERARCHICAL_MODE_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> Only available to IS-Manager, Infrastructure-Manager, and Context-Manager roles.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Resources successfully retrieved</li>
	 * <li><strong>400 Bad Request</strong>: Invalid resource type, relation type, facet type, UUID format, or query parameters</li>
	 * <li><strong>403 Forbidden</strong>: Insufficient permissions to execute prepared queries or access hierarchical data</li>
	 * <li><strong>404 Not Found</strong>: Specified types do not exist in the model.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>Resource, relation, and facet type names must be valid Information System model types</li>
	 * <li>The reference facet UUID must be a valid UUID format</li>
	 * <li>Polymorphic parameter controls whether subtypes are included in results.</li>
	 * </ul>
	 * 
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>Resources are returned from the current context based on authorization</li>
	 * <li>Hierarchical mode can be enabled for administrative users to query across context hierarchies</li>
	 * <li>Results respect context boundaries and user access permissions.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Full access to execute prepared queries with hierarchical querying across all contexts; receives complete, unfiltered metadata</li>
	 * <li><strong>Infrastructure-Manager</strong>: Full access to execute prepared queries with hierarchical querying across all contexts; receives complete, unfiltered metadata</li>
	 * <li><strong>All Other Users</strong>: Basic access with metadata filtering; receive metadata with sensitive information filtered when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryClient client = ResourceRegistryClientFactory.create();
	 * 
	 * // Find all EServices identified by a SoftwareFacet with specific UUID
	 * UUID softwareFacetUUID = UUID.fromString("12345678-1234-1234-1234-123456789abc");
	 * String servicesJson = client.getResourcesFromReferenceFacet(
	 *     "EService", "IsIdentifiedBy", "SoftwareFacet", softwareFacetUUID, true);
	 * </pre>
	 * 
	 * @param resourceType The name of the resource type to retrieve
	 * @param consistsOfType The name of the relation type connecting resources to facets
	 * @param facetType The name of the facet type used as reference
	 * @param referenceFacetUUID The UUID of the specific facet instance to use as reference. This value is automatically 
	 *                           set in the {@code _reference} query parameter of the REST API call.
	 * @param polymorphic Whether to include subtypes in the results. When true, includes all subtypes 
	 *                    of the specified type; when false, returns only instances of the exact specified type. 
	 *                    This parameter is automatically set in the {@code _polymorphic} query parameter of the 
	 *                    REST API call.
	 * @return JSON string representation of resources connected to the specified facet instance
	 * @throws ResourceRegistryException If an error occurs during query execution
	 */
	public String getResourcesFromReferenceFacet(String resourceType, String consistsOfType, String facetType,
			UUID referenceFacetUUID, boolean polymorphic) throws ResourceRegistryException;

	/**
	 * Gets filtered resources based on facet constraints and relations (typed version).
	 * 
	 * <strong>Corresponding REST API:</strong> {@code GET /access/query/{resource-type-name}/{relation-type-name}/{reference-type-name}[?_polymorphic={true|false}&_direction={out|in}&{constraint-key}={constraint-value}&limit={number}&offset={number}&includeMeta={true|false}&allMeta={true|false}&hierarchical={true|false}&includeContexts={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Retrieves all resources connected to facets of the specified type that match the given constraints</li>
	 * <li>Applies property-based filtering on connected facets through specified relations</li>
	 * <li>Returns strongly-typed resource objects filtered based on user authorization</li>
	 * <li>Supports polymorphic queries to include subtypes and sophisticated filtering options.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>_direction</strong> (automatically set to "out"):
	 * <ul>
	 * <li>Direction of relation traversal from resource perspective</li>
	 * <li>Fixed value: "out" (outgoing relations from resource to facet)</li>
	 * <li>Represents: Resources having outgoing ConsistsOf relations to facets matching constraints</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#_DIRECTION_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>Constraint parameters</strong> (automatically configured from facetConstraint map):
	 * <ul>
	 * <li>Each key-value pair in facetConstraint becomes a query parameter</li>
	 * <li>Used to filter facets by their property values</li>
	 * <li>Example: facetConstraint.put("name", "data-transfer-service") becomes ?name=data-transfer-service</li>
	 * <li>Only resources connected to facets matching ALL constraint criteria are returned.</li>
	 * </ul>
	 * 
	 * <strong>limit</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Maximum number of resources to return in a single response</li>
	 * <li>Default value: server configuration dependent</li>
	 * <li>Usage: {@code client.setLimit(50)}</li>
	 * <li>Example: Set to 50 to get at most 50 resources per request</li>
	 * <li>Unlimited results: Set to {@code RequestInfo.UNBOUNDED_LIMIT} (-1) for unlimited results (may cause timeout if results are too many)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#LIMIT_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>offset</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Number of resources to skip from the beginning of the result set</li>
	 * <li>Default value: 0</li>
	 * <li>Usage: {@code client.setOffset(10)}</li>
	 * <li>Example: Set to 10 to skip the first 10 resources (useful for pagination)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#OFFSET_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response resources</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code client.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata for all nested elements (ConsistsOf relations, Facets, etc.) in the response</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main resources, more human-readable)</li>
	 * <li>Usage: {@code client.setAllMeta(true)}</li>
	 * <li>Values: true (complete metadata) | false (main resources only, more readable)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include the list of contexts where the resources and their nested elements are available in the response</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Usage: {@code client.setIncludeContexts(true)}</li>
	 * <li>Values: true (shows context availability) | false (no context information)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#INCLUDE_CONTEXTS_QUERY_PARAMETER}</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present</li>
	 * <li><strong>Note:</strong> A Facet's context availability depends on the Resources that include it via ConsistsOf relations.</li>
	 * </ul>
	 * 
	 * <strong>hierarchical</strong> (configurable via client configuration, admin-only):
	 * <ul>
	 * <li>Whether to include resources from child contexts of the current context</li>
	 * <li>Default value: false (child contexts not included)</li>
	 * <li>Usage: {@code client.setHierarchicalMode(true)}</li>
	 * <li>Values: true (includes child contexts) | false (current context only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#HIERARCHICAL_MODE_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> Only available to IS-Manager, Infrastructure-Manager, and Context-Manager roles.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Resources successfully retrieved</li>
	 * <li><strong>400 Bad Request</strong>: Invalid resource type, relation type, facet type, or query parameters</li>
	 * <li><strong>403 Forbidden</strong>: Insufficient permissions to execute prepared queries or access hierarchical data</li>
	 * <li><strong>404 Not Found</strong>: Specified types do not exist in the model.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>Resource, relation, and facet classes are automatically converted to type names</li>
	 * <li>Constraint map is converted to query parameters for facet property filtering</li>
	 * <li>Polymorphic parameter controls whether subtypes are included in results.</li>
	 * </ul>
	 * 
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>Resources are returned from the current context based on authorization</li>
	 * <li>Hierarchical mode can be enabled for administrative users to query across context hierarchies</li>
	 * <li>Results respect context boundaries and user access permissions.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Full access to execute prepared queries with hierarchical querying across all contexts; receives complete, unfiltered metadata</li>
	 * <li><strong>Infrastructure-Manager</strong>: Full access to execute prepared queries with hierarchical querying across all contexts; receives complete, unfiltered metadata</li>
	 * <li><strong>All Other Users</strong>: Basic access with metadata filtering; receive metadata with sensitive information filtered when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryClient client = ResourceRegistryClientFactory.create();
	 * 
	 * // Find all EServices identified by SoftwareFacets with specific name and group
	 * Map&lt;String, String&gt; constraints = new HashMap&lt;&gt;();
	 * constraints.put("name", "data-transfer-service");
	 * constraints.put("group", "DataTransfer");
	 * List&lt;EService&gt; services = client.getFilteredResources(
	 *     EService.class, IsIdentifiedBy.class, SoftwareFacet.class, true, constraints);
	 * </pre>
	 * 
	 * @param <R> The specific Resource type to retrieve
	 * @param <C> The specific ConsistsOf relation type connecting resources to facets
	 * @param <F> The specific Facet type used for filtering
	 * @param resourceClass The class of resources to retrieve
	 * @param consistsOfClass The class of the relation connecting resources to facets
	 * @param facetClass The class of facets used for filtering
	 * @param polymorphic Whether to include subtypes in the results. When true, includes all subtypes 
	 *                    of the specified type; when false, returns only instances of the exact specified type. 
	 *                    This parameter is automatically set in the {@code _polymorphic} query parameter of the 
	 *                    REST API call.
	 * @param facetConstraint Map of property name-value pairs to filter facets
	 * @return List of resources connected to facets matching the specified constraints
	 * @throws ResourceRegistryException If an error occurs during query execution
	 */
	public <R extends Resource, C extends ConsistsOf<?, ?>, F extends Facet> List<R> getFilteredResources(
			Class<R> resourceClass, Class<C> consistsOfClass, Class<F> facetClass, boolean polymorphic,
			Map<String, String> facetConstraint) throws ResourceRegistryException;

	/**
	 * Gets filtered resources based on facet constraints and relations (string version).
	 * 
	 * <strong>Corresponding REST API:</strong> {@code GET /access/query/{resource-type-name}/{relation-type-name}/{reference-type-name}[?_polymorphic={true|false}&_direction={out|in}&{constraint-key}={constraint-value}&limit={number}&offset={number}&includeMeta={true|false}&allMeta={true|false}&hierarchical={true|false}&includeContexts={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Retrieves all resources connected to facets of the specified type that match the given constraints</li>
	 * <li>Applies property-based filtering on connected facets through specified relations</li>
	 * <li>Returns JSON string representation of resources filtered based on user authorization</li>
	 * <li>Supports polymorphic queries to include subtypes and sophisticated filtering options.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>_direction</strong> (automatically set to "out"):
	 * <ul>
	 * <li>Direction of relation traversal from resource perspective</li>
	 * <li>Fixed value: "out" (outgoing relations from resource to facet)</li>
	 * <li>Represents: Resources having outgoing ConsistsOf relations to facets matching constraints</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#_DIRECTION_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>Constraint parameters</strong> (automatically configured from facetConstraint map):
	 * <ul>
	 * <li>Each key-value pair in facetConstraint becomes a query parameter</li>
	 * <li>Used to filter facets by their property values</li>
	 * <li>Example: facetConstraint.put("name", "data-transfer-service") becomes ?name=data-transfer-service</li>
	 * <li>Only resources connected to facets matching ALL constraint criteria are returned.</li>
	 * </ul>
	 * 
	 * <strong>limit</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Maximum number of resources to return in a single response</li>
	 * <li>Default value: server configuration dependent</li>
	 * <li>Usage: {@code client.setLimit(50)}</li>
	 * <li>Example: Set to 50 to get at most 50 resources per request</li>
	 * <li>Unlimited results: Set to {@code RequestInfo.UNBOUNDED_LIMIT} (-1) for unlimited results (may cause timeout if results are too many)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#LIMIT_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>offset</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Number of resources to skip from the beginning of the result set</li>
	 * <li>Default value: 0</li>
	 * <li>Usage: {@code client.setOffset(10)}</li>
	 * <li>Example: Set to 10 to skip the first 10 resources (useful for pagination)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#OFFSET_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response resources</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code client.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata for all nested elements (ConsistsOf relations, Facets, etc.) in the response</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main resources, more human-readable)</li>
	 * <li>Usage: {@code client.setAllMeta(true)}</li>
	 * <li>Values: true (complete metadata) | false (main resources only, more readable)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include the list of contexts where the resources and their nested elements are available in the response</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Usage: {@code client.setIncludeContexts(true)}</li>
	 * <li>Values: true (shows context availability) | false (no context information)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#INCLUDE_CONTEXTS_QUERY_PARAMETER}</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present</li>
	 * <li><strong>Note:</strong> A Facet's context availability depends on the Resources that include it via ConsistsOf relations.</li>
	 * </ul>
	 * 
	 * <strong>hierarchical</strong> (configurable via client configuration, admin-only):
	 * <ul>
	 * <li>Whether to include resources from child contexts of the current context</li>
	 * <li>Default value: false (child contexts not included)</li>
	 * <li>Usage: {@code client.setHierarchicalMode(true)}</li>
	 * <li>Values: true (includes child contexts) | false (current context only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#HIERARCHICAL_MODE_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> Only available to IS-Manager, Infrastructure-Manager, and Context-Manager roles.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Resources successfully retrieved</li>
	 * <li><strong>400 Bad Request</strong>: Invalid resource type, relation type, facet type, or query parameters</li>
	 * <li><strong>403 Forbidden</strong>: Insufficient permissions to execute prepared queries or access hierarchical data</li>
	 * <li><strong>404 Not Found</strong>: Specified types do not exist in the model.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>Resource, relation, and facet type names must be valid Information System model types</li>
	 * <li>Constraint map is converted to query parameters for facet property filtering</li>
	 * <li>Polymorphic parameter controls whether subtypes are included in results.</li>
	 * </ul>
	 * 
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>Resources are returned from the current context based on authorization</li>
	 * <li>Hierarchical mode can be enabled for administrative users to query across context hierarchies</li>
	 * <li>Results respect context boundaries and user access permissions.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Full access to execute prepared queries with hierarchical querying across all contexts; receives complete, unfiltered metadata</li>
	 * <li><strong>Infrastructure-Manager</strong>: Full access to execute prepared queries with hierarchical querying across all contexts; receives complete, unfiltered metadata</li>
	 * <li><strong>All Other Users</strong>: Basic access with metadata filtering; receive metadata with sensitive information filtered when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryClient client = ResourceRegistryClientFactory.create();
	 * 
	 * // Find all EServices identified by SoftwareFacets with specific name and group
	 * Map&lt;String, String&gt; constraints = new HashMap&lt;&gt;();
	 * constraints.put("name", "data-transfer-service");
	 * constraints.put("group", "DataTransfer");
	 * String servicesJson = client.getFilteredResources(
	 *     "EService", "IsIdentifiedBy", "SoftwareFacet", true, constraints);
	 * </pre>
	 * 
	 * @param resourceType The name of the resource type to retrieve
	 * @param consistsOfType The name of the relation type connecting resources to facets
	 * @param facetType The name of the facet type used for filtering
	 * @param polymorphic Whether to include subtypes in the results. When true, includes all subtypes 
	 *                    of the specified type; when false, returns only instances of the exact specified type. 
	 *                    This parameter is automatically set in the {@code _polymorphic} query parameter of the 
	 *                    REST API call.
	 * @param facetConstraint Map of property name-value pairs to filter facets
	 * @return JSON string representation of resources connected to facets matching the specified constraints
	 * @throws ResourceRegistryException If an error occurs during query execution
	 */
	public String getFilteredResources(String resourceType, String consistsOfType, String facetType,
			boolean polymorphic, Map<String, String> facetConstraint) throws ResourceRegistryException;

	/**
	 * Gets resources related to a specific reference resource through an IsRelatedTo relation (typed version with instance).
	 * 
	 * <strong>Corresponding REST API:</strong> {@code GET /access/query/{resource-type-name}/{relation-type-name}/{reference-type-name}[?_reference={reference-uuid}&_polymorphic={true|false}&_direction={out|in}&limit={number}&offset={number}&includeMeta={true|false}&allMeta={true|false}&hierarchical={true|false}&includeContexts={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Retrieves all resources related to the specified reference resource through the specified IsRelatedTo relation</li>
	 * <li>Returns strongly-typed resource objects filtered based on user authorization</li>
	 * <li>Supports polymorphic queries to include subtypes and directional relationship traversal</li>
	 * <li>Results are filtered based on context access and user permissions.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>limit</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Maximum number of resources to return in a single response</li>
	 * <li>Default value: server configuration dependent</li>
	 * <li>Usage: {@code client.setLimit(50)}</li>
	 * <li>Example: Set to 50 to get at most 50 resources per request</li>
	 * <li>Unlimited results: Set to {@code RequestInfo.UNBOUNDED_LIMIT} (-1) for unlimited results (may cause timeout if results are too many)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#LIMIT_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>offset</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Number of resources to skip from the beginning of the result set</li>
	 * <li>Default value: 0</li>
	 * <li>Usage: {@code client.setOffset(10)}</li>
	 * <li>Example: Set to 10 to skip the first 10 resources (useful for pagination)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#OFFSET_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response resources</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code client.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata for all nested elements (ConsistsOf relations, Facets, etc.) in the response</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main resources, more human-readable)</li>
	 * <li>Usage: {@code client.setAllMeta(true)}</li>
	 * <li>Values: true (complete metadata) | false (main resources only, more readable)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include the list of contexts where the resources and their nested elements are available in the response</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Usage: {@code client.setIncludeContexts(true)}</li>
	 * <li>Values: true (shows context availability) | false (no context information)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#INCLUDE_CONTEXTS_QUERY_PARAMETER}</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present</li>
	 * <li><strong>Note:</strong> A Facet's context availability depends on the Resources that include it via ConsistsOf relations.</li>
	 * </ul>
	 * 
	 * <strong>hierarchical</strong> (configurable via client configuration, admin-only):
	 * <ul>
	 * <li>Whether to include resources from child contexts of the current context</li>
	 * <li>Default value: false (child contexts not included)</li>
	 * <li>Usage: {@code client.setHierarchicalMode(true)}</li>
	 * <li>Values: true (includes child contexts) | false (current context only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#HIERARCHICAL_MODE_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> Only available to IS-Manager, Infrastructure-Manager, and Context-Manager roles.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Resources successfully retrieved</li>
	 * <li><strong>400 Bad Request</strong>: Invalid resource type, relation type, reference resource type, or query parameters</li>
	 * <li><strong>403 Forbidden</strong>: Insufficient permissions to execute prepared queries or access hierarchical data</li>
	 * <li><strong>404 Not Found</strong>: Specified types do not exist in the model.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>Resource, relation, and reference resource classes are automatically converted to type names</li>
	 * <li>The reference resource UUID is extracted from the provided reference resource instance</li>
	 * <li>Direction parameter controls the traversal direction of the IsRelatedTo relation.</li>
	 * </ul>
	 * 
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>Resources are returned from the current context based on authorization</li>
	 * <li>Hierarchical mode can be enabled for administrative users to query across context hierarchies</li>
	 * <li>Results respect context boundaries and user access permissions.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Full access to execute prepared queries with hierarchical querying across all contexts; receives complete, unfiltered metadata</li>
	 * <li><strong>Infrastructure-Manager</strong>: Full access to execute prepared queries with hierarchical querying across all contexts; receives complete, unfiltered metadata</li>
	 * <li><strong>All Other Users</strong>: Basic access with metadata filtering; receive metadata with sensitive information filtered when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryClient client = ResourceRegistryClientFactory.create();
	 * 
	 * // Find all EServices hosted by a specific HostingNode (incoming relation)
	 * HostingNode hostingNode = // ... obtained from somewhere
	 * List&lt;EService&gt; hostedServices = client.getRelatedResourcesFromReferenceResource(
	 *     EService.class, Hosts.class, hostingNode, Direction.IN, true);
	 * 
	 * // Find all HostingNodes that host a specific EService (outgoing relation)
	 * EService eservice = // ... obtained from somewhere
	 * List&lt;HostingNode&gt; hostingNodes = client.getRelatedResourcesFromReferenceResource(
	 *     HostingNode.class, Hosts.class, eservice, Direction.OUT, true);
	 * </pre>
	 * 
	 * @param <R> The specific Resource type to retrieve
	 * @param <I> The specific IsRelatedTo relation type connecting resources
	 * @param <RR> The specific reference Resource type
	 * @param resourceClass The class of resources to retrieve
	 * @param isRelatedToClass The class of the IsRelatedTo relation connecting resources
	 * @param referenceResource The specific reference resource instance. The UUID of this resource is automatically 
	 *                          set in the {@code _reference} query parameter of the REST API call.
	 * @param direction The direction of relation traversal (OUT or IN). Values: OUT finds resources that the 
	 *                  reference resource relates to; IN finds resources that relate to the reference resource. 
	 *                  This parameter is automatically set in the {@code _direction} query parameter of the REST API call.
	 * @param polymorphic Whether to include subtypes in the results. When true, includes all subtypes 
	 *                    of the specified type; when false, returns only instances of the exact specified type. 
	 *                    This parameter is automatically set in the {@code _polymorphic} query parameter of the 
	 *                    REST API call.
	 * @return List of resources related to the specified reference resource
	 * @throws ResourceRegistryException If an error occurs during query execution
	 */
	public <R extends Resource, I extends IsRelatedTo<?, ?>, RR extends Resource> List<R> getRelatedResourcesFromReferenceResource(
			Class<R> resourceClass, Class<I> isRelatedToClass, RR referenceResource, Direction direction,
			boolean polymorphic) throws ResourceRegistryException;

	/**
	 * Gets resources related to a specific reference resource through an IsRelatedTo relation (typed version with UUID).
	 * 
	 * <strong>Corresponding REST API:</strong> {@code GET /access/query/{resource-type-name}/{relation-type-name}/{reference-type-name}[?_reference={reference-uuid}&_polymorphic={true|false}&_direction={out|in}&limit={number}&offset={number}&includeMeta={true|false}&allMeta={true|false}&hierarchical={true|false}&includeContexts={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Retrieves all resources related to a reference resource with the specified UUID through the specified IsRelatedTo relation</li>
	 * <li>Returns strongly-typed resource objects filtered based on user authorization</li>
	 * <li>Supports polymorphic queries to include subtypes and directional relationship traversal</li>
	 * <li>Results are filtered based on context access and user permissions.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>limit</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Maximum number of resources to return in a single response</li>
	 * <li>Default value: server configuration dependent</li>
	 * <li>Usage: {@code client.setLimit(50)}</li>
	 * <li>Example: Set to 50 to get at most 50 resources per request</li>
	 * <li>Unlimited results: Set to {@code RequestInfo.UNBOUNDED_LIMIT} (-1) for unlimited results (may cause timeout if results are too many)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#LIMIT_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>offset</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Number of resources to skip from the beginning of the result set</li>
	 * <li>Default value: 0</li>
	 * <li>Usage: {@code client.setOffset(10)}</li>
	 * <li>Example: Set to 10 to skip the first 10 resources (useful for pagination)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#OFFSET_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response resources</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code client.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata for all nested elements (ConsistsOf relations, Facets, etc.) in the response</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main resources, more human-readable)</li>
	 * <li>Usage: {@code client.setAllMeta(true)}</li>
	 * <li>Values: true (complete metadata) | false (main resources only, more readable)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include the list of contexts where the resources and their nested elements are available in the response</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Usage: {@code client.setIncludeContexts(true)}</li>
	 * <li>Values: true (shows context availability) | false (no context information)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#INCLUDE_CONTEXTS_QUERY_PARAMETER}</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present</li>
	 * <li><strong>Note:</strong> A Facet's context availability depends on the Resources that include it via ConsistsOf relations.</li>
	 * </ul>
	 * 
	 * <strong>hierarchical</strong> (configurable via client configuration, admin-only):
	 * <ul>
	 * <li>Whether to include resources from child contexts of the current context</li>
	 * <li>Default value: false (child contexts not included)</li>
	 * <li>Usage: {@code client.setHierarchicalMode(true)}</li>
	 * <li>Values: true (includes child contexts) | false (current context only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#HIERARCHICAL_MODE_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> Only available to IS-Manager, Infrastructure-Manager, and Context-Manager roles.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Resources successfully retrieved</li>
	 * <li><strong>400 Bad Request</strong>: Invalid resource type, relation type, reference resource type, UUID format, or query parameters</li>
	 * <li><strong>403 Forbidden</strong>: Insufficient permissions to execute prepared queries or access hierarchical data</li>
	 * <li><strong>404 Not Found</strong>: Specified types do not exist in the model.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>Resource, relation, and reference resource classes are automatically converted to type names</li>
	 * <li>The reference resource UUID must be a valid UUID format</li>
	 * <li>Direction parameter controls the traversal direction of the IsRelatedTo relation.</li>
	 * </ul>
	 * 
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>Resources are returned from the current context based on authorization</li>
	 * <li>Hierarchical mode can be enabled for administrative users to query across context hierarchies</li>
	 * <li>Results respect context boundaries and user access permissions.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Full access to execute prepared queries with hierarchical querying across all contexts; receives complete, unfiltered metadata</li>
	 * <li><strong>Infrastructure-Manager</strong>: Full access to execute prepared queries with hierarchical querying across all contexts; receives complete, unfiltered metadata</li>
	 * <li><strong>All Other Users</strong>: Basic access with metadata filtering; receive metadata with sensitive information filtered when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryClient client = ResourceRegistryClientFactory.create();
	 * 
	 * // Find all EServices hosted by a HostingNode with specific UUID (incoming relation)
	 * UUID hostingNodeUUID = UUID.fromString("12345678-1234-1234-1234-123456789abc");
	 * List&lt;EService&gt; hostedServices = client.getRelatedResourcesFromReferenceResource(
	 *     EService.class, Hosts.class, HostingNode.class, hostingNodeUUID, Direction.IN, true);
	 * </pre>
	 * 
	 * @param <R> The specific Resource type to retrieve
	 * @param <I> The specific IsRelatedTo relation type connecting resources
	 * @param <RR> The specific reference Resource type
	 * @param resourceClass The class of resources to retrieve
	 * @param isRelatedToClass The class of the IsRelatedTo relation connecting resources
	 * @param referenceResourceClass The class of the reference resource
	 * @param referenceResourceUUID The UUID of the specific reference resource instance. This value is automatically 
	 *                              set in the {@code _reference} query parameter of the REST API call.
	 * @param direction The direction of relation traversal (OUT or IN). Values: OUT finds resources that the 
	 *                  reference resource relates to; IN finds resources that relate to the reference resource. 
	 *                  This parameter is automatically set in the {@code _direction} query parameter of the REST API call.
	 * @param polymorphic Whether to include subtypes in the results. When true, includes all subtypes 
	 *                    of the specified type; when false, returns only instances of the exact specified type. 
	 *                    This parameter is automatically set in the {@code _polymorphic} query parameter of the 
	 *                    REST API call.
	 * @return List of resources related to the specified reference resource
	 * @throws ResourceRegistryException If an error occurs during query execution
	 */
	public <R extends Resource, I extends IsRelatedTo<?, ?>, RR extends Resource> List<R> getRelatedResourcesFromReferenceResource(
			Class<R> resourceClass, Class<I> isRelatedToClass, Class<RR> referenceResourceClass,
			UUID referenceResourceUUID, Direction direction, boolean polymorphic) throws ResourceRegistryException;

	/**
	 * Gets resources related to a specific reference resource through an IsRelatedTo relation (string version).
	 * 
	 * <strong>Corresponding REST API:</strong> {@code GET /access/query/{resource-type-name}/{relation-type-name}/{reference-type-name}[?_reference={reference-uuid}&_polymorphic={true|false}&_direction={out|in}&limit={number}&offset={number}&includeMeta={true|false}&allMeta={true|false}&hierarchical={true|false}&includeContexts={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Retrieves all resources related to a reference resource with the specified UUID through the specified IsRelatedTo relation</li>
	 * <li>Returns JSON string representation of resources filtered based on user authorization</li>
	 * <li>Supports polymorphic queries to include subtypes and directional relationship traversal</li>
	 * <li>Results are filtered based on context access and user permissions.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>limit</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Maximum number of resources to return in a single response</li>
	 * <li>Default value: server configuration dependent</li>
	 * <li>Usage: {@code client.setLimit(50)}</li>
	 * <li>Example: Set to 50 to get at most 50 resources per request</li>
	 * <li>Unlimited results: Set to {@code RequestInfo.UNBOUNDED_LIMIT} (-1) for unlimited results (may cause timeout if results are too many)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#LIMIT_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>offset</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Number of resources to skip from the beginning of the result set</li>
	 * <li>Default value: 0</li>
	 * <li>Usage: {@code client.setOffset(10)}</li>
	 * <li>Example: Set to 10 to skip the first 10 resources (useful for pagination)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#OFFSET_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response resources</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code client.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata for all nested elements (ConsistsOf relations, Facets, etc.) in the response</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main resources, more human-readable)</li>
	 * <li>Usage: {@code client.setAllMeta(true)}</li>
	 * <li>Values: true (complete metadata) | false (main resources only, more readable)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include the list of contexts where the resources and their nested elements are available in the response</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Usage: {@code client.setIncludeContexts(true)}</li>
	 * <li>Values: true (shows context availability) | false (no context information)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#INCLUDE_CONTEXTS_QUERY_PARAMETER}</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present</li>
	 * <li><strong>Note:</strong> A Facet's context availability depends on the Resources that include it via ConsistsOf relations.</li>
	 * </ul>
	 * 
	 * <strong>hierarchical</strong> (configurable via client configuration, admin-only):
	 * <ul>
	 * <li>Whether to include resources from child contexts of the current context</li>
	 * <li>Default value: false (child contexts not included)</li>
	 * <li>Usage: {@code client.setHierarchicalMode(true)}</li>
	 * <li>Values: true (includes child contexts) | false (current context only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#HIERARCHICAL_MODE_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> Only available to IS-Manager, Infrastructure-Manager, and Context-Manager roles.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Resources successfully retrieved</li>
	 * <li><strong>400 Bad Request</strong>: Invalid resource type, relation type, reference resource type, UUID format, or query parameters</li>
	 * <li><strong>403 Forbidden</strong>: Insufficient permissions to execute prepared queries or access hierarchical data</li>
	 * <li><strong>404 Not Found</strong>: Specified types do not exist in the model.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>Resource, relation, and reference resource type names must be valid Information System model types</li>
	 * <li>The reference resource UUID must be a valid UUID format</li>
	 * <li>Direction parameter controls the traversal direction of the IsRelatedTo relation.</li>
	 * </ul>
	 * 
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>Resources are returned from the current context based on authorization</li>
	 * <li>Hierarchical mode can be enabled for administrative users to query across context hierarchies</li>
	 * <li>Results respect context boundaries and user access permissions.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Full access to execute prepared queries with hierarchical querying across all contexts; receives complete, unfiltered metadata</li>
	 * <li><strong>Infrastructure-Manager</strong>: Full access to execute prepared queries with hierarchical querying across all contexts; receives complete, unfiltered metadata</li>
	 * <li><strong>All Other Users</strong>: Basic access with metadata filtering; receive metadata with sensitive information filtered when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryClient client = ResourceRegistryClientFactory.create();
	 * 
	 * // Find all EServices hosted by a HostingNode with specific UUID (incoming relation)
	 * UUID hostingNodeUUID = UUID.fromString("12345678-1234-1234-1234-123456789abc");
	 * String hostedServicesJson = client.getRelatedResourcesFromReferenceResource(
	 *     "EService", "Hosts", "HostingNode", hostingNodeUUID, Direction.IN, true);
	 * </pre>
	 * 
	 * @param resourceType The name of the resource type to retrieve
	 * @param isRelatedToType The name of the IsRelatedTo relation type connecting resources
	 * @param referenceResourceType The name of the reference resource type
	 * @param referenceResourceUUID The UUID of the specific reference resource instance. This value is automatically 
	 *                              set in the {@code _reference} query parameter of the REST API call.
	 * @param direction The direction of relation traversal (OUT or IN). Values: OUT finds resources that the 
	 *                  reference resource relates to; IN finds resources that relate to the reference resource. 
	 *                  This parameter is automatically set in the {@code _direction} query parameter of the REST API call.
	 * @param polymorphic Whether to include subtypes in the results. When true, includes all subtypes 
	 *                    of the specified type; when false, returns only instances of the exact specified type. 
	 *                    This parameter is automatically set in the {@code _polymorphic} query parameter of the 
	 *                    REST API call.
	 * @return JSON string representation of resources related to the specified reference resource
	 * @throws ResourceRegistryException If an error occurs during query execution
	 */
	public String getRelatedResourcesFromReferenceResource(String resourceType, String isRelatedToType,
			String referenceResourceType, UUID referenceResourceUUID, Direction direction, boolean polymorphic)
			throws ResourceRegistryException;

	/**
	 * Gets all resources related to any resource of a specific type through an IsRelatedTo relation (typed version).
	 * 
	 * <strong>Corresponding REST API:</strong> {@code GET /access/query/{resource-type-name}/{relation-type-name}/{reference-type-name}[?_polymorphic={true|false}&_direction={out|in}&limit={number}&offset={number}&includeMeta={true|false}&allMeta={true|false}&hierarchical={true|false}&includeContexts={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Retrieves all resources related to any resource of the specified reference type through the specified IsRelatedTo relation</li>
	 * <li>Returns strongly-typed resource objects filtered based on user authorization</li>
	 * <li>Supports polymorphic queries to include subtypes and directional relationship traversal</li>
	 * <li>Does not filter by specific reference resource instance (returns all related resources).</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>_reference</strong> (not used):
	 * <ul>
	 * <li>No specific reference resource UUID is specified</li>
	 * <li>Returns resources related to ANY resource of the specified reference type.</li>
	 * </ul>
	 * 
	 * <strong>limit</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Maximum number of resources to return in a single response</li>
	 * <li>Default value: server configuration dependent</li>
	 * <li>Usage: {@code client.setLimit(50)}</li>
	 * <li>Example: Set to 50 to get at most 50 resources per request</li>
	 * <li>Unlimited results: Set to {@code RequestInfo.UNBOUNDED_LIMIT} (-1) for unlimited results (may cause timeout if results are too many)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#LIMIT_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>offset</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Number of resources to skip from the beginning of the result set</li>
	 * <li>Default value: 0</li>
	 * <li>Usage: {@code client.setOffset(10)}</li>
	 * <li>Example: Set to 10 to skip the first 10 resources (useful for pagination)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#OFFSET_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response resources</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code client.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata for all nested elements (ConsistsOf relations, Facets, etc.) in the response</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main resources, more human-readable)</li>
	 * <li>Usage: {@code client.setAllMeta(true)}</li>
	 * <li>Values: true (complete metadata) | false (main resources only, more readable)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include the list of contexts where the resources and their nested elements are available in the response</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Usage: {@code client.setIncludeContexts(true)}</li>
	 * <li>Values: true (shows context availability) | false (no context information)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#INCLUDE_CONTEXTS_QUERY_PARAMETER}</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present</li>
	 * <li><strong>Note:</strong> A Facet's context availability depends on the Resources that include it via ConsistsOf relations.</li>
	 * </ul>
	 * 
	 * <strong>hierarchical</strong> (configurable via client configuration, admin-only):
	 * <ul>
	 * <li>Whether to include resources from child contexts of the current context</li>
	 * <li>Default value: false (child contexts not included)</li>
	 * <li>Usage: {@code client.setHierarchicalMode(true)}</li>
	 * <li>Values: true (includes child contexts) | false (current context only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#HIERARCHICAL_MODE_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> Only available to IS-Manager, Infrastructure-Manager, and Context-Manager roles.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Resources successfully retrieved</li>
	 * <li><strong>400 Bad Request</strong>: Invalid resource type, relation type, reference resource type, or query parameters</li>
	 * <li><strong>403 Forbidden</strong>: Insufficient permissions to execute prepared queries or access hierarchical data</li>
	 * <li><strong>404 Not Found</strong>: Specified types do not exist in the model.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>Resource, relation, and reference resource classes are automatically converted to type names</li>
	 * <li>Direction parameter controls the traversal direction of the IsRelatedTo relation</li>
	 * <li>Polymorphic parameter controls whether subtypes are included in results.</li>
	 * </ul>
	 * 
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>Resources are returned from the current context based on authorization</li>
	 * <li>Hierarchical mode can be enabled for administrative users to query across context hierarchies</li>
	 * <li>Results respect context boundaries and user access permissions.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Full access to execute prepared queries with hierarchical querying across all contexts; receives complete, unfiltered metadata</li>
	 * <li><strong>Infrastructure-Manager</strong>: Full access to execute prepared queries with hierarchical querying across all contexts; receives complete, unfiltered metadata</li>
	 * <li><strong>All Other Users</strong>: Basic access with metadata filtering; receive metadata with sensitive information filtered when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryClient client = ResourceRegistryClientFactory.create();
	 * 
	 * // Find all EServices hosted by any HostingNode (incoming relation)
	 * List&lt;EService&gt; allHostedServices = client.getRelatedResources(
	 *     EService.class, Hosts.class, HostingNode.class, Direction.IN, true);
	 * 
	 * // Find all HostingNodes that host any EService (outgoing relation)
	 * List&lt;HostingNode&gt; allHostingNodes = client.getRelatedResources(
	 *     HostingNode.class, Hosts.class, EService.class, Direction.OUT, true);
	 * </pre>
	 * 
	 * @param <R> The specific Resource type to retrieve
	 * @param <I> The specific IsRelatedTo relation type connecting resources
	 * @param <RR> The specific reference Resource type
	 * @param resourceClass The class of resources to retrieve
	 * @param isRelatedToClass The class of the IsRelatedTo relation connecting resources
	 * @param referenceResourceClass The class of the reference resource type
	 * @param direction The direction of relation traversal (OUT or IN). Values: OUT finds resources that any 
	 *                  reference resource relates to; IN finds resources that relate to any reference resource. 
	 *                  This parameter is automatically set in the {@code _direction} query parameter of the REST API call.
	 * @param polymorphic Whether to include subtypes in the results. When true, includes all subtypes 
	 *                    of the specified type; when false, returns only instances of the exact specified type. 
	 *                    This parameter is automatically set in the {@code _polymorphic} query parameter of the 
	 *                    REST API call.
	 * @return List of resources related to any resource of the specified reference type
	 * @throws ResourceRegistryException If an error occurs during query execution
	 */
	public <R extends Resource, I extends IsRelatedTo<?, ?>, RR extends Resource> List<R> getRelatedResources(
			Class<R> resourceClass, Class<I> isRelatedToClass, Class<RR> referenceResourceClass, Direction direction,
			boolean polymorphic) throws ResourceRegistryException;

	/**
	 * Gets all resources related to any resource of a specific type through an IsRelatedTo relation (string version).
	 * 
	 * <strong>Corresponding REST API:</strong> {@code GET /access/query/{resource-type-name}/{relation-type-name}/{reference-type-name}[?_polymorphic={true|false}&_direction={out|in}&limit={number}&offset={number}&includeMeta={true|false}&allMeta={true|false}&hierarchical={true|false}&includeContexts={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Retrieves all resources related to any resource of the specified reference type through the specified IsRelatedTo relation</li>
	 * <li>Returns JSON string representation of resources filtered based on user authorization</li>
	 * <li>Supports polymorphic queries to include subtypes and directional relationship traversal</li>
	 * <li>Does not filter by specific reference resource instance (returns all related resources).</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>_reference</strong> (not used):
	 * <ul>
	 * <li>No specific reference resource UUID is specified</li>
	 * <li>Returns resources related to ANY resource of the specified reference type.</li>
	 * </ul>
	 * 
	 * <strong>limit</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Maximum number of resources to return in a single response</li>
	 * <li>Default value: server configuration dependent</li>
	 * <li>Usage: {@code client.setLimit(50)}</li>
	 * <li>Example: Set to 50 to get at most 50 resources per request</li>
	 * <li>Unlimited results: Set to {@code RequestInfo.UNBOUNDED_LIMIT} (-1) for unlimited results (may cause timeout if results are too many)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#LIMIT_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>offset</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Number of resources to skip from the beginning of the result set</li>
	 * <li>Default value: 0</li>
	 * <li>Usage: {@code client.setOffset(10)}</li>
	 * <li>Example: Set to 10 to skip the first 10 resources (useful for pagination)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#OFFSET_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response resources</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code client.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata for all nested elements (ConsistsOf relations, Facets, etc.) in the response</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main resources, more human-readable)</li>
	 * <li>Usage: {@code client.setAllMeta(true)}</li>
	 * <li>Values: true (complete metadata) | false (main resources only, more readable)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include the list of contexts where the resources and their nested elements are available in the response</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Usage: {@code client.setIncludeContexts(true)}</li>
	 * <li>Values: true (shows context availability) | false (no context information)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#INCLUDE_CONTEXTS_QUERY_PARAMETER}</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present</li>
	 * <li><strong>Note:</strong> A Facet's context availability depends on the Resources that include it via ConsistsOf relations.</li>
	 * </ul>
	 * 
	 * <strong>hierarchical</strong> (configurable via client configuration, admin-only):
	 * <ul>
	 * <li>Whether to include resources from child contexts of the current context</li>
	 * <li>Default value: false (child contexts not included)</li>
	 * <li>Usage: {@code client.setHierarchicalMode(true)}</li>
	 * <li>Values: true (includes child contexts) | false (current context only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#HIERARCHICAL_MODE_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> Only available to IS-Manager, Infrastructure-Manager, and Context-Manager roles.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Resources successfully retrieved</li>
	 * <li><strong>400 Bad Request</strong>: Invalid resource type, relation type, reference resource type, or query parameters</li>
	 * <li><strong>403 Forbidden</strong>: Insufficient permissions to execute prepared queries or access hierarchical data</li>
	 * <li><strong>404 Not Found</strong>: Specified types do not exist in the model.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>Resource, relation, and reference resource type names must be valid Information System model types</li>
	 * <li>Direction parameter controls the traversal direction of the IsRelatedTo relation</li>
	 * <li>Polymorphic parameter controls whether subtypes are included in results.</li>
	 * </ul>
	 * 
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>Resources are returned from the current context based on authorization</li>
	 * <li>Hierarchical mode can be enabled for administrative users to query across context hierarchies</li>
	 * <li>Results respect context boundaries and user access permissions.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Full access to execute prepared queries with hierarchical querying across all contexts; receives complete, unfiltered metadata</li>
	 * <li><strong>Infrastructure-Manager</strong>: Full access to execute prepared queries with hierarchical querying across all contexts; receives complete, unfiltered metadata</li>
	 * <li><strong>All Other Users</strong>: Basic access with metadata filtering; receive metadata with sensitive information filtered when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryClient client = ResourceRegistryClientFactory.create();
	 * 
	 * // Find all EServices hosted by any HostingNode (incoming relation)
	 * String allHostedServicesJson = client.getRelatedResources(
	 *     "EService", "Hosts", "HostingNode", Direction.IN, true);
	 * </pre>
	 * 
	 * @param resourceType The name of the resource type to retrieve
	 * @param isRelatedToType The name of the IsRelatedTo relation type connecting resources
	 * @param referenceResourceType The name of the reference resource type
	 * @param direction The direction of relation traversal (OUT or IN). Values: OUT finds resources that any 
	 *                  reference resource relates to; IN finds resources that relate to any reference resource. 
	 *                  This parameter is automatically set in the {@code _direction} query parameter of the REST API call.
	 * @param polymorphic Whether to include subtypes in the results. When true, includes all subtypes 
	 *                    of the specified type; when false, returns only instances of the exact specified type. 
	 *                    This parameter is automatically set in the {@code _polymorphic} query parameter of the 
	 *                    REST API call.
	 * @return JSON string representation of resources related to any resource of the specified reference type
	 * @throws ResourceRegistryException If an error occurs during query execution
	 */
	public String getRelatedResources(String resourceType, String isRelatedToType, String referenceResourceType,
			Direction direction, boolean polymorphic) throws ResourceRegistryException;

	
	/* ------------------------- JSON Queries APIs -------------------------- */

	/**
	 * Executes a JSON-based query to find instances matching the specified structure in the Resource Registry.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code POST /access/query[?polymorphic={true|false}&includeMeta={true|false}&allMeta={true|false}&includeContexts={true|false}&hierarchical={true|false}&limit={number}&offset={number}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Executes a JSON-based structural query to find instances matching the specified criteria</li>
	 * <li>Returns JSON string representation of matching instances</li>
	 * <li>Supports complex queries including relations, properties, and nested structures</li>
	 * <li>Results are filtered based on user authorization and context access.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response instances</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code client.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata; other users see filtered metadata.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata for all nested instances</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main instances)</li>
	 * <li>Usage: {@code client.setAllMeta(true)}</li>
	 * <li>Values: true (includes metadata for all nested elements) | false (metadata only for main instances)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include the list of contexts where each instance is available</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Usage: {@code client.setIncludeContexts(true)}</li>
	 * <li>Values: true (shows context availability for each instance) | false (no context information)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#INCLUDE_CONTEXTS_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>hierarchical</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include instances from child contexts of the current context</li>
	 * <li>Default value: false (current context only)</li>
	 * <li>Usage: {@code client.setHierarchicalMode(true)}</li>
	 * <li>Restriction: Only available to IS-Manager, Infrastructure-Manager, and Context-Manager roles</li>
	 * <li>Values: true (includes instances from current and child contexts) | false (current context only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#HIERARCHICAL_MODE_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>limit</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Maximum number of instances to return</li>
	 * <li>Default value: server configuration dependent</li>
	 * <li>Usage: {@code client.setLimit(50)}</li>
	 * <li>Example: Set to 50 to get at most 50 instances per request</li>
	 * <li>Unlimited results: Set to {@code RequestInfo.UNBOUNDED_LIMIT} (-1) for unlimited results (may cause timeout if results are too many)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#LIMIT_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>offset</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Number of instances to skip from the beginning of the result set</li>
	 * <li>Default value: 0 (starts from the first instance)</li>
	 * <li>Usage: {@code client.setOffset(100)}</li>
	 * <li>Example: Set to 100 to skip the first 100 instances (useful for pagination)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#OFFSET_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: JSON query successfully executed</li>
	 * <li><strong>400 Bad Request</strong>: Invalid query structure, syntax errors, or invalid parameters</li>
	 * <li><strong>403 Forbidden</strong>: Insufficient permissions to execute JSON queries or access hierarchical data</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>Query must be a valid JSON string describing the structure to search for</li>
	 * <li>Must specify the target type and optionally its relations and properties</li>
	 * <li>Relations can include ConsistsOf, IsRelatedTo, and other defined relation types</li>
	 * <li>Properties can specify exact values or patterns to match.</li>
	 * </ul>
	 * 
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>Query execution respects context boundaries based on user authorization</li>
	 * <li>Hierarchical mode can be enabled for administrative users to query across context hierarchies</li>
	 * <li>Results are filtered based on current context and user access permissions.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Full access to execute JSON queries with hierarchical querying across all contexts; receives complete, unfiltered metadata</li>
	 * <li><strong>Infrastructure-Manager</strong>: Full access to execute JSON queries with hierarchical querying across all contexts; receives complete, unfiltered metadata</li>
	 * <li><strong>Context-Manager</strong>: Access to execute JSON queries with hierarchical querying within managed contexts; receives complete, unfiltered metadata</li>
	 * <li><strong>All Other Users</strong>: Basic access with metadata filtering; receive metadata with sensitive information filtered when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryClient client = ResourceRegistryClientFactory.create();
	 * 
	 * // JSON query to find EServices with specific state and software
	 * String jsonQuery = """
	 *     {
	 *       "type": "EService",
	 *       "consistsOf": [
	 *         {
	 *           "type": "ConsistsOf",
	 *           "target": {
	 *             "type": "StateFacet",
	 *             "value": "down"
	 *           }
	 *         },
	 *         {
	 *           "type": "IsIdentifiedBy",
	 *           "target": {
	 *             "type": "SoftwareFacet",
	 *             "name": "data-transfer-service"
	 *           }
	 *         }
	 *       ]
	 *     }
	 *     """;
	 * 
	 * String resultsJson = client.jsonQuery(jsonQuery);
	 * </pre>
	 * 
	 * @param query JSON string describing the structure to search for
	 * @return JSON string containing query results
	 * @throws InvalidQueryException If the query structure is invalid or contains syntax errors
	 * @throws ResourceRegistryException If an error occurs during query execution
	 */
	public String jsonQuery(final String query)
			throws InvalidQueryException, ResourceRegistryException;

	/**
	 * Executes a JSON-based query to find instances matching the specified structure and returns typed results.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code POST /access/query[?polymorphic={true|false}&includeMeta={true|false}&allMeta={true|false}&includeContexts={true|false}&hierarchical={true|false}&limit={number}&offset={number}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Executes a JSON-based structural query to find instances matching the specified criteria</li>
	 * <li>Returns strongly-typed list of Entity objects automatically deserialized from JSON results</li>
	 * <li>Supports complex queries including relations, properties, and nested structures</li>
	 * <li>Results are filtered based on user authorization and context access.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response instances</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Usage: {@code client.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata; other users see filtered metadata.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata for all nested instances</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main instances)</li>
	 * <li>Usage: {@code client.setAllMeta(true)}</li>
	 * <li>Values: true (includes metadata for all nested elements) | false (metadata only for main instances)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#INCLUDE_META_IN_ALL_INSTANCES_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include the list of contexts where each instance is available</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Usage: {@code client.setIncludeContexts(true)}</li>
	 * <li>Values: true (shows context availability for each instance) | false (no context information)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#INCLUDE_CONTEXTS_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>hierarchical</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include instances from child contexts of the current context</li>
	 * <li>Default value: false (current context only)</li>
	 * <li>Usage: {@code client.setHierarchicalMode(true)}</li>
	 * <li>Restriction: Only available to IS-Manager, Infrastructure-Manager, and Context-Manager roles</li>
	 * <li>Values: true (includes instances from current and child contexts) | false (current context only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#HIERARCHICAL_MODE_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>limit</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Maximum number of instances to return</li>
	 * <li>Default value: server configuration dependent</li>
	 * <li>Usage: {@code client.setLimit(50)}</li>
	 * <li>Example: Set to 50 to get at most 50 instances per request</li>
	 * <li>Unlimited results: Set to {@code RequestInfo.UNBOUNDED_LIMIT} (-1) for unlimited results (may cause timeout if results are too many)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#LIMIT_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>offset</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Number of instances to skip from the beginning of the result set</li>
	 * <li>Default value: 0 (starts from the first instance)</li>
	 * <li>Usage: {@code client.setOffset(100)}</li>
	 * <li>Example: Set to 100 to skip the first 100 instances (useful for pagination)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#OFFSET_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: JSON query successfully executed</li>
	 * <li><strong>400 Bad Request</strong>: Invalid query structure, syntax errors, or invalid parameters</li>
	 * <li><strong>403 Forbidden</strong>: Insufficient permissions to execute JSON queries or access hierarchical data</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>JsonNode must represent a valid JSON structure describing the search criteria</li>
	 * <li>Must specify the target type and optionally its relations and properties</li>
	 * <li>Relations can include ConsistsOf, IsRelatedTo, and other defined relation types</li>
	 * <li>Properties can specify exact values or patterns to match.</li>
	 * </ul>
	 * 
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>Query execution respects context boundaries based on user authorization</li>
	 * <li>Hierarchical mode can be enabled for administrative users to query across context hierarchies</li>
	 * <li>Results are filtered based on current context and user access permissions.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Full access to execute JSON queries with hierarchical querying across all contexts; receives complete, unfiltered metadata</li>
	 * <li><strong>Infrastructure-Manager</strong>: Full access to execute JSON queries with hierarchical querying across all contexts; receives complete, unfiltered metadata</li>
	 * <li><strong>Context-Manager</strong>: Access to execute JSON queries with hierarchical querying within managed contexts; receives complete, unfiltered metadata</li>
	 * <li><strong>All Other Users</strong>: Basic access with metadata filtering; receive metadata with sensitive information filtered when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryClient client = ResourceRegistryClientFactory.create();
	 * 
	 * // Create JsonNode query to find EServices with specific software
	 * ObjectMapper mapper = new ObjectMapper();
	 * JsonNode queryNode = mapper.createObjectNode()
	 *     .put("type", "EService")
	 *     .set("consistsOf", mapper.createArrayNode()
	 *         .add(mapper.createObjectNode()
	 *             .put("type", "IsIdentifiedBy")
	 *             .set("target", mapper.createObjectNode()
	 *                 .put("type", "SoftwareFacet")
	 *                 .put("name", "data-transfer-service"))));
	 * 
	 * List&lt;EService&gt; services = client.jsonQuery(queryNode);
	 * </pre>
	 * 
	 * @param <E> The specific Entity type expected in results
	 * @param jsonNode JsonNode representing the query structure
	 * @return List of Entity instances matching the query criteria
	 * @throws InvalidQueryException If the query structure is invalid or contains syntax errors
	 * @throws ResourceRegistryException If an error occurs during query execution
	 */
	public <E extends Entity> List<E> jsonQuery(final JsonNode jsonNode)
			throws InvalidQueryException, ResourceRegistryException;
	
	
	/* -------------------------- Raw Queries APIs -------------------------- */
	
	/**
	 * Executes a raw OrientDB SQL query against the Resource Registry backend.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code GET /access/query[?q={query}&raw={false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Executes a direct OrientDB SQL query against the backend database</li>
	 * <li>Returns results formatted according to the Information System model</li>
	 * <li>Only idempotent queries (SELECT statements) are accepted for security</li>
	 * <li>Results are filtered based on user authorization and context access.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>q</strong> (automatically configured from query parameter):
	 * <ul>
	 * <li>The OrientDB SQL query to execute against the backend</li>
	 * <li>Must be URL-encoded for special characters</li>
	 * <li>Only idempotent queries are accepted (SELECT statements)</li>
	 * <li>Example: "SELECT FROM EService WHERE out('ConsistsOf').@class = 'StateFacet'"</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#Q_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>raw</strong> (automatically set to false):
	 * <ul>
	 * <li>Whether to return the raw backend response or format it according to the Information System model</li>
	 * <li>Fixed value: false (format results according to IS model)</li>
	 * <li>Results are converted to standard Information System model format with proper typing and structure</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#RAW_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Graph query successfully executed</li>
	 * <li><strong>400 Bad Request</strong>: Invalid query syntax or non-idempotent query attempted</li>
	 * <li><strong>403 Forbidden</strong>: Insufficient permissions to execute graph queries (only IS-Manager and Infrastructure-Manager allowed)</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>Query must be a valid OrientDB SQL statement</li>
	 * <li>Must be idempotent (SELECT-only operations)</li>
	 * <li>Query is automatically URL-encoded before transmission</li>
	 * <li>Only SELECT operations are allowed for security reasons.</li>
	 * </ul>
	 * 
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>Query execution respects context boundaries based on user authorization</li>
	 * <li>Results are filtered based on current context and user access permissions</li>
	 * <li>Administrative users may have broader query capabilities.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Full access to execute raw queries with complete, unfiltered results</li>
	 * <li><strong>Infrastructure-Manager</strong>: Full access to execute raw queries with complete, unfiltered results</li>
	 * <li><strong>All Other Users</strong>: Limited or no access to raw query functionality (implementation dependent).</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryClient client = ResourceRegistryClientFactory.create();
	 * 
	 * // Execute a raw OrientDB query to find EServices with StateFacets
	 * String query = "SELECT FROM EService WHERE out('ConsistsOf').@class = 'StateFacet'";
	 * String results = client.rawQuery(query);
	 * </pre>
	 * 
	 * @param query The OrientDB SQL query to execute against the backend
	 * @return JSON string containing formatted query results according to Information System model
	 * @throws InvalidQueryException If the query is invalid, non-idempotent, or contains syntax errors
	 * @throws ResourceRegistryException If an error occurs during query execution
	 */
	public String rawQuery(final String query)
			throws InvalidQueryException, ResourceRegistryException;

	/**
	 * Executes a raw OrientDB SQL query against the Resource Registry backend with configurable result formatting.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code GET /access/query[?q={query}&raw={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Executes a direct OrientDB SQL query against the backend database</li>
	 * <li>Returns results either as raw OrientDB format or formatted according to the Information System model</li>
	 * <li>Only idempotent queries (SELECT statements) are accepted for security</li>
	 * <li>Results are filtered based on user authorization and context access.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>q</strong> (automatically configured from query parameter):
	 * <ul>
	 * <li>The OrientDB SQL query to execute against the backend</li>
	 * <li>Must be URL-encoded for special characters</li>
	 * <li>Only idempotent queries are accepted (SELECT statements)</li>
	 * <li>Example: "SELECT FROM EService WHERE out('ConsistsOf').@class = 'StateFacet'"</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#Q_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>raw</strong> (automatically configured from raw parameter):
	 * <ul>
	 * <li>Whether to return the raw backend response or format it according to the Information System model</li>
	 * <li>Configured by method parameter value</li>
	 * <li>Values: true (returns raw OrientDB response format) | false (format results according to IS model)</li>
	 * <li>When false: Results are converted to standard Information System model format with proper typing and structure</li>
	 * <li>When true: Results are returned exactly as provided by the OrientDB backend without any formatting or conversion</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.AccessPath#RAW_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Graph query successfully executed</li>
	 * <li><strong>400 Bad Request</strong>: Invalid query syntax or non-idempotent query attempted</li>
	 * <li><strong>403 Forbidden</strong>: Insufficient permissions to execute graph queries (only IS-Manager and Infrastructure-Manager allowed)</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>Query must be a valid OrientDB SQL statement</li>
	 * <li>Must be idempotent (SELECT-only operations)</li>
	 * <li>Query is automatically URL-encoded before transmission</li>
	 * <li>Raw parameter controls result formatting.</li>
	 * </ul>
	 * 
	 * <strong>Context Access:</strong>
	 * <ul>
	 * <li>Query execution respects context boundaries based on user authorization</li>
	 * <li>Results are filtered based on current context and user access permissions</li>
	 * <li>Administrative users may have broader query capabilities and access to raw data.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Full access to execute raw queries with complete, unfiltered results; can request raw backend data</li>
	 * <li><strong>Infrastructure-Manager</strong>: Full access to execute raw queries with complete, unfiltered results; can request raw backend data</li>
	 * <li><strong>All Other Users</strong>: Limited or no access to raw query functionality (implementation dependent).</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryClient client = ResourceRegistryClientFactory.create();
	 * 
	 * // Execute query with formatted results (IS model)
	 * String query = "SELECT FROM EService WHERE out('ConsistsOf').@class = 'StateFacet'";
	 * String formattedResults = client.rawQuery(query, false);
	 * 
	 * // Execute query with raw OrientDB results
	 * String rawResults = client.rawQuery(query, true);
	 * </pre>
	 * 
	 * @param query The OrientDB SQL query to execute against the backend
	 * @param raw Whether to return raw backend response (true) or format according to IS model (false)
	 * @return JSON string containing query results (raw OrientDB format or IS model format)
	 * @throws InvalidQueryException If the query is invalid, non-idempotent, or contains syntax errors
	 * @throws ResourceRegistryException If an error occurs during query execution
	 */
	public String rawQuery(final String query, boolean raw)
			throws InvalidQueryException, ResourceRegistryException;

}
