package org.gcube.indexmanagement.storagehandling;

import java.rmi.RemoteException;

import org.gcube.common.core.contexts.GCUBEServiceContext;
import org.gcube.common.core.faults.GCUBEUnrecoverableFault;
import org.gcube.common.core.porttypes.GCUBEPortType;
import org.gcube.common.core.utils.logging.GCUBELog;
import org.gcube.indexmanagement.common.IndexException;
import org.gcube.indexmanagement.storagehandling.stubs.ConnectLookup;
import org.gcube.indexmanagement.storagehandling.stubs.ConnectUpdater;
import org.gcube.indexmanagement.storagehandling.stubs.ConnectUpdaterResponse;
import org.gcube.indexmanagement.storagehandling.stubs.DeltaFileInfoType;
import org.gcube.indexmanagement.storagehandling.stubs.DisconnectUpdaterResponse;
import org.gcube.indexmanagement.storagehandling.stubs.GetDeltaCollectionID;
import org.gcube.indexmanagement.storagehandling.stubs.GetDeltaFileList;
import org.gcube.indexmanagement.storagehandling.stubs.GetDeltaFileListResponse;
import org.gcube.indexmanagement.storagehandling.stubs.IsUpdating;
import org.gcube.indexmanagement.storagehandling.stubs.MergeResponse;
import org.globus.wsrf.ResourceContext;
import org.globus.wsrf.Topic;
import org.globus.wsrf.TopicList;
import org.globus.wsrf.TopicListAccessor;
import org.globus.wsrf.impl.notification.GetCurrentMessageProvider;
import org.oasis.wsn.GetCurrentMessage;
import org.oasis.wsn.GetCurrentMessageResponse;
import org.oasis.wsn.InvalidTopicExpressionFaultType;
import org.oasis.wsn.NoCurrentMessageOnTopicFaultType;
import org.oasis.wsn.ResourceUnknownFaultType;
import org.oasis.wsn.TopicNotSupportedFaultType;


public class DeltaListManagementProvider extends GCUBEPortType {

	/** logger */
	static GCUBELog logger = new GCUBELog(DeltaListManagementProvider.class);

	/** The "get current message" provider used by the notification producer */ 
	private GetCurrentMessageProvider gcmProvider = new GetCurrentMessageProvider();
	
	/**
	 * Constructs a new DeltaListManagementProvider
	 */
	public DeltaListManagementProvider(){
	}

	/**
	 * Returns a list of all retrieved delta files
	 * @param request the request
	 * @return GetDeltaFileListResponse true if updating
	 * @throws GCUBEUnrecoverableFault an error occured
	 */ //should use enum in the future, and not respond true before update
	public GetDeltaFileListResponse getDeltaFileList(GetDeltaFileList request) throws GCUBEUnrecoverableFault {
		try {
			DeltaListManagementHandler handler = getHandler();
			handler.setIndexResourceScopeToDeltaListManagementScope();
			GetDeltaFileListResponse response = new GetDeltaFileListResponse();
			response.setDeltaFiles(handler.getDeltaFileList());
			return response;
		} catch (Exception e) {
			logger.error("Failed to get delta file list.", e);
			throw new GCUBEUnrecoverableFault(e.getMessage());
		}
	}

	/**
	 * Returns a delta file's particulars based on its index (placement in the DeltaFileList)
	 * @param deltaFileIdx The index of the DeltaFile in question
	 * @return DeltaFileInfoType   The particulars of the DeltaFile in question              
	 * @throws GCUBEUnrecoverableFault  an error occured
	 */ //should use enum in the future, and not respond true before update
	public DeltaFileInfoType getDeltaFileInfo(int deltaFileIdx) throws GCUBEUnrecoverableFault {
		try {
			DeltaListManagementHandler handler = getHandler();
			handler.setIndexResourceScopeToDeltaListManagementScope();
			return handler.getDeltaFileInfo(deltaFileIdx);
		} catch (Exception e) {
			logger.error("Failed to get delta file info.", e);
			throw new GCUBEUnrecoverableFault(e.getMessage());
		}
	}
	
	/**
	 * Returns the collection ID of the collection that stores the 
	 * documents related to deltafiles
	 * @return {@link String} the deltafile collection ID
	 */ //should use enum in the future, and not respond true before update
	public String getDeltaCollectionID(GetDeltaCollectionID empty) throws GCUBEUnrecoverableFault {
		try {
			DeltaListManagementHandler handler = getHandler();
			handler.setIndexResourceScopeToDeltaListManagementScope();
			return handler.getDeltaFileCollectionID();
		} catch (Exception e) {
			logger.error("Failed to get delta file info.", e);
			throw new GCUBEUnrecoverableFault(e.getMessage());
		}
	}


	/**
	 * Merges rowset information into an index.
	 * @param deltaFileInfo the delta file to merge
	 * @return MergeResponse empty stub response
	 * @throws GCUBEUnrecoverableFault an error occured
	 */

	public MergeResponse mergeDeltaFile(DeltaFileInfoType deltaFileInfo) throws GCUBEUnrecoverableFault {
		try{
			DeltaListManagementHandler handler = getHandler();
			handler.setIndexResourceScopeToDeltaListManagementScope();
			
			if(getManagerResource().isDeltaValid(deltaFileInfo)){
				handler.mergeDeltaFile(deltaFileInfo, true);
			}
			else{
				throw new IndexException("The delta file is not valid");
			}

			logger.info("Added deltaFile [" + deltaFileInfo.getDeltaFileID() + "] containing " + deltaFileInfo.getDocumentCount() + 
			" documents to the delta file list.");
		}
		catch(Exception e){
			logger.error("Unable to merge deltaFile: " + deltaFileInfo.getDeltaFileID(), e);
			throw new GCUBEUnrecoverableFault(e.getMessage());
		}

		return new MergeResponse();
	}

	/**
	 * Creates a connection number to identify an Index replication (Lookup resource)
	 * @param empty empty stub request
	 * @return int the new connection ID     
	 * @throws GCUBEUnrecoverableFault an error occured
	 */
	public int connectLookup(ConnectLookup empty) throws GCUBEUnrecoverableFault {
		try{
			DeltaListManagementHandler handler = getHandler();
			handler.setIndexResourceScopeToDeltaListManagementScope();
			return handler.connectLookup(); 
		}
		catch(Exception ie){
			logger.error("Unable to connect lookup to DeltaListManagementProvider.", ie);
			throw new GCUBEUnrecoverableFault(ie.getMessage());
		}
	}

	/**
	 * Informs the index that an update has started.
	 * @param empty empty stub request 
	 * @return ConnectUpdaterResponse  the connectionID of this connection and the collectionID to upload deltaFiles to     
	 * @throws GCUBEUnrecoverableFault an error occured
	 */
	public ConnectUpdaterResponse connectUpdater(ConnectUpdater empty) throws GCUBEUnrecoverableFault {
		try{
			DeltaListManagementHandler handler = getHandler();
			handler.setIndexResourceScopeToDeltaListManagementScope();
			ConnectUpdaterResponse response = new ConnectUpdaterResponse();
			response.setConnectionID(handler.connectUpdater());
			response.setDeltaFileCollectionID(handler.getDeltaFileCollectionID());
			return response;
		}
		catch(Exception ie){
			logger.error("Unable to connect updater to DeltaListManagementProvider.", ie);
			throw new GCUBEUnrecoverableFault(ie.getMessage());
		}
	}

	/**
	 * Informs the index that an update is finished.
	 * @param updaterID ID of the finished updater resource
	 * @return DisconnectUpdaterResponse empty stub response
	 * @throws GCUBEUnrecoverableFault an error occured
	 */
	public DisconnectUpdaterResponse disconnectUpdater(int updaterID) throws GCUBEUnrecoverableFault {
		try {
			DeltaListManagementHandler handler = getHandler();
			handler.setIndexResourceScopeToDeltaListManagementScope();
			handler.disconnectUpdater(updaterID);
			return new DisconnectUpdaterResponse();
		} catch (Exception e) {
			logger.error("Unable to disconnect updater from DeltaListManagementProvider.", e);
			throw new GCUBEUnrecoverableFault(e.getMessage());
		}
	}


	/**
	 * Informs the caller if the Generator is currently updating
	 * @param empty empty stub request
	 * @return boolean true if updating
	 * @throws GCUBEUnrecoverableFault an error occured
	 */ //should not respond true before update
	public boolean isUpdating(IsUpdating empty) throws GCUBEUnrecoverableFault {
		try {
			return getHandler().isUpdating();
		} catch (Exception e) {
			logger.error("Failed to get the manager resource.", e);
			throw new GCUBEUnrecoverableFault(e);
		}
	}

	/**
	 * Returns the delta list management handler in use.
	 * @return the delta list management handler
	 * @throws Exception an error occured
	 */
	private DeltaListManagementHandler getHandler() throws Exception {
		return getManagerResource().getIndexManagerHandler();
	}

	/**
	 * Returns the delta list management resource.
	 * @return the delta list management resource
	 * @throws Exception an error occured
	 */
	private DeltaListManager getManagerResource() throws Exception {
		Object resource = ResourceContext.getResourceContext().getResource();
		return (DeltaListManager) resource;
	}
	
	/* (non-Javadoc)
	 * @see org.gcube.common.core.porttypes.GCUBEStartupPortType#getServiceContext()
	 */
	@Override
	protected GCUBEServiceContext getServiceContext() {
		return ServiceContext.getContext();
	}

	/**
	 * Provider for the getCurrentMessage operation of the notification producer.
	 * If the topic whose current message is requested is the "delta file added"
	 * or "delta file deleted" topic, then don't return any message, because:
	 * If a consumer (index lookup resource) goes down and comes back again, it
	 * will get the previously sent message again upon re-subscribing, effectively
	 * adding or deleting the same data more than one time.
	 * For every other topic, the request is dispatched to the default, generic
	 * getCurrentMessage provider which returns the last message sent by the
	 * producer on this topic. 
	 *
	 * @param request the request
	 * @return the current message
	 * @throws RemoteException
	 * @throws ResourceUnknownFaultType
	 * @throws InvalidTopicExpressionFaultType
	 * @throws TopicNotSupportedFaultType
	 * @throws NoCurrentMessageOnTopicFaultType
	 */
	public GetCurrentMessageResponse getCurrentMessage(GetCurrentMessage request) throws RemoteException, ResourceUnknownFaultType, InvalidTopicExpressionFaultType, TopicNotSupportedFaultType, NoCurrentMessageOnTopicFaultType {
		logger.info("'getCurrentMessage' called on DeltaListManagementProvider");
		logger.info("Requested resourceKey: " + ResourceContext.getResourceContext().getResourceKey());
		
		/* Get the topic whose current message is requested */
		Topic topic = null;
		try {
			Object resource = ResourceContext.getResourceContext().getResource();
			TopicList topicList = ((TopicListAccessor) resource).getTopicList();
			topic = (Topic) topicList.getTopics(request.getTopic()).toArray()[0];
		} catch (Exception e) {
			logger.error("Failed to retrieve current message.", e);
			throw new RemoteException("Failed to retrieve current message.", e);
		}
		
		/* Check the topic name and throw a NoCurrentMessageOnTopic fault type if it's one of
		 * "add delta file" or "delete delta file"
		 */
		String topicName = topic.getName().getLocalPart();
		logger.info("Current message requested for Topic: " + topic.getName());
		if (topicName.equals(DeltaListManagementHandler.TOPIC_ADD_DELTA) || topicName.equals(DeltaListManagementHandler.TOPIC_DELETE_DELTA)) {
			logger.info("Throwing a NoCurrentMessageOnTopicFaultType.");
			throw new NoCurrentMessageOnTopicFaultType();
		}
		
		logger.info("Dispatching request to the default getCurrentMessage provider.");
		return gcmProvider.getCurrentMessage(request);
	}
}
